% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_wms_raster.R
\name{get_wms_raster}
\alias{get_wms_raster}
\title{Download WMS raster layer}
\usage{
get_wms_raster(shape,
               apikey = "altimetrie",
               layer_name = "ELEVATION.ELEVATIONGRIDCOVERAGE.HIGHRES",
               resolution = 5,
               filename = tempfile(fileext = ".tif"),
               crs = 2154,
               overwrite = FALSE,
               version = "1.3.0",
               styles = "",
               interactive = FALSE)
}
\arguments{
\item{shape}{Object of class \code{sf}. Needs to be located in
France.}

\item{apikey}{API key from \code{get_apikeys()} or directly
from \href{https://geoservices.ign.fr/services-web-experts}{IGN website}.}

\item{layer_name}{Name of the layer from \code{get_layers_metadata(apikey, "wms")}
or directly from
\href{https://geoservices.ign.fr/services-web-experts}{IGN website}.}

\item{resolution}{Cell size in meter. See detail for more information about resolution.}

\item{filename}{Either a character string naming a file or a connection open
for writing. (ex : "test.tif" or "~/test.tif"). If \code{NULL}, layer_name is used.
Default drivers is ".tif" but all gdal drivers are supported, see details
for more info. To avoid re-downloads, \verb{get_wms raster} checks that there is
not already a raster with that name. If it does, it is imported into R without
further downloading if \code{overwrite} is set to FALSE.}

\item{crs}{Numeric, character, or object of class sf or sfc. It is set to EPSG:2154
by default. See \code{\link[sf:st_crs]{sf::st_crs()}} for more detail.}

\item{overwrite}{If TRUE, output raster is overwrite.}

\item{version}{The version of the service used. See detail for more
information about \code{version}.}

\item{styles}{The rendering style of the layers. Set to "" by default.
See detail for more information about \code{styles}.}

\item{interactive}{If set to TRUE, no need to specify \code{apikey} and \code{layer_name},
you'll be ask.}
}
\value{
\code{get_wms_raster} return an object of class \code{SpatRaster} from \code{terra} package.
}
\description{
Download a raster layer from IGN Web Mapping Services (WMS).
To do that, it need a location giving by a shape, an apikey
and the name of layer. You can find those information from
\href{https://geoservices.ign.fr/services-web-experts}{IGN website}
or with \code{get_apikeys()} and \code{get_layers_metadata()}.
}
\details{
\itemize{
\item Raster tile are limited to 2048x2048 pixels so depending of the shape
and the resolution, correct number of tiles to download is calculated.
This mean that setting the \code{resolution} argument higher than the base resolution
of the layer multiplies the number of pixels without increasing
the precision. For example, the download of the BD Alti layer from
IGN will be optimal for a resolution of 25m.
\item \code{version} and \code{styles} arguments are detailed on
\href{https://geoservices.ign.fr/documentation/services/api-et-services-ogc/images-wms-ogc}{IGN documentation}
\item Using the \code{crs} argument avoids post-reprojection which can be time consuming
\item All GDAL supported drivers can be found \href{https://gdal.org/drivers/raster/index.html}{here}
}
}
\examples{
\dontrun{
library(sf)
library(tmap)

# shape from the best town in France
penmarch <- read_sf(system.file("extdata/penmarch.shp", package = "happign"))

# For quick testing, use interactive = TRUE
raster <- get_wms_raster(shape = penmarch, interactive = TRUE)

# For specific use, choose apikey with get_apikey() and layer_name with get_layers_metadata()
apikey <- get_apikeys()[4]  # altimetrie
metadata_table <- get_layers_metadata(apikey, "wms") # all layers for altimetrie wms
layer_name <- as.character(metadata_table[2,1]) # ELEVATION.ELEVATIONGRIDCOVERAGE

# Downloading digital elevation model from IGN
mnt <- get_wms_raster(penmarch, apikey, layer_name, resolution = 25)

# Preparing raster for plotting
mnt[mnt < 0] <- NA # remove negative values in case of singularity
names(mnt) <- "Elevation [m]" # Rename raster ie the title legend

# Plotting result
tm_shape(mnt)+
   tm_raster(legend.show = FALSE)+
tm_shape(penmarch)+
   tm_borders(col = "blue", lwd  = 3)
}
}
\seealso{
\code{\link[=get_apikeys]{get_apikeys()}}, \code{\link[=get_layers_metadata]{get_layers_metadata()}}
}
