% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cols_add.R
\name{cols_add}
\alias{cols_add}
\title{Add one or more columns to a \strong{gt} table}
\usage{
cols_add(.data, ..., .before = NULL, .after = NULL)
}
\arguments{
\item{.data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{...}{\emph{Cell data assignments}

\verb{<multiple expressions>} // (or, use \code{.list})

Expressions for the assignment of cell values to the new columns.
Name-value pairs, in the form of \verb{<column> = <value vector>} will work, so
long as any \verb{<column>} value does not already exist in the table. The
\verb{<value vector>} may be an expression that uses one or more column names in
the table to generate a vector of values. Single values in \verb{<value vector>}
will be repeated down the new column. A vector where the length is exactly
the number of rows in the table can also be used.}

\item{.before, .after}{\emph{Column used as anchor}

\verb{<column-targeting expression>} // \emph{default:} \code{NULL} (\code{optional})

A single column-resolving expression or column index can be given to either
\code{.before} or \code{.after}. The column specifies where the new columns should be
positioned among the existing columns in the input data table. While select
helper functions such as \code{\link[=starts_with]{starts_with()}} and \code{\link[=ends_with]{ends_with()}} can be used for
column targeting, it's recommended that a single column name or index be
used. This is to ensure that exactly one column is provided to either of
these arguments (otherwise, the function will be stopped). If nothing is
provided for either argument then any new column will be placed at the end
of the column series.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
We can add new columns to a table with \code{cols_add()} and it works quite a bit
like \code{dplyr::mutate()} does. The idea is that you supply name-value pairs
where the name is the new column name and the value part describes the data
that will go into the column. The latter can: (1) be a vector where the
length of the number of rows in the data table, (2) be a single value
(which will be repeated all the way down), or (3) involve other columns in
the table (as they represent vectors of the correct length). The new columns
are added to the end of the column series by default but can instead be added
internally by using either the \code{.before} or \code{.after} arguments. If entirely
empty (i.e., all \code{NA}) columns need to be added, you can use any of the \code{NA}
types (e.g., \code{NA}, \code{NA_character_}, \code{NA_real_}, etc.) for such columns.
}
\section{Targeting the column for insertion with \code{.before} or \code{.after}}{


The targeting of a column for insertion is done through the \code{.before} or
\code{.after} arguments (only one of these options should be be used). While
\strong{tidyselect}-style expressions or indices can used to target a column, it's
advised that a single column name be used. This is to avoid the possibility
of inadvertently resolving multiple columns (since the requirement is for a
single column).
}

\section{Examples}{


Let's take a subset of the \code{\link{exibble}} dataset and make a simple \strong{gt} table
with it (using the \code{row} column for labels in the stub). We'll add a single
column to the right of all the existing columns and call it \code{country}. This
new column needs eight values and these will be supplied when using
\code{cols_add()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  dplyr::select(num, char, datetime, currency, group) |>
  gt(rowname_col = "row") |>
  cols_add(
    country = c("TL", "PY", "GL", "PA", "MO", "EE", "CO", "AU")
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_add_1.png" alt="This image of a table was generated from the first code example in the `cols_add()` help file." style="width:100\%;">
}}

We can add multiple columns with a single use of \code{cols_add()}. The columns
generated can be formatted and otherwise manipulated just as any column could
be in a \strong{gt} table. The following example extends the first one by adding
more columns and immediately using them in various function calls like
\code{\link[=fmt_flag]{fmt_flag()}} and \code{\link[=fmt_units]{fmt_units()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  dplyr::select(num, char, datetime, currency, group) |>
  gt(rowname_col = "row") |>
  cols_add(
    country = c("TL", "PY", "GL", "PA", "MO", "EE", "CO", "AU"),
    empty = NA_character_,
    units = c(
      "k m s^-2", "N m^-2", "degC", "m^2 kg s^-2",
      "m^2 kg s^-3", "/s", "A s", "m^2 kg s^-3 A^-1"
    ),
    big_num = num ^ 3
  ) |>
  fmt_flag(columns = country) |>
  sub_missing(columns = empty, missing_text = "") |>
  fmt_units(columns = units) |>
  fmt_scientific(columns = big_num)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_add_2.png" alt="This image of a table was generated from the second code example in the `cols_add()` help file." style="width:100\%;">
}}

In this table generated from a portion of the \code{\link{towny}} dataset, we add two
new columns (\code{land_area} and \code{density}) through a single use of \code{cols_add()}.
The new \code{land_area} column is a conversion of land area from square
kilometers to square miles and the \code{density} column is calculated by through
division of \code{population_2021} by that new \code{land_area} column. We hide the
now unneeded \code{land_area_km2} with \code{\link[=cols_hide]{cols_hide()}} and also perform some column
labeling and adjustments to column widths with \code{\link[=cols_label]{cols_label()}} and
\code{\link[=cols_width]{cols_width()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::select(name, population_2021, land_area_km2) |>
  dplyr::filter(population_2021 > 100000) |>
  dplyr::slice_max(population_2021, n = 10) |>
  gt() |>
  cols_add(
    land_area = land_area_km2 / 2.58998811,
    density = population_2021 / land_area
  ) |>
  fmt_integer() |>
  cols_hide(columns = land_area_km2) |>
  cols_label(
    population_2021 = "Population",
    density = "Density, \{\{*persons* / sq mi\}\}",
    land_area ~ "Area, \{\{mi^2\}\}"
  ) |>
  cols_width(everything() ~ px(120))
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_add_3.png" alt="This image of a table was generated from the third code example in the `cols_add()` help file." style="width:100\%;">
}}

It's possible to start with an empty table (i.e., no columns and no rows) and
add one or more columns to that. You can, for example, use \code{dplyr::tibble()}
or \code{data.frame()} to create a completely empty table. The first \code{cols_add()}
call for an empty table can have columns of arbitrary length but subsequent
uses of \code{cols_add()} must adhere to the rule of new columns being the same
length as existing.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble() |>
  gt() |>
  cols_add(
    num = 1:5,
    chr = vec_fmt_spelled_num(1:5)
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_add_4.png" alt="This image of a table was generated from the fourth code example in the `cols_add()` help file." style="width:100\%;">
}}

Tables can contain no rows, yet have columns. In the following example, we'll
create a zero-row table with three columns (\code{num}, \code{chr}, and \code{ext}) and
perform the same \code{cols_add()}-based addition of two columns of data. This is
another case where the function allows for arbitrary-length columns (since
always adding zero-length columns is impractical). Furthermore, here we can
reference columns that already exist (\code{num} and \code{chr}) and add values to
them.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  num = numeric(0),
  chr = character(0),
  ext = character(0)
) |>
  gt() |>
  cols_add(
    num = 1:5,
    chr = vec_fmt_spelled_num(1:5)
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_add_5.png" alt="This image of a table was generated from the fifth code example in the `cols_add()` help file." style="width:100\%;">
}}

We should note that the \code{ext} column did not receive any values from
\code{cols_add()} but the table was expanded to having five rows nonetheless. So,
each cell of \code{ext} was by necessity filled with an \code{NA} value.
}

\section{Function ID}{

5-7
}

\section{Function Introduced}{

\code{v0.10.0} (October 7, 2023)
}

\seealso{
Other column modification functions: 
\code{\link{cols_align}()},
\code{\link{cols_align_decimal}()},
\code{\link{cols_hide}()},
\code{\link{cols_label}()},
\code{\link{cols_label_with}()},
\code{\link{cols_merge}()},
\code{\link{cols_merge_n_pct}()},
\code{\link{cols_merge_range}()},
\code{\link{cols_merge_uncert}()},
\code{\link{cols_move}()},
\code{\link{cols_move_to_end}()},
\code{\link{cols_move_to_start}()},
\code{\link{cols_nanoplot}()},
\code{\link{cols_unhide}()},
\code{\link{cols_units}()},
\code{\link{cols_width}()}
}
\concept{column modification functions}
