#' @title  Gravity dataset without zero trade flows
#' 
#' @description 
#' An edited version of the full gravity dataset: The "square" gravity dataset 
#' for all possible pairs of Countries worldwide, 1948-2006, which is used in 
#' the article: Head, K., T. Mayer and J. Ries, 2010, "The erosion of 
#' colonial linkages after independence". Journal of International Economics, 
#' 81(1):1-14 (lead article).
#' Dataset \code{Gravity_no_zeros} corresponds to the dataset without zero
#' trade flows, \code{Gravity_zeros}, on the other hand, includes
#' zero trade flows.
#' 
#' @details 
#' An edited version of the full gravity dataset: The "square" gravity dataset 
#' for all possible pairs of Countries worldwide, 1948-2006, which is used in 
#' the article: Head, K., T. Mayer and J. Ries, 2010, "The erosion of 
#' colonial linkages after independence". Journal of International Economics, 
#' 81(1):1-14 (lead article).
#' 
#' In order to have a dataset suited for all functions, a cross-sectional
#' dataset is chosen. All incomplete rows and observations with missing
#' trade flows are therefore excluded from the dataset.
#' As some of the functions in the package are capable of
#' handling zero values in trade flows and some are not, 
#' two datasets, \code{Gravity_zeros} and \code{Gravity_no_zeros},
#' are provided. 
#' 
#' The original dataset downloaded at 
#' \url{http://econ.sciences-po.fr/sites/default/files/file/tmayer/data/col_regfile09.zip} 
#' was edited in the following way:
#' 
#' # Reading in the dataset
#' 
#' library(foreign)
#' 
#' col_regfile09 <- read.dta("col_regfile09.dta")
#' 
#' # Isolation of one year
#' 
#' data06 <- col_regfile09[col_regfile09$year == 2006,]
#' 
#' # Choosing variables (select columns)
#' 
#' data06 <- data06[,c(2, 3, 6, 8, 12, 27, 34, 4, 5, 29)]
#' 
#' # Transforming data
#' 
#' # Isolation of complete cases
#' 
#' data06 <- data06[complete.cases(data06) == TRUE,]
#' 
#' # Exclusion of trade flows equal to 0
#' 
#' Gravity_no_zeros <- data06[data06$flow != 0,]
#' 
#' row.names(Gravity_no_zeros) <- 1:length(row.names(Gravity_no_zeros))
#' 
#' # Divide GDPs by 1,000,000 for scaling purposes
#' 
#' Gravity_no_zeros$gdp_o <- Gravity_no_zeros$gdp_o / 1000000
#' 
#' Gravity_no_zeros$gdp_d <- Gravity_no_zeros$gdp_d / 1000000
#' 
#' @docType data
#' 
#' @usage data(Gravity_no_zeros)
#' 
#' @format A data frame with 17088 rows and 10 variables:
#' \describe{
#'   \item{iso_o}{ISO-Code of country of origin}
#'   \item{iso_d}{ISO-Code of country of destination}
#'   \item{distw}{weighted distance}
#'   \item{gdp_o}{GDP of country of origin}
#'   \item{gdp_d}{GDP of country of destination}
#'   \item{rta}{regional trade agreement}
#'   \item{flow}{trade flow}
#'   \item{contig}{contiguity}
#'   \item{comlang_off}{common official language}
#'   \item{comcur}{common currency}
#' }
#' 
#' @name Gravity_no_zeros
#' 
#' @keywords datasets
#' 
#' @source \url{http://econ.sciences-po.fr/node/131}
#' \url{https://sites.google.com/site/hiegravity/data-sources}
#' 
#' @references 
#' Head, K. and Mayer, T. (2014). Chapter 3 - gravity equations: 
#' Workhorse,toolkit, and cookbook. 
#' In Gita Gopinath, E. H. and Rogoff, K., editors, 
#' Handbook of International Economics, volume 4 of Handbook of International 
#' Economics, pages 131-195. Elsevier.
#' (\href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook})
#' 
#' Head, K., T. Mayer and J. Ries, 2010, "The erosion od colonial linkages after 
#' independence". Journal of International Economics, 81(1):1-14 
#' (lead article). 
#' 
NULL