% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MPG.R
\docType{methods}
\name{MPG}
\alias{MPG}
\alias{MPG,RasterLayer,RasterLayer-method}
\alias{MPG,RasterLayer,numeric-method}
\title{Extract a minimum planar graph (MPG) model from a landscape resistance surface}
\usage{
MPG(cost, patch, ...)

\S4method{MPG}{RasterLayer,RasterLayer}(cost, patch, ...)

\S4method{MPG}{RasterLayer,numeric}(cost, patch, ...)
}
\arguments{
\item{cost}{A \code{RasterLayer} giving a landscape resistance surface,
where the values of each raster cell are proportional to the
resistance to movement, dispersal, or gene flow for an organism
in the landscape feature they represent.
Missing values \code{NA} are acceptable (but see below).
Negative values are not.
To extract an MPG with Euclidean links (i.e., and not least-cost
path links) set \code{cost[] <- 1}.}

\item{patch}{A raster of class \code{RasterLayer} for a patch-based analysis
OR an integer for a lattice analysis.
If a raster is given it must be of the same extent, origin and
projection as \code{cost} and be binary, without missing values,
where patches=1 and non-patches=0.
For lattice analyses, an integer gives the spacing in raster
cells between focal points in the lattice.}

\item{...}{Additional arguments (not used).}
}
\value{
A \code{\link[=mpg-class]{mpg}} object.
}
\description{
Extracts a minimum planar graph (MPG) and is also the first step
in grains of connectivity (GOC) modelling.
Both patch-based and lattice MPGs can be extracted.
}
\details{
Use this function to create a minimum planar graph (MPG) that can be
further analyzed using \code{\link{igraph}} routines.
It is also the first step in grains of connectivity (GOC) modelling.
}
\note{
Researchers should consider whether the use of a patch-based MPG or a lattice
MPG model is appropriate based on the patch-dependency of the organism under study.
Patch-based models make most sense when animals are restricted to, or dependent on,
a resource patch. Lattice models can be used as a generalized and functional
approach to scaling resistance surfaces.

Rasters should be projected and not in geographic coordinates (i.e. \code{projection(cost)}
should not contain \code{"+proj=longlat"}) or the function will issue a warning.
In unprojected cases consider using \code{\link{projectRaster}} to change to an appropriate
coordinate system for the location and extent of interest that balances both distance and areal
accuracy. See \url{http://www.spatialreference.org} for location-specific suggestions.
Use of geographic coordinates will result in inaccurate areal and distance measurements,
rendering the models themselves inaccurate.
}
\examples{
## Load raster landscape
tiny <- raster::raster(system.file("extdata/tiny.asc", package = "grainscape"))

## Create a resistance surface from a raster using an is-becomes reclassification
tinyCost <- raster::reclassify(tiny, rcl = cbind(c(1, 2, 3, 4), c(1, 5, 10, 12)))

## Produce a patch-based MPG where patches are resistance features=1
tinyPatchMPG <- MPG(cost = tinyCost, patch = tinyCost == 1)

## Explore the graph structure and node/link attributes
graphdf(tinyPatchMPG)

## Find the mean patch area (see igraph manual for use of V() and E())
mean(igraph::V(tinyPatchMPG@mpg)$patchArea)

## Quick visualization of the MPG
if (interactive())
  plot(tinyPatchMPG, col = c("grey", "black"), legend = FALSE)

## Additional graph extraction scenarios
## Produce a lattice MPG where focal points are spaced 10 cells apart
tinyLatticeMPG <- MPG(cost = tinyCost, patch = 10)
if (interactive())
  plot(tinyLatticeMPG)
}
\references{
Fall, A., M.-J. Fortin, M. Manseau, D. O'Brien. (2007) Spatial graphs:
Principles and applications for habitat connectivity. Ecosystems 10:448:461.

Galpern, P., M. Manseau. (2013a) Finding the functional grain: comparing methods
for scaling resistance surfaces. Landscape Ecology 28:1269-1291.

Galpern, P., M. Manseau. (2013b) Modelling the influence of landscape connectivity
on animal distribution: a functional grain approach. Ecography 36:1004-1016.

Galpern, P., M. Manseau, A. Fall. (2011) Patch-based graphs of landscape connectivity:
a guide to construction, analysis, and application for conservation.
Biological Conservation 144:44-55.

Galpern, P., M. Manseau, P.J. Wilson. (2012) Grains of connectivity: analysis
at multiple spatial scales in landscape genetics. Molecular Ecology 21:3996-4009.
}
\seealso{
\code{\link{GOC}, \link{threshold}}
}
\author{
Paul Galpern, Sam Doctolero, Alex Chubaty
}
