% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynPred.R
\name{dynPred}
\alias{dynPred}
\title{Dynamic predictions for survival sub-model in a multivariate joint model.}
\usage{
dynPred(
  data,
  id,
  fit,
  u = NULL,
  nsim = 200,
  progress = TRUE,
  b.density = c("normal", "t"),
  scale = NULL,
  df = NULL
)
}
\arguments{
\item{data}{the data to which the original \code{joint} model was fit.}

\item{id}{subject identifier, i.e. for which subject is the conditional survival probabilities 
desired?}

\item{fit}{a joint model fit by the  \code{joint} function.}

\item{u}{a numeric \code{vector} of candidate follow-up times for which a survival probability 
should be calculated. Note that the first item \code{u[1]} denotes the start of the "window"
and is dropped from calculations. If \code{u=NULL} (the default), then the probability of 
surviving all failure times after the \code{id}'s final longitudinal \code{time} is calculated.}

\item{nsim}{how many Monte Carlo simulations should be carried out? Defaults to 
\code{nsim=200}. First-order estimates are calculated if \code{nsim=0}.}

\item{progress}{a logical, if \code{progress=TRUE} (the default) then a progress bar displaying
the current percentage of simulations have been completed.}

\item{b.density}{character string imposing the density for the current and proposal for each
draw of the random effects \eqn{\boldsymbol{b}^{(l)}, l=1,\dots,\code{nsim}}. Options are 
\code{b.density="normal"} i.e. using the assumption that the conditional density of the random
effects is 

\deqn{\boldsymbol{b}_i^{(l)}|\boldsymbol{Y}_i(u);\boldsymbol{\Omega}^{(l)}\sim 
 N(\hat{\boldsymbol{b}}_i^{(l)},\hat{\boldsymbol{\Sigma}}_i^{(l)})}

(i.e. in line with proposal by Bernhardt \emph{et al.} (2015)), or in keeping with other 
literature surrounding dynamic predictions (e.g. Rizopoulos (2011)) in imposing the \code{
b.density="t"} distribution.}

\item{scale}{if \code{b.density = "t"} then this numeric scales the variance-covariance 
parameter in the proposal distribution for the Metropolis-Hastings algorithm. Defaults to 
\code{scale = NULL} which doesn't scale the variance term at all. Users are encouraged to
experiment with values here.}

\item{df}{if \code{b.density = "t"} then this numeric denotes the degrees of freedom of the 
proposed \eqn{t} distribution on the random effects. \code{df=4} is suggested.}
}
\value{
A list of class \code{dynPred} which consists of three items: \describe{
  \item{\code{pi}}{A \code{data.frame} which contains each candidate failure time (supplied by
  \code{u}), with the mean, median and 2.5\% and 97.5\% quantiles of probability of survival 
  until this failure time.}
  \item{\code{pi.raw}}{A \code{matrix} of with \code{nsim} rows and \code{length(u)} columns,
  each row represents the \eqn{l}th conditional survival probability of survival each \code{u}
  survival time. This is largely for debugging purposes.}
  \item{MH.accept}{The acceptance rate of the Metropolis-Hastings algorithm on the random
  effects.}
}
}
\description{
Calculates individualised conditional survival probabilities for subjects
during a period of follow-up using a \code{joint} model fit along with requisite longitudinal 
process history. 

\strong{Note} that this function is largely designed for use within the ROC function
which assesses discriminatory power of the joint model, however it \emph{does} function
by itself with proper use of its arguments.
}
\details{
Dynamic predictions for the time-to-event process based on information available
on the subject's longitudinal process up to given time \eqn{t} are calculated by Monte Carlo
simulation outlined in Rizopoulos (2011). For a subject last observed at time \eqn{t}, the 
probability that they survive until future time \eqn{u} is

\deqn{
  Pr(T_i \ge u | T \ge t; \boldsymbol{Y}_i, \boldsymbol{b}_i; \boldsymbol{\Omega}) \approx
  \frac{S(u|\hat{\boldsymbol{b}}_i; \boldsymbol{\Omega})}
  {S(t|\hat{\boldsymbol{b}}_i; \boldsymbol{\Omega})}
}
where \eqn{T_i} is the true failure time for subject \eqn{i}, \eqn{\boldsymbol{Y}_i} their
longitudinal measurements up to time \eqn{t}, and \eqn{S()} the survival function.

\eqn{\boldsymbol{\Omega}} is drawn from the multivariate normal distribution with mean
\eqn{\hat{\boldsymbol{\Omega}}} and its variance taken from a fitted \code{joint} object.
\eqn{\hat{\boldsymbol{b}}} is drawn from either the (multivariate) Normal, or \eqn{t} 
distribution by means of a Metropolis-Hastings algorithm with \code{nsim} iterations.
}
\examples{
\donttest{
data(PBC)
PBC$serBilir <- log(PBC$serBilir)
# Univariate -----------------------------------------------------------
long.formulas <- list(serBilir ~ drug * time + (1 + time|id))
surv.formula <- Surv(survtime, status) ~ drug
family <- list('gaussian')
fit <- joint(long.formulas, surv.formula, PBC, family)
preds <- dynPred(PBC, id = 81, fit = fit, u = NULL, nsim = 200, b.density = 'normal',
                 scale = 0.18)
preds
plot(preds)
# Bivariate ------------------------------------------------------------
# Does introduction of albumin affect conditional survival probability?
long.formulas <- list(
  serBilir ~ drug * time + I(time^2) + (1 + time + I(time^2)|id),
  albumin ~ drug * time + (1 + time|id)
)
fit <- joint(long.formulas, surv.formula, data = PBC, family = list("gaussian", "gaussian"))
bi.preds <- dynPred(PBC, id = 81, fit = fit, u = NULL, nsim = 200, b.density = 'normal',
                    scale = 0.66)
bi.preds
plot(bi.preds) # Appears to level-off quicker; perhaps indicative of this id's albumin levels.
}
}
\references{
Bernhardt PW, Zhang D and Wang HJ. A fast EM Algorithm for Fitting Joint Models of a Binary 
Response to Multiple Longitudinal Covariates Subject to Detection Limits. 
\emph{Computational Statistics and Data Analysis} 2015; \strong{85}; 37--53

Rizopoulos D. Dynamic predictions and prospective accuracy in joint models
for longitudinal and time-to-event data. \emph{Biometrics} 2011;
\strong{67}: 819–829.
}
\seealso{
\code{\link{ROC}} and \code{\link{plot.dynPred}}.
}
\author{
James Murray (\email{j.murray7@ncl.ac.uk}).
}
