% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glmtrans_inf.R
\name{glmtrans_inf}
\alias{glmtrans_inf}
\title{Calculate asymptotic confidence intervals based on desparsified Lasso and two-step transfer learning method.}
\usage{
glmtrans_inf(
  target,
  source = NULL,
  family = c("gaussian", "binomial", "poisson"),
  beta.hat = NULL,
  nodewise.transfer.source.id = "all",
  cores = 1,
  level = 0.95,
  intercept = TRUE,
  ...
)
}
\arguments{
\item{target}{target data. Should be a list with elements x and y, where x indicates a predictor matrix with each row/column as a(n) observation/variable, and y indicates the response vector.}

\item{source}{source data. Should be a list with some sublists, where each of the sublist is a source data set, having elements x and y with the same meaning as in target data.}

\item{family}{response type. Can be "gaussian", "binomial" or "poisson". Default = "gaussian".
\itemize{
\item "gaussian": Gaussian distribution.
\item "binomial": logistic distribution. When \code{family = "binomial"}, the input response in both \code{target} and \code{source} should be 0/1.
\item "poisson": poisson distribution. When \code{family = "poisson"}, the input response in both \code{target} and \code{source} should be non-negative.
}}

\item{beta.hat}{initial estimate of the coefficient vector (the intercept should be the first component). Can be from the output of function \code{glmtrans}.}

\item{nodewise.transfer.source.id}{transferable source indices in the infernce (the set A in Algorithm 3 of Tian, Y. and Feng, Y., 2021). Can be either a subset of \code{{1, ..., length(source)}}, "all" or \code{NULL}. Default = \code{"all"}.
\itemize{
\item a subset of \code{{1, ..., length(source)}}: only transfer sources with the specific indices.
\item "all": transfer all sources.
\item NULL: don't transfer any sources and only use target data.
}}

\item{cores}{the number of cores used for parallel computing. Default = 1.}

\item{level}{the level of confidence interval. Default = 0.95. Note that the level here refers to the asymptotic level of confidence interval of a single component rather than the multiple intervals.}

\item{intercept}{whether the model includes the intercept or not. Default = TRUE. Should be set as TRUE if the intercept of \code{beta.hat} is not zero.}

\item{...}{additional arguments.}
}
\value{
a list of output. b.hat = b.hat, beta.hat = beta.hat, CI = CI, var.est = var.est
\item{b.hat}{the center of confidence intervals. A \code{p}-dimensional vector, where \code{p} is the number of predictors.}
\item{beta.hat}{the initial estimate of the coefficient vector (the same as input).}
\item{CI}{confidence intervals (CIs) with the specific level. A \code{p} by 3 matrix, where three columns indicate the center, lower limit and upper limit of CIs, respectively. Each row represents a coefficient component.}
\item{var.est}{the estimate of variances in the CLT (Theta transpose times Sigma times Theta, in section 2.5 of Tian, Y. and Feng, Y., 2021). A \code{p}-dimensional vector, where \code{p} is the number of predictors.}
}
\description{
Given the point esimate of the coefficient vector from \code{glmtrans}, calculate the asymptotic confidence interval of each component. The detailed inference algorithm can be found as Algorithm 3 in the latest version of Tian, Y. and Feng, Y., 2021. The algorithm is consructed based on a modified version of desparsified Lasso (Van de Geer, S. et al, 2014; Dezeure, R. et al, 2015).
}
\examples{
\dontrun{
set.seed(0, kind = "L'Ecuyer-CMRG")

# generate binomial data
D.training <- models("binomial", type = "all", K = 2, p = 200)

# fit a logistic regression model via two-step transfer learning method
fit.binomial <- glmtrans(D.training$target, D.training$source, family = "binomial")

# calculate the CI based on the point estimate from two-step transfer learning method
fit.inf <- glmtrans_inf(target = D.training$target, source = D.training$source,
family = "binomial", beta.hat = fit.binomial$beta, cores = 2)
}
}
\references{
Tian, Y. and Feng, Y., 2021. \emph{Transfer Learning under High-dimensional Generalized Linear Models. arXiv preprint arXiv:2105.14328.}

Van de Geer, S., Bühlmann, P., Ritov, Y.A. and Dezeure, R., 2014. \emph{On asymptotically optimal confidence regions and tests for high-dimensional models. The Annals of Statistics, 42(3), pp.1166-1202.}

Dezeure, R., Bühlmann, P., Meier, L. and Meinshausen, N., 2015. \emph{High-dimensional inference: confidence intervals, p-values and R-software hdi. Statistical science, pp.533-558.}
}
\seealso{
\code{\link{glmtrans}}.
}
