% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg.R
\name{gkwreg}
\alias{gkwreg}
\title{Fit Generalized Kumaraswamy Regression Models}
\usage{
gkwreg(
  formula,
  data,
  family = c("gkw", "bkw", "kkw", "ekw", "mc", "kw", "beta"),
  link = NULL,
  link_scale = NULL,
  start = NULL,
  fixed = NULL,
  method = c("nlminb", "BFGS", "Nelder-Mead", "CG", "SANN", "L-BFGS-B"),
  hessian = TRUE,
  plot = TRUE,
  conf.level = 0.95,
  optimizer.control = list(),
  subset = NULL,
  weights = NULL,
  offset = NULL,
  na.action = getOption("na.action"),
  contrasts = NULL,
  x = FALSE,
  y = TRUE,
  model = TRUE,
  silent = TRUE,
  ...
)
}
\arguments{
\item{formula}{An object of class \code{\link[Formula]{Formula}} (or one that
can be coerced to that class). It should be structured as
\code{y ~ model_alpha | model_beta | model_gamma | model_delta | model_lambda},
where \code{y} is the response variable and each \code{model_*} part specifies
the linear predictor for the corresponding parameter (\eqn{\alpha}, \eqn{\beta},
\eqn{\gamma}, \eqn{\delta}, \eqn{\lambda}). If a part is omitted or specified
as \code{~ 1} or \code{.}, an intercept-only model is used for that parameter.
See Details for parameter correspondence in subfamilies.}

\item{data}{A data frame containing the variables specified in the \code{formula}.}

\item{family}{A character string specifying the desired distribution family.
Defaults to \code{"gkw"}. Supported families are:
\itemize{
\item \code{"gkw"}: Generalized Kumaraswamy (5 parameters: \eqn{\alpha, \beta, \gamma, \delta, \lambda})
\item \code{"bkw"}: Beta-Kumaraswamy (4 parameters: \eqn{\alpha, \beta, \gamma, \delta}; \eqn{\lambda = 1} fixed)
\item \code{"kkw"}: Kumaraswamy-Kumaraswamy (4 parameters: \eqn{\alpha, \beta, \delta, \lambda}; \eqn{\gamma = 1} fixed)
\item \code{"ekw"}: Exponentiated Kumaraswamy (3 parameters: \eqn{\alpha, \beta, \lambda}; \eqn{\gamma = 1, \delta = 0} fixed)
\item \code{"mc"}: McDonald / Beta Power (3 parameters: \eqn{\gamma, \delta, \lambda}; \eqn{\alpha = 1, \beta = 1} fixed)
\item \code{"kw"}: Kumaraswamy (2 parameters: \eqn{\alpha, \beta}; \eqn{\gamma = 1, \delta = 0, \lambda = 1} fixed)
\item \code{"beta"}: Beta distribution (2 parameters: \eqn{\gamma, \delta}; \eqn{\alpha = 1, \beta = 1, \lambda = 1} fixed)
}}

\item{link}{Either a single character string specifying the same link function
for all relevant parameters, or a named list specifying the link function for each
modeled parameter (e.g., \code{list(alpha = "log", beta = "log", delta = "logit")}).
Defaults are \code{"log"} for \eqn{\alpha, \beta, \gamma, \lambda} (parameters > 0)
and \code{"logit"} for \eqn{\delta} (parameter in (0, 1)).
Supported link functions are:
\itemize{
\item \code{"log"}: logarithmic link, maps \eqn{(0, \infty)} to \eqn{(-\infty, \infty)}
\item \code{"identity"}: identity link, no transformation
\item \code{"inverse"}: inverse link, maps \eqn{x} to \eqn{1/x}
\item \code{"sqrt"}: square root link, maps \eqn{x} to \eqn{\sqrt{x}}
\item \code{"inverse-square"}: inverse squared link, maps \eqn{x} to \eqn{1/x^2}
\item \code{"logit"}: logistic link, maps \eqn{(0, 1)} to \eqn{(-\infty, \infty)}
\item \code{"probit"}: probit link, using normal CDF
\item \code{"cloglog"}: complementary log-log
\item \code{"cauchy"}: Cauchy link, using Cauchy CDF
}}

\item{link_scale}{Either a single numeric value specifying the same scale for all
link functions, or a named list specifying the scale for each parameter's link
function (e.g., \code{list(alpha = 10, beta = 5, delta = 1)}). The scale affects
how the link function transforms the linear predictor. Default is 10 for most
parameters and 1 for parameters using probability-type links (such as \code{delta}).
For probability-type links (logit, probit, cloglog, cauchy), smaller values
produce more extreme transformations.}

\item{start}{An optional named list providing initial values for the regression
coefficients. Parameter names should match the distribution parameters (alpha,
beta, etc.), and values should be vectors corresponding to the coefficients
in the respective linear predictors (including intercept). If \code{NULL}
(default), suitable starting values are automatically determined based on
global parameter estimates.}

\item{fixed}{An optional named list specifying parameters or coefficients to be
held fixed at specific values during estimation. Currently not fully implemented.}

\item{method}{Character string specifying the optimization algorithm to use.
Options are \code{"nlminb"} (default, using \code{\link[stats]{nlminb}}),
\code{"BFGS"}, \code{"Nelder-Mead"}, \code{"CG"}, \code{"SANN"}, or \code{"L-BFGS-B"}.
If \code{"nlminb"} is selected, R's \code{\link[stats]{nlminb}} function is used;
otherwise, R's \code{\link[stats]{optim}} function is used with the specified method.}

\item{hessian}{Logical. If \code{TRUE} (default), the Hessian matrix is computed
via \code{\link[TMB]{sdreport}} to obtain standard errors and the covariance
matrix of the estimated coefficients. Setting to \code{FALSE} speeds up fitting
but prevents calculation of standard errors and confidence intervals.}

\item{plot}{Logical. If \code{TRUE} (default), enables the generation of diagnostic
plots when calling the generic \code{plot()} function on the fitted object.
Actual plotting is handled by the \code{plot.gkwreg} method.}

\item{conf.level}{Numeric. The confidence level (1 - alpha) for constructing
confidence intervals for the parameters. Default is 0.95. Used only if
\code{hessian = TRUE}.}

\item{optimizer.control}{A list of control parameters passed directly to the
chosen optimizer (\code{\link[stats]{nlminb}} or \code{\link[stats]{optim}}).
See their respective documentation for details.}

\item{subset}{An optional vector specifying a subset of observations from \code{data}
to be used in the fitting process.}

\item{weights}{An optional vector of prior weights (e.g., frequency weights)
to be used in the fitting process. Should be \code{NULL} or a numeric vector
of non-negative values.}

\item{offset}{An optional numeric vector or matrix specifying an a priori known
component to be included \emph{on the scale of the linear predictor} for each parameter.
If a vector, it's applied to the predictor of the first parameter in the standard
order (\eqn{\alpha}). If a matrix, columns must correspond to parameters in the
order \eqn{\alpha, \beta, \gamma, \delta, \lambda}.}

\item{na.action}{A function which indicates what should happen when the data
contain \code{NA}s. The default (\code{na.fail}) stops if \code{NA}s are
present. Other options include \code{\link[stats]{na.omit}} or
\code{\link[stats]{na.exclude}}.}

\item{contrasts}{An optional list specifying the contrasts to be used for factor
variables in the model. See the \code{contrasts.arg} of
\code{\link[stats]{model.matrix.default}}.}

\item{x}{Logical. If \code{TRUE}, the list of model matrices (one for each modeled
parameter) is returned as component \code{x} of the fitted object. Default \code{FALSE}.}

\item{y}{Logical. If \code{TRUE} (default), the response variable (after processing
by \code{na.action}, \code{subset}) is returned as component \code{y}.}

\item{model}{Logical. If \code{TRUE} (default), the model frame (containing all
variables used from \code{data}) is returned as component \code{model}.}

\item{silent}{Logical. If \code{TRUE} (default), suppresses progress messages
from TMB compilation and optimization. Set to \code{FALSE} for verbose output.}

\item{...}{Additional arguments, currently unused or passed down to internal
methods (potentially).}
}
\value{
An object of class \code{gkwreg}. This is a list containing the
following components:
\item{call}{The matched function call.}
\item{family}{The specified distribution family string.}
\item{formula}{The \code{\link[Formula]{Formula}} object used.}
\item{coefficients}{A named vector of estimated regression coefficients.}
\item{fitted.values}{Vector of estimated means (expected values) of the response.}
\item{residuals}{Vector of response residuals (observed - fitted mean).}
\item{fitted_parameters}{List containing the estimated mean value for each distribution parameter (\eqn{\alpha, \beta, \gamma, \delta, \lambda}).}
\item{parameter_vectors}{List containing vectors of the estimated parameters (\eqn{\alpha, \beta, \gamma, \delta, \lambda}) for each observation, evaluated on the link scale.}
\item{link}{List of link functions used for each parameter.}
\item{param_names}{Character vector of names of the parameters modeled by the family.}
\item{fixed_params}{Named list indicating which parameters are fixed by the family definition.}
\item{loglik}{The maximized log-likelihood value.}
\item{aic}{Akaike Information Criterion.}
\item{bic}{Bayesian Information Criterion.}
\item{deviance}{The deviance ( -2 * loglik ).}
\item{df.residual}{Residual degrees of freedom (nobs - npar).}
\item{nobs}{Number of observations used in the fit.}
\item{npar}{Total number of estimated parameters (coefficients).}
\item{vcov}{The variance-covariance matrix of the coefficients (if \code{hessian = TRUE}).}
\item{se}{Standard errors of the coefficients (if \code{hessian = TRUE}).}
\item{convergence}{Convergence code from the optimizer (0 typically indicates success).}
\item{message}{Convergence message from the optimizer.}
\item{iterations}{Number of iterations used by the optimizer.}
\item{rmse}{Root Mean Squared Error of response residuals.}
\item{efron_r2}{Efron's pseudo R-squared.}
\item{mean_absolute_error}{Mean Absolute Error of response residuals.}
\item{x}{List of model matrices (if \code{x = TRUE}).}
\item{y}{The response vector (if \code{y = TRUE}).}
\item{model}{The model frame (if \code{model = TRUE}).}
\item{tmb_object}{The raw object returned by \code{\link[TMB]{MakeADFun}}.}
}
\description{
Fits regression models using the Generalized Kumaraswamy (GKw) family of
distributions for response variables strictly bounded in the interval (0, 1).
The function allows modeling parameters from all seven submodels of the GKw
family as functions of predictors using appropriate link functions. Estimation
is performed using Maximum Likelihood via the TMB (Template Model Builder) package.
Requires the \code{Formula} and \code{TMB} packages.
}
\details{
The \code{gkwreg} function provides a regression framework for the Generalized
Kumaraswamy (GKw) family and its submodels, extending density estimation
to include covariates. The response variable must be strictly bounded in the
(0, 1) interval.

\strong{Model Specification:}
The extended \code{\link[Formula]{Formula}} syntax is crucial for specifying
potentially different linear predictors for each relevant distribution parameter.
The parameters (\eqn{\alpha, \beta, \gamma, \delta, \lambda}) correspond sequentially
to the parts of the formula separated by \code{|}. For subfamilies where some
parameters are fixed by definition (see \code{family} argument), the corresponding
parts of the formula are automatically ignored. For example, in a \code{family = "kw"}
model, only the first two parts (for \eqn{\alpha} and \eqn{\beta}) are relevant.

\strong{Parameter Constraints and Link Functions:}
The parameters \eqn{\alpha, \beta, \gamma, \lambda} are constrained to be positive,
while \eqn{\delta} is constrained to the interval (0, 1). The default link functions
(\code{"log"} for positive parameters, \code{"logit"} for \eqn{\delta}) ensure these
constraints during estimation. Users can specify alternative link functions suitable
for the parameter's domain via the \code{link} argument.

\strong{Link Scales:}
The \code{link_scale} parameter allows users to control how aggressively the link
function transforms the linear predictor. For probability-type links (logit, probit,
cloglog, cauchy), smaller values (e.g., 1) produce more extreme transformations,
while larger values (e.g., 10) produce more gradual transformations. For continuous
parameters, scale values control the sensitivity of the transformation.

\strong{Families and Parameters:}
The function automatically handles parameter fixing based on the chosen \code{family}:
\itemize{
\item \strong{GKw}: All 5 parameters (\eqn{\alpha, \beta, \gamma, \delta, \lambda}) modeled.
\item \strong{BKw}: Models \eqn{\alpha, \beta, \gamma, \delta}; fixes \eqn{\lambda = 1}.
\item \strong{KKw}: Models \eqn{\alpha, \beta, \delta, \lambda}; fixes \eqn{\gamma = 1}.
\item \strong{EKw}: Models \eqn{\alpha, \beta, \lambda}; fixes \eqn{\gamma = 1, \delta = 0}.
\item \strong{Mc} (McDonald): Models \eqn{\gamma, \delta, \lambda}; fixes \eqn{\alpha = 1, \beta = 1}.
\item \strong{Kw} (Kumaraswamy): Models \eqn{\alpha, \beta}; fixes \eqn{\gamma = 1, \delta = 0, \lambda = 1}.
\item \strong{Beta}: Models \eqn{\gamma, \delta}; fixes \eqn{\alpha = 1, \beta = 1, \lambda = 1}. This parameterization corresponds to the standard Beta distribution with shape1 = \eqn{\gamma} and shape2 = \eqn{\delta}.
}

\strong{Estimation Engine:}
Maximum Likelihood Estimation (MLE) is performed using C++ templates via the
\code{TMB} package, which provides automatic differentiation and efficient
optimization capabilities. The specific TMB template used depends on the chosen \code{family}.

\strong{Optimizer Method (\code{method} argument):}
\itemize{
\item \code{"nlminb"}: Uses R's built-in \code{stats::nlminb} optimizer. Good for problems with box constraints. Default option.
\item \code{"Nelder-Mead"}: Uses R's \code{stats::optim} with the Nelder-Mead simplex algorithm, which doesn't require derivatives.
\item \code{"BFGS"}: Uses R's \code{stats::optim} with the BFGS quasi-Newton method for unconstrained optimization.
\item \code{"CG"}: Uses R's \code{stats::optim} with conjugate gradients method for unconstrained optimization.
\item \code{"SANN"}: Uses R's \code{stats::optim} with simulated annealing, a global optimization method useful for problems with multiple local minima.
\item \code{"L-BFGS-B"}: Uses R's \code{stats::optim} with the limited-memory BFGS method with box constraints.
}
}
\examples{
\donttest{
## -------------------------------------------------------------------------
## 1. Real-world Case Studies
## -------------------------------------------------------------------------

## Example 1: Food Expenditure Data
# Load required package
require(gkwreg)

# Get FoodExpenditure data and create response variable 'y' as proportion of income spent on food
food_data <- get_bounded_datasets("FoodExpenditure")
food_data <- within(food_data, {
  y <- food / income
})

# Define formula: y depends on 'persons' with 'income' as predictor for second parameter
formu_fe <- y ~ persons | income

# Fit Kumaraswamy model with log link for both parameters
kw_model <- gkwreg(formu_fe, food_data,
  family = "kw",
  link = rep("log", 2), method = "nlminb"
)

# Display model summary and diagnostics
summary(kw_model)
plot(kw_model, use_ggplot = TRUE, arrange_plots = TRUE, sub.caption = "")

## Example 2: Gasoline Yield Data
# Load GasolineYield dataset
gasoline_data <- get_bounded_datasets("GasolineYield")

# Formula: yield depends on batch and temperature
# First part (for alpha/gamma) includes batch and temp
# Second part (for beta/delta/phi) includes only temp
formu_gy <- yield ~ batch + temp | temp

# Fit Kumaraswamy model with log link and BFGS optimization
kw_model_gas <- gkwreg(formu_gy, gasoline_data,
  family = "kw",
  link = rep("log", 2), method = "BFGS"
)

# Display results
summary(kw_model_gas)
plot(kw_model_gas, use_ggplot = TRUE, arrange_plots = TRUE, sub.caption = "")

## Example 3: SDAC Cancer Data
# Load cancer survival dataset
sdac_data <- get_bounded_datasets("sdac")

# Formula: relative cumulative density ~ age adjustment + chemotherapy
formu_sd <- rcd ~ ageadj + chemo

# Fit Extended Kumaraswamy model
ekw_model_gas <- gkwreg(formu_sd, sdac_data, family = "ekw", method = "BFGS")
summary(ekw_model_gas)
plot(ekw_model_gas, use_ggplot = TRUE, arrange_plots = TRUE, sub.caption = "")

## Example 4: Retinal Data
# Load retinal dataset
retinal_data <- get_bounded_datasets("retinal")

# Formula for three parameters with different predictors
# alpha ~ LogT + LogT2 + Level
# beta  ~ LogT + Level
# gamma ~ Time
formu_rt <- Gas ~ LogT + LogT2 + Level | LogT + Level | Time

# Fit Extended Kumaraswamy model
ekw_model_ret <- gkwreg(formu_rt, retinal_data, family = "ekw", method = "nlminb")
summary(ekw_model_ret)
plot(ekw_model_ret, use_ggplot = TRUE, arrange_plots = TRUE, sub.caption = "")

## Example 5: Weather Task Agreement Data
# Load the WeatherTask dataset
df_weather <- get_bounded_datasets("WeatherTask")

# Fit all seven distribution families to the 'agreement' variable
fitall_weather <- gkwfitall(df_weather$agreement, method = "BFGS")

# Compare model performance
summary(fitall_weather) # Displays the comparison table

# Identify the best family based on AIC
best_family_code <- fitall_weather$comparison$Family[1]

# Refit the best model for detailed analysis
fit_best_weather <- gkwfit(
  df_weather$agreement,
  family = best_family_code,
  method = "BFGS", profile = TRUE, plot = TRUE, silent = TRUE
)

# Generate Goodness-of-Fit report
gof_report <- gkwgof(
  fit_best_weather,
  theme = ggplot2::theme_classic(),
  plot = TRUE, print_summary = FALSE, verbose = FALSE
)
summary(gof_report) # Display GoF statistics

# Extract fit statistics for all families
results_weathertask_df <- do.call(rbind, lapply(fitall_weather$fits, function(f) {
  extract_gof_stats(gkwgof(f,
    plot = FALSE,
    print_summary = FALSE, verbose = FALSE
  ))
}))
results_weathertask_df <- results_weathertask_df[order(results_weathertask_df$AIC), ]
row.names(results_weathertask_df) <- NULL

# Generate diagnostic plots for best model
plot(gkwgof(fit_best_weather, theme = ggplot2::theme_classic()), title = "")

# Display formatted comparison table
results_weathertask_df[
  ,
  c(
    "family", "n_params", "logLik", "AIC", "BIC",
    "KS", "AD", "RMSE", "pseudo_R2"
  )
]

## -------------------------------------------------------------------------
## 2. Simulation Studies
## -------------------------------------------------------------------------

## Example 1: Simple Kumaraswamy Regression Model
# Set seed for reproducibility
set.seed(123)
n <- 1000
x1 <- runif(n, -2, 2)
x2 <- rnorm(n)

# Define true regression coefficients
alpha_coef <- c(0.8, 0.3, -0.2) # Intercept, x1, x2
beta_coef <- c(1.2, -0.4, 0.1) # Intercept, x1, x2

# Generate linear predictors and transform using exponential link
eta_alpha <- alpha_coef[1] + alpha_coef[2] * x1 + alpha_coef[3] * x2
eta_beta <- beta_coef[1] + beta_coef[2] * x1 + beta_coef[3] * x2
alpha_true <- exp(eta_alpha)
beta_true <- exp(eta_beta)

# Generate responses from Kumaraswamy distribution
y <- rkw(n, alpha = alpha_true, beta = beta_true)
df1 <- data.frame(y = y, x1 = x1, x2 = x2)

# Fit Kumaraswamy regression model with formula notation
# Model: alpha ~ x1 + x2 and beta ~ x1 + x2
kw_reg <- gkwreg(y ~ x1 + x2 | x1 + x2, data = df1, family = "kw", silent = TRUE)

# Alternative model with custom link scales
kw_reg2 <- gkwreg(y ~ x1 + x2 | x1 + x2,
  data = df1, family = "kw",
  link_scale = list(alpha = 5, beta = 8), silent = TRUE
)

# Display model summary
summary(kw_reg)

## Example 2: Generalized Kumaraswamy Regression
# Set seed for reproducibility
set.seed(456)
n <- 1000
x1 <- runif(n, -1, 1)
x2 <- rnorm(n)
x3 <- factor(rbinom(n, 1, 0.5), labels = c("A", "B")) # Factor variable

# Define true regression coefficients for all parameters
alpha_coef <- c(0.5, 0.2) # Intercept, x1
beta_coef <- c(0.8, -0.3, 0.1) # Intercept, x1, x2
gamma_coef <- c(0.6, 0.4) # Intercept, x3B
delta_coef <- c(0.0, 0.2) # Intercept, x3B (logit scale)
lambda_coef <- c(-0.2, 0.1) # Intercept, x2

# Create design matrices
X_alpha <- model.matrix(~x1, data = data.frame(x1 = x1))
X_beta <- model.matrix(~ x1 + x2, data = data.frame(x1 = x1, x2 = x2))
X_gamma <- model.matrix(~x3, data = data.frame(x3 = x3))
X_delta <- model.matrix(~x3, data = data.frame(x3 = x3))
X_lambda <- model.matrix(~x2, data = data.frame(x2 = x2))

# Generate parameters through linear predictors and appropriate link functions
alpha <- exp(X_alpha \%*\% alpha_coef)
beta <- exp(X_beta \%*\% beta_coef)
gamma <- exp(X_gamma \%*\% gamma_coef)
delta <- plogis(X_delta \%*\% delta_coef) # logit link for delta
lambda <- exp(X_lambda \%*\% lambda_coef)

# Generate response from Generalized Kumaraswamy distribution
y <- rgkw(n, alpha = alpha, beta = beta, gamma = gamma, delta = delta, lambda = lambda)
df2 <- data.frame(y = y, x1 = x1, x2 = x2, x3 = x3)

# Fit GKw regression with parameter-specific formulas
gkw_reg <- gkwreg(y ~ x1 | x1 + x2 | x3 | x3 | x2, data = df2, family = "gkw")

# Alternative model with custom link scales
gkw_reg2 <- gkwreg(y ~ x1 | x1 + x2 | x3 | x3 | x2,
  data = df2, family = "gkw",
  link_scale = list(
    alpha = 12, beta = 12, gamma = 12,
    delta = 0.8, lambda = 12
  )
)

# Compare true vs. estimated coefficients
print("Estimated Coefficients (GKw):")
print(coef(gkw_reg))
print("True Coefficients (approx):")
print(list(
  alpha = alpha_coef, beta = beta_coef, gamma = gamma_coef,
  delta = delta_coef, lambda = lambda_coef
))

## Example 3: Beta Regression for Comparison
# Set seed for reproducibility
set.seed(789)
n <- 1000
x1 <- runif(n, -1, 1)

# True coefficients for Beta parameters (gamma = shape1, delta = shape2)
gamma_coef <- c(1.0, 0.5) # Intercept, x1 (log scale)
delta_coef <- c(1.5, -0.7) # Intercept, x1 (log scale)

# Generate parameters through linear predictors and log link
X_beta_eg <- model.matrix(~x1, data.frame(x1 = x1))
gamma_true <- exp(X_beta_eg \%*\% gamma_coef)
delta_true <- exp(X_beta_eg \%*\% delta_coef)

# Generate response from Beta distribution
y <- rbeta_(n, gamma_true, delta_true)
df_beta <- data.frame(y = y, x1 = x1)

# Fit Beta regression model using gkwreg
beta_reg <- gkwreg(y ~ x1 | x1,
  data = df_beta, family = "beta",
  link = list(gamma = "log", delta = "log")
)

## Example 4: Model Comparison using AIC/BIC
# Fit an alternative model (Kumaraswamy) to the same beta-generated data
kw_reg2 <- try(gkwreg(y ~ x1 | x1, data = df_beta, family = "kw"))

# Compare models using information criteria
print("AIC Comparison (Beta vs Kw):")
c(AIC(beta_reg), AIC(kw_reg2))
print("BIC Comparison (Beta vs Kw):")
c(BIC(beta_reg), BIC(kw_reg2))

## Example 5: Prediction with Fitted Models
# Create new data for predictions
newdata <- data.frame(x1 = seq(-1, 1, length.out = 20))

# Predict expected response (mean of the Beta distribution)
pred_response <- predict(beta_reg, newdata = newdata, type = "response")

# Predict parameters on the scale of the link function
pred_link <- predict(beta_reg, newdata = newdata, type = "link")

# Predict parameters on the original scale
pred_params <- predict(beta_reg, newdata = newdata, type = "parameter")

# Visualize fitted model and data
plot(df_beta$x1, df_beta$y,
  pch = 20, col = "grey", xlab = "x1", ylab = "y",
  main = "Beta Regression Fit (using gkwreg)"
)
lines(newdata$x1, pred_response, col = "red", lwd = 2)
legend("topright", legend = "Predicted Mean", col = "red", lty = 1, lwd = 2)
}

}
\references{
Kumaraswamy, P. (1980). A generalized probability density function for double-bounded
random processes. \emph{Journal of Hydrology}, \strong{46}(1-2), 79-88.

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized distributions.
\emph{Journal of Statistical Computation and Simulation}, \strong{81}(7), 883-898.

Ferrari, S. L. P., & Cribari-Neto, F. (2004). Beta regression for modelling rates and
proportions. \emph{Journal of Applied Statistics}, \strong{31}(7), 799-815.

Kristensen, K., Nielsen, A., Berg, C. W., Skaug, H., & Bell, B. M. (2016). TMB:
Automatic Differentiation and Laplace Approximation. \emph{Journal of Statistical
Software}, \strong{70}(5), 1-21.
(Underlying TMB package)

Zeileis, A., Kleiber, C., Jackman, S. (2008). Regression Models for Count Data in R.
\emph{Journal of Statistical Software}, \strong{27}(8), 1-25.

Smithson, M., & Verkuilen, J. (2006). A Better Lemon Squeezer? Maximum-Likelihood
Regression with Beta-Distributed Dependent Variables. \emph{Psychological Methods},
\strong{11}(1), 54–71.
}
\seealso{
\code{\link{summary.gkwreg}}, \code{\link{predict.gkwreg}},
\code{\link{plot.gkwreg}}, \code{\link{coef.gkwreg}}, \code{\link{vcov.gkwreg}},
\code{\link[stats]{logLik}}, \code{\link[stats]{AIC}},
\code{\link[Formula]{Formula}}, \code{\link[TMB]{MakeADFun}},
\code{\link[TMB]{sdreport}}
}
\author{
Lopes, J. E.
}
\keyword{hoss}
\keyword{models}
\keyword{regression}
