#' Draw Text Annotations to a ggplot Object
#' @description
#' This function overlays text annotations onto any ggplot object. It is particularly useful for adding annotations from CSV files generated by the ggsem Shiny app but can also be used with custom annotation data.
#' @param p A ggplot object. The plot to which the annotations will be added.
#' @param annotations_data A data frame containing annotation information. Typically, this comes from a CSV file generated by the ggsem Shiny app. The required columns include:
#' \itemize{
#'   \item \code{text}: The text to annotate (character).
#'   \item \code{x}, \code{y}: The coordinates for the text (numeric).
#'   \item \code{font}: The font family to use (character, e.g., "serif").
#'   \item \code{size}: The size of the text (numeric).
#'   \item \code{color}: The color of the text (character, valid hex color).
#'   \item \code{angle}: The rotation angle of the text (numeric, in degrees).
#'   \item \code{alpha}: The transparency of the text (numeric, 0 to 1).
#'   \item \code{fontface}: The font style (character, e.g., "bold").
#'   \item \code{math_expression}: Logical, whether the text should be parsed as a mathematical expression.
#' }
#'
#' @param zoom_level Numeric. Adjusts the size of annotations based on the zoom level. Default is \code{1}.
#'
#' @return
#' A ggplot object with the specified annotations added.
#' @export
#'
#' @examples
#' library(ggplot2)
#'
#' annotations_data <- data.frame(
#' text = 'Square One', x = 26, y = 300, font = 'serif',
#' size = 20, color = '#000000', angle = 0, alpha = 1,
#' fontface = 'bold', math_expression = FALSE,
#' lavaan = FALSE, network = FALSE, locked = FALSE
#' )
#'
#' p <- ggplot()
#'
#' draw_annotations(p, annotations_data, zoom_level = 1.2)
draw_annotations <- function(p, annotations_data, zoom_level = 1) {
  if (nrow(annotations_data) > 0) {
    annotations_data$color <- sapply(annotations_data$color, valid_hex)
    annotations_data$alpha <- sapply(annotations_data$alpha, valid_alpha)
    annotations_data$fontface <- sapply(annotations_data$fontface, valid_fontface)
    annotations_data$font <- sapply(annotations_data$font, valid_font)

    for (i in 1:nrow(annotations_data)) {
      annotation_text <- if (annotations_data$math_expression[i]) {
        suppressWarnings(tryCatch(parse(text = annotations_data$text[i]), error = function(e) annotations_data$text[i]))
      } else {
        annotations_data$text[i]
      }

      adjusted_size <- (annotations_data$size[i] / 3) / zoom_level

      # Add annotation to the plot
      p <- p + annotate("text",
                        x = annotations_data$x[i],
                        y = annotations_data$y[i],
                        label = annotation_text,
                        size = adjusted_size,
                        color = annotations_data$color[i],
                        alpha = annotations_data$alpha[i],
                        angle = annotations_data$angle[i],
                        family = annotations_data$font[i],
                        fontface = annotations_data$fontface[i]
      )
    }
  }
  return(p)
}
