% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/position-nudge-line.R
\name{position_nudge_line}
\alias{position_nudge_line}
\title{Nudge labels away from a line}
\usage{
position_nudge_line(
  x = NA_real_,
  y = NA_real_,
  xy_relative = c(0.03, 0.03),
  abline = NULL,
  method = NULL,
  formula = y ~ x,
  direction = NULL,
  line_nudge = 1
)
}
\arguments{
\item{x, y}{Amount of vertical and horizontal distance to move. A numeric
vector of length 1, or of the same length as rows there are in `data`.}

\item{xy_relative}{Nudge relative to _x_ and _y_ data expanse, ignored
unless `x` and `y` are both `NA`s.}

\item{abline}{a vector of length two giving the intercept and slope.}

\item{method}{One of `"spline"`, `"lm"` or `"auto"`.}

\item{formula}{A model formula for [lm()] when `method = "lm"`. Ignored
otherwise.}

\item{direction}{One of "none", or "split".}

\item{line_nudge}{A positive multiplier >= 1, increasing nudging
away from the curve or line compared to nudging from points.}
}
\value{
A \code{"Position"} object.
}
\description{
`position_nudge_line` is generally useful for adjusting the starting
position of labels or text to be repelled while preserving the original
position as the start of the segments. The difference compared to
[position_nudge_center()] is that the nudging is away from from a line or
curve fitted to the data points or supplied as coefficients. While
[position_nudge_center()] is most useful for "round-shaped", vertically- or
horizontally elongated clouds of points, [position_nudge_line()] is most
suitable when observations follow a linear or curvilinear relationship
between _x_ and _y_ values. In contrast to [ggplot2::position_nudge],
`position_nudge_line()` returns in `data` both the original
coordinates and the nudged coordinates.
}
\details{
The default ammount of nudging is 3% of the spread of the data along
  _x_ and _y_ axes, which in most cases is good. In most cases it is best to
  apply nudging along a direction perpendicular to the line or curve, if this
  is the aim, passing an argument to only one of `x`, `y` or `xy_relative`
  will be enough. When `direction = "split"` nudging is away from an implicit
  line or curve on either side with positive nudging. The line of curve can
  be smooth spline or linear regression fitted on-the-fly to the data points,
  or a straight line defined by its coefficients passed to `abline`. The
  fitting is well defined only if the observations fall roughly on a curve or
  straight line that is monotonic in `y`. By means of `line_nudge` one can
  increment nudging away from the line or curve compared to away from the
  points, which is useful for example to keep labels outside of a confidence
  band. Direction defaults to `"split"` when `line_nudge > 1`, and otherwise
  to `"none"`.
}
\note{
For `method = "lm"` only model formulas corresponding to polynomials
  with no missing terms are supported. If using [poly()], `raw = TRUE` is
  required.

  In practice, `x` and `y` should have the same sign for nudging to work
  correctly.

  This position is most useful when labeling points conforming a cloud along
  an arbitrary curve or line.
}
\examples{

set.seed(16532)
df <- data.frame(
  x = -10:10,
  y = (-10:10)^2,
  yy = (-10:10)^2 + rnorm(21, 0, 4),
  yyy = (-10:10) + rnorm(21, 0, 4),
  l = letters[1:21]
)

# Setting the nudging distance

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line())

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(xy_relative = -0.03))

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(x = 0.6))

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(y = 3.2))

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(x = 0.6, y = 3.2))

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(x = -0.6, y = -4))

# Other curves, using defaults

ggplot(df, aes(x, -y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line())

ggplot(df, aes(x, y - 40, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line())

ggplot(subset(df, x >= 0), aes(y, sqrt(y), label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line())

# nudging outwards and downwards from a curve

ggplot(subset(df, x >= 0), aes(y, sqrt(y), label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(xy_relative = -0.03))

# an arbitrary straight line

ggplot(df, aes(x, x * 2 + 5, label = l)) +
  geom_abline(intercept = 5, slope = 2, linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(abline = c(5, 2)))

# Points scattered near a curve or line, we use 'direction = "split"'

ggplot(subset(df, x >= 0), aes(x, yyy)) +
  stat_smooth(method = "lm", formula = y ~ x) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(direction = "split"))

ggplot(df, aes(x)) +
  geom_line(aes(y = y), linetype = "dotted") +
  geom_point(aes(y = yy)) +
  geom_text(aes(y = yy, label = l),
            position = position_nudge_line(direction = "split"))

ggplot(subset(df, x >= 0), aes(y, yy)) +
  stat_smooth(method = "lm", formula = y ~ x) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(direction = "split"))

# increasing the nudging for labels near the line

ggplot(subset(df, x >= 0), aes(y, yy)) +
  stat_smooth(method = "lm", formula = y ~ x) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(line_nudge = 2,
                                           direction = "split"))

# fitting a linear model instead of the default spline

ggplot(subset(df, x >= 0), aes(y, yy)) +
  stat_smooth(method = "lm", formula = y ~ x) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(method = "lm",
                                           direction = "split"))

ggplot(subset(df, x >= 0), aes(x, x^2)) +
  stat_smooth(method = "lm", formula = y ~ poly(x, 2, raw = TRUE)) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(method = "lm",
                                           formula = y ~ poly(x, 2, raw = TRUE)))

ggplot(subset(df, x >= 0), aes(x, x^2)) +
  stat_smooth(method = "lm", formula = y ~ x + I(x^2)) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(method = "lm",
                                           formula = y ~ x + I(x^2)))

# grouping is supported

df <- data.frame(x = rep(1:10, 2),
                 y = c(1:10, 10:1),
                 group = rep(c("a", "b"), c(10, 10)),
                 l = "+")

ggplot(df, aes(x, y, label = l, color = group)) +
  geom_line(linetype = "dotted") +
  geom_text() +
  geom_text(position = position_nudge_line()) +
  geom_text(position = position_nudge_line(xy_relative = -0.03))

# one needs to ensure that grouping is in effect in the geoms with nudging

ggplot(df, aes(x, y, label = l, color = group, group = group)) +
  geom_line(linetype = "dotted") +
  geom_text() +
  geom_text(color = "red",
            position = position_nudge_line()) +
  geom_text(color = "blue",
            position = position_nudge_line(xy_relative = -0.03)) +
  coord_equal()

# facets are also supported

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_text() +
  geom_text(position = position_nudge_line(xy_relative = c(0.06, 0.03)),
            color = "red") +
  geom_text(position = position_nudge_line(xy_relative = -c(0.06, 0.03)),
            color = "blue") +
  facet_wrap(~group) +
  coord_equal(ratio = 1.5)

}
\seealso{
[ggplot::position_nudge()], [ggrepel::position_nudge_repel()].

Other position adjustments: 
\code{\link{position_nudge_center}()},
\code{\link{position_nudge_to}()}
}
\concept{position adjustments}
