% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gerbil.R
\name{gerbil}
\alias{gerbil}
\title{General Efficient Regression-Based Imputation with Latent processes}
\usage{
gerbil(
  dat,
  m = 1,
  mcmciter = 25,
  predMat = NULL,
  type = NULL,
  visitSeq = NULL,
  ords = NULL,
  semi = NULL,
  bincat = NULL,
  cont.meth = "EMP",
  num.cat = 12,
  r = 5,
  verbose = TRUE,
  n.cores = NULL,
  cl.type = NULL,
  mass = rep(0, length(semi)),
  ineligible = NULL,
  trace = TRUE,
  seed = NULL
)
}
\arguments{
\item{dat}{The dataset that is to be imputed.  Missing values must be coded with \code{NA}.}

\item{m}{The number of multiply imputed datasets to be created.  By default, \code{m = 1}.}

\item{mcmciter}{The number of iterations of Markov chain Monte Carlo that will be used to create each imputed dataset. By default, \code{m = 25}.}

\item{predMat}{A numeric matrix of \code{ncol(dat)} columns and no more than \code{nrow(dat)} rows, containing 0/1 data specifying the set of predictors to be used for each target row. Each row corresponds to a variable. A value of 1 means that the column variable is used as a predictor for the variable in the target row. By default, \code{predMat} is a square matrix of \code{ncol(dat)} rows and columns with 1's below the diagonal and 0's on and above the diagonal.  Any non-zero value on or above the diagonal will be set to zero.}

\item{type}{A named vector that gives the type of each variable contained in \code{dat}.  Possible types include \code{'binary'}, \code{'categorical'}, \code{'ordinal'}, \code{'semicont'} (semi-continuous), and \code{'continuous'}.  The vector type should be named where the names indicate the corresponding column of \code{dat}.  Types for variables not listed in type will be determined by default, in which case a variable with no more than \code{num.cat} possible values will be set as binary/categorical and is set as continuous otherwise.}

\item{visitSeq}{A vector of variable names that has (at least) contains all names of each column of \code{dat} that has missing values.  Within the I-Step and P-Step of gerbil, the variables will be modeled and imputed in the sequence given by \code{visitSeq}.   If \code{visitSeq = TRUE}, \code{visitSeq} is reset as being equal to the columns of \code{dat} ordered from least to most missingness.  If \code{visitSeq = NULL} (the default) or \code{visitSeq = FALSE} variables are ordered in accordance with the order of the rows of \code{predMat} or (if unavailable) the order in which they appear in the \code{dat}.}

\item{ords}{A character string giving a set of the column names of \code{dat} that indicate which variables are to be treated as ordinal. Elements of \code{ords} are overridden by any conflicting information in \code{type}. By default, \code{ords = NULL}.}

\item{semi}{A character string giving a set of the column names of \code{dat} that indicate which variables are to be treated as semi-continuous. Elements of \code{semi} are overridden by any conflicting information in \code{type}. By default, \code{semi = NULL}.}

\item{bincat}{A character string giving a set of the column names of \code{dat} that indicate which variables are to be treated as binary or unordered categorical. Elements of \code{bincat} are overridden by any conflicting information in \code{type}. By default, \code{bincat = NULL}.}

\item{cont.meth}{The type of marginal transformation used for continuous variables.  Set to \code{"EMP"} by default for the empirical distribution transformation of Robbins (2014). The current version also includes an option for no transformation (\code{cont.meth = "none"}). Other transformation types will be available in future versions of \code{gerbil}.  .}

\item{num.cat}{Any variable that does not have a type specified by any of the other parameters will be treated as categorical if it takes on no more than \code{num.cat} possible values and as continuous if it takes on more than \code{num.cat} possible values. By default, \code{num.cat = 12}.}

\item{r}{The number of pairwise completely observed cases that must be available for any pair of variables to have dependencies enabled within the conditional models for imputation. By default, \code{r = 5}.}

\item{verbose}{If \code{TRUE} (the default), history is printed on console. Use \code{verbose = FALSE} for silent computation.}

\item{n.cores}{The number of CPU cores to use for parallelization. If \code{n.cores} is not specified by the user, it is guessed using the \code{detectCores} function in the parallel package.  If \code{TRUE}  (the default), it is set as \code{detectCores()}.  If \code{NULL}, it is set as \code{floor((detectCores()+1)/2)}.  If \code{FALSE}, it is set as \code{1}, in which case parallelization is not invoked.  Note that the documentation for \code{detectCores} makes clear that it is not failsafe and could return a spurious number of available cores. By default, \code{n.cores} is set as \code{floor((n + 1)/2)}, where \code{n} is the number of available clusters.}

\item{cl.type}{The cluster type that is passed into the \code{makeCluster()} function in the \code{parallel} package.  Defaults to \code{'PSOCK'}.}

\item{mass}{A named vector of the same length as the number of semi-continuous variables in \code{dat} that gives the location (value) of the point mass for each such variable. The point of mass for each semicontinuous variable is set to zero by default.}

\item{ineligible}{Either a scalar or a matrix that is used to determined which values are to be considered missing but ineligible for imputation. Such values will be imputed internally within \code{gerbil} to ensure a coherent imputation model but will be reset as missing after imputations have been created. If \code{ineligible} is a scalar, all data points that take on the respective value will be considered missing but ineligible for imputation. If \code{ineligible} is a matrix (with the same number of rows as \code{dat} and column names that overlap with \code{dat}), entries of \code{TRUE} or \code{1} in \code{ineligible} indicate values that are missing but ineligible for imputation. If \code{ineligible = NULL} (the default), all missing values will be considered eligible for imputation.}

\item{trace}{A logical that, if \code{TRUE}, implies that means and variances of variables are tracked across iterations. Set to \code{FALSE} to save computation time. However, trace plots and R hat statistics are disabled for \code{gerbil} objects created with \code{trace = FALSE}. Defaults to \code{TRUE}.}

\item{seed}{An integer that, when specified, is used to set the random number generator via \code{set.seed()}.}
}
\value{
\code{gerbil()} returns an object the class \code{gerbil} that contains the following slots:

\describe{
       \item{imputed}{A list of length \code{m} that contains the imputed datasets.}
       \item{missing}{A matrix \code{0}s, \code{1}s, \code{2}s, and \code{4}s of the same dimension as \code{dat} that indicates which values were observed or missing.  A \code{0} indicates a fully observed value, a \code{1} indicates a missing value that was imputed, and a \code{4} indicates a missing value that was ineligible for imputation.}
       \item{summary}{A matrix with \code{ncol(dat)} number of rows that contains summary information, including the type of each variable and missingness rates. Note that for continuous variables, the type listed indicates the method of transformation used.}
       \item{chainSeq}{A list of six elements. Each element is a matrix with \code{mcmciter} columns and up to \code{ncol(dat)} rows. Objects \code{means.all} and \code{means.mis} give the variables means of data process across iterations of MCMC when all observations are incorporated and when only imputed values are incorporated, respectively. (Means of continuous variables are given on the transformed scale.) Similar objects are provided to track variances of variables. Variables are listed in the order provided by the \code{gerbil} object \code{visitSeq.latent}. Variables reported in this output are those contained in the dataset that has been expanded to include binary indicators for categorical and semi-continuous variables.}
       \item{R.hat}{The value of the R hat statistics of Gelman and Rubin (1992) for the means and variances of each variable. The R hat statistic is also provided for mean of binary variables.  Variables include those contained in the expanded dataset and are listed in the order provided by object \code{visitSeq.latent}. Only calculated if \code{m > 2} and \code{mcmciter >= 4}.}
       \item{missing.latent}{A matrix of the same dimensions as the expanded dataset, but used to indicate missingness in the expanded dataset. In this matrix, \code{0}s indicate fully observed values, \code{1}s indicate fully missing values, \code{3}s indicate values that have imposed missingness (for binary indicators corresponding to categorical or semi-continuous variables), and \code{4} indicates a missing value that is ineligible for imputation (as determined by the input \code{'ineligible'})..}
       \item{visitSeq.initial}{A vector of variable names giving the sequential ordering of variables that is used for imputation prior to expanding the dataset include nested binary and point-mass indicators.  Variables without missing values are excluded.}
       \item{visitSeq.final}{A vector of variable names giving the sequential ordering of variables in the expanded dataset that is used for imputation.  Variables without missing values are excluded.}
       \item{predMat.initial}{A matrix of ones and zeros indicating the dependencies enabled in the conditional models used for imputation.  This matrix is determined from the input 'predMat'. Rows corresponding to variables with no missing values are removed.}
       \item{predMat.final}{A matrix of ones and zeros indicating the dependencies enabled in the conditional models used for imputation.  This is of a similar format to the input 'predMat' but pertains to the expanded dataset.  Rows corresponding to variables with no missing values are removed.}
       \item{drops}{A list of length equal to the number of variables in the expanded dataset that have missing values.  Elements of the list indicate which variables were dropped from the conditional model for the corresponding variable due to either insufficient pairwise complete observations (see the input 'r') or perfect collinearities.}
       \item{forms}{A list of length equal to the number of variables in the expanded dataset that have missing values.  Elements of the list indicate the regression formula used for imputation of the respective variable.}
       \item{mass.final}{The final version of the input parameter \code{mass}.}
       \item{ineligibles}{A logical matrix with the same number of rows and columns as \code{dat} that indicates which elements are considered missing but ineligible for imputation.}
       \item{nams.out}{A vector used to link column names in the expanded data to corresponding names in the original data.}
}
}
\description{
Coherent multiple imputation of general multivariate data as implemented through the GERBIL algorithm described by Robbins (2020).  The algorithm is

\itemize{
\item \strong{coherent} in that imputations are sampled from a valid joint distribution, ensuring MCMC convergence;
\item \strong{general} in that data of general structure (binary, categorical, continuous, etc.) may be allowed;
\item \strong{efficient} in that computational performance is optimized using the SWEEP operator for both modeling and sampling;
\item \strong{regression-based} in that the joint distribution is built through a sequence of conditional regression models;
\item \strong{latent} in that a latent multivariate normal process underpins all variables; and
\item \strong{flexible} in that the user may specify which dependencies are enabled within the conditional models.
}
}
\details{
\code{gerbil} is designed to handle the following classes of variables:

\itemize{

\item \code{'continuous'}: Variables are transformed to be (nearly) standard normal prior to imputation. The default transformation method is based on empirical distributions (see Robbins, 2014) and ensures that imputed values of a variable are sampled from the observed values of that variable.
\item \code{'binary'}: Dichotomous variables are handled through probit-type models in that they are underpinned by a unit-variance normally distributed random variable.
\item \code{'categorical'}: Unordered categorical variables are handled by creating nested binary variables that underpin the categorical data.  Missingness is artificially imposed in the nested variables in order to ensure conditional independence between them.  See Robbins (2020) for details.
\item \code{'ordinal'}: Ordered categorical variables (ordinal) are handled through a probit-type model in that a latent normal distribution is assumed to underpin the ordinal observations.  See Robbins (2020) for details.
\item \code{'semicont'}: Mixed discrete/continuous (semi-continuous) variables are assumed to observe a mass at a specific value (most often zero) and are continuous otherwise.  A binary variable is created that indicates whether the semi-continuous variable takes on the point-mass value; the continuous portion is set as missing when the observed semi-continuous variable takes on the value at the point-mass.  See Robbins et al. (2013) for details.
}

The parameter \code{type} allows the user to specify the class for each variable.  Routines are in place to establish the class by default for variables not stated in \code{type}. Note that it is not currently possible for a variable to be assigned a class of semi-continuous by default.

\code{gerbil} uses a joint modeling approach to imputation that builds a joint model using a sequence of conditional models, as outlined in Robbins et al. (2013).
This approach differs from fully conditional specification in that the regression model for any given variable is only allowed to depend upon variables that preceed it in an index ordering.
The order is established by the parameter \code{visitSeq}. \code{gerbil} contains the flexibility to allow its user to establish which of the permissible dependencies are enabled within the conditional models.
Enabled dependencies are stated within the parameter \code{predMat}.  Note that the data matrix used for imputation is an expanded version of the data that are fed into the algorithm (variables are created that underpin unordered categorical and semi-continuous variables).
Note also that conditional dependencies between the nested binary variables of a single undordered categorical variables or the discrete and continuous portions of a semi-continuous variable are not permitted.

The output of \code{gerbil} is an object of class \code{gerbil} which is a list that contains the imputed datasets (\code{imputed}), missingness indicators (\code{missing} and \code{missing.latent}), summary information (\code{summary}), output used for MCMC convergence diagostics (\code{chainSeq} and \code{R.hat}),
and modeling summaries (\code{visitSeq.initial}, \code{visitSeq.final}, \code{predMat.initial}, \code{predMat.final}, \code{drops}, and \code{forms}).
Some output regarding convergence diagnostics and modeling regards the expanded dataset used for imputation (the expanded dataset includes binary indicators for unordered categorical and semi-continuous variables).
Note that the nested binary variables corresponding to an unordered categorical variable \code{X} with categories labeled \code{a}, \code{b}, \code{c}, etc., are named \code{X.a}, \code{X.b}, \code{X.c}, and so forth in the expanded dataset.
Likewise, the binary variable indicating the point mass of a semi-continuous variable \code{Y} is named \code{Y.B} in the expanded dataset, and the positive portion (with missingness imposed) is left as being named \code{Y}.

\code{gerbil} automatically checks each regression model for perfect collinearities and reduces the model as needed. 
Variables that have been dropped from a given model are listed in the element named \code{'drops'} in a \code{gerbil} object.
}
\examples{
#Load the India Human Development Survey-II dataset
data(ihd_mcar) 

# Gerbil without types specified
imps.gerbil <- gerbil(ihd_mcar, m = 1, mcmciter = 10)
\donttest{
# Gerbil with types specified (method #1)
types.gerbil <- c(
       sex = "binary", age = "continuous", 
       marital_status = "binary", job_field = "categorical", 
       farm_labour_days = "semicont", own_livestock = "binary", 
       education_level = "ordinal", income = "continuous")
imps.gerbil <- gerbil(ihd_mcar, m = 1, type = types.gerbil)

# Gerbil with types specified (method #2)
imps.gerbil <- gerbil(ihd_mcar, m = 1, ords = "education_level", semi = "farm_labour_days", 
       bincat = c("sex", "marital_status", "job_field", "own_livestock"))

# Gerbil with types specified (method #3)
types.gerbil <- c("binary", "continuous", "binary", "categorical", "semicont", 
       "binary", "ordinal", "continuous")
imps.gerbil <- gerbil(ihd_mcar, m = 1, type = types.gerbil)

# Variables of class factor are treated as binary/categorical by default
ihd.fac <- ihd_mcar
ihd.fac$sex <- factor(ihd_mcar$sex)
ihd.fac$marital_status <- factor(ihd_mcar$marital_status)
ihd.fac$job_field <- factor(ihd_mcar$job_field)
ihd.fac$own_livestock <- factor(ihd_mcar$own_livestock)
ihd.fac$education_level <- ordered(ihd_mcar$education_level)
imps.gerbil <- gerbil(ihd.fac, m = 1)

# Univariate plotting of one variable
plot(imps.gerbil, type = 1, y = "job_field")

# gerbil with predMat specified (method #1)
predMat <- matrix(c(1, 0, 0, 1), 2, 2)
dimnames(predMat) <- list(c("education_level", "income"), c("sex", "job_field"))
imps.gerbil <- gerbil(ihd_mcar, m = 1, type = types.gerbil, predMat = predMat)

# gerbil with predMat specified (method #2)
predMat <- rbind(
       c(0, 0, 0, 0, 0, 0, 0, 0), 
       c(1, 0, 0, 0, 0, 0, 0, 0), 
       c(1, 1, 0, 0, 0, 0, 0, 0), 
       c(1, 1, 1, 0, 0, 0, 0, 0), 
       c(1, 1, 1, 1, 0, 0, 0, 0), 
       c(1, 1, 1, 1, 1, 0, 0, 0), 
       c(1, 1, 1, 0, 1, 1, 0, 0), 
       c(0, 1, 1, 1, 1, 1, 1, 0) 
       )
imps.gerbil <- gerbil(ihd_mcar, type = types.gerbil, predMat = predMat)

# Multiple imputation with more iterations
imps.gerbil.5 <- gerbil(ihd_mcar, m = 5, mcmciter = 100, ords = "education_level", 
       semi = "farm_labour_days", bincat = "job_field", n.cores = 1)

plot(imps.gerbil.5, type = 1, y = "job_field", imp = 1:5) 

# Extract the first imputed dataset
imputed.gerb <- imputed(imps.gerbil.5, imp = 1)

# Write all imputed datasets to an Excel file
write.gerbil(imps.gerbil.5, file = file.path(tempdir(), "gerbil_example.xlsx"), imp = 1:5)

if(requireNamespace('mice')){
# Impute using mice for comparison

types.mice <- c("logreg", "pmm", "logreg", "polyreg", "pmm", "logreg", "pmm", "pmm")
imps.mice <- mice(ihd.fac, m = 1, method = types.mice, maxit = 100)

imps.mice1 <- mice(ihd.fac, m = 1, method = "pmm", maxit = 100)

imps.gerbil <- gerbil(ihd_mcar, m = 1, mcmciter = 100, ords = "education_level", 
    semi = "farm_labour_days", bincat = "job_field")

# Compare the performance of mice and gerbil

# Replace some gerbil datasets with mice datasets
imps.gerbil.m <- imps.gerbil.5
imps.gerbil.m$imputed[[2]] <- complete(imps.mice, action = 1)
imps.gerbil.m$imputed[[3]] <- complete(imps.mice1, action = 1)

# Perform comparative correaltion analysis
cor_gerbil(imps.gerbil.m, imp = 1, log = "income")
cor_gerbil(imps.gerbil.m, imp = 2, log = "income")
cor_gerbil(imps.gerbil.m, imp = 3, log = "income")

# Perform comparative univariate goodness-of-fit testing
gof_gerbil(imps.gerbil.m, type = 1, imp = 1)
gof_gerbil(imps.gerbil.m, type = 1, imp = 2)
gof_gerbil(imps.gerbil.m, type = 1, imp = 3)

# Perform comparative bivariate goodness-of-fit testing
gof_gerbil(imps.gerbil.m, type = 2, imp = 1)
gof_gerbil(imps.gerbil.m, type = 2, imp = 2)
gof_gerbil(imps.gerbil.m, type = 2, imp = 3)

# Produce univariate plots for comparisons 
plot(imps.gerbil.m, type = 1, file = file.path(tempdir(), "gerbil_vs_mice_univariate.pdf"), 
     imp = c(1, 2, 3), log = "income", lty = c(1, 2, 4, 5), col = c("blue4", "brown2", 
     "green3", "orange2"), legend = c("Observed", "gerbil", "mice: logistic", "mice: pmm"))

### Produce bivariate plots for comparisons 
plot(imps.gerbil.m, type = 2, file = file.path(tempdir(), "gerbil_vs_mice_bivariate.pdf"), 
    imp = c(1, 2, 3), log = "income", lty = c(1, 2, 4, 5), col = c("blue4", "brown2", 
    "green3", "orange2"), pch = c(1, 3, 4, 5), legend = c("Observed", "gerbil", 
    "mice: logistic", "mice: pmm"))

}
}
}
\references{
Gelman, A., & Rubin, D. B. (1992). Inference from iterative simulation using multiple sequences. \emph{Statistical Science}, 7(4), 457-472.

  Robbins, M. W. (2014). The Utility of Nonparametric Transformations for Imputation of Survey Data. \emph{Journal of Official Statistics}, 30(4), 675-700.

  Robbins, M. W. (2020). A flexible and efficient algorithm for joint imputation of general data. arXiv preprint arXiv:2008.02243.

  Robbins, M. W., Ghosh, S. K., & Habiger, J. D. (2013). Imputation in high-dimensional economic data as applied to the Agricultural Resource Management Survey. \emph{Journal of the American Statistical Association}, 108(501), 81-95.
}
