% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trajectory.analysis.r
\name{trajectory.analysis}
\alias{trajectory.analysis}
\title{Quantify and compare shape change trajectories}
\usage{
trajectory.analysis(f1, f2 = NULL, iter = 999, seed = NULL,
  traj.pts = NULL, data = NULL, print.progress = TRUE)
}
\arguments{
\item{f1}{A formula for the linear model, for trajectories (e.g., Y ~ A or Y ~ A * B).
The right hand side of this formula can contain only one or two factors.}

\item{f2}{A formula for additional covariates  (e.g.,  ~ x1 + x2)}

\item{iter}{Number of iterations for significance testing}

\item{seed}{An optional argument for setting the seed for random permutations of the resampling procedure.
If left NULL (the default), the exact same P-values will be found for repeated runs of the analysis (with the same number of iterations).
If seed = "random", a random seed will be used, and P-values will vary.  One can also specify an integer for specific seed values,
which might be of interest for advanced users.}

\item{traj.pts}{An optional value specifying the number of points in each trajectory (if f1 contains a single factor)}

\item{data}{A data frame for the function environment, see \code{\link{geomorph.data.frame}}}

\item{print.progress}{A logical value to indicate whether a progress bar should be printed to the screen.
This is helpful for long-running analyses.}

\item{...}{Arguments passed on to procD.fit (typically associated with the \code{\link{lm}} function)}
}
\value{
An object of class "trajectory.analysis" returns a list of the following:
  \item{aov.table}{Procrustes ANOVA table.}
  \item{means}{The observed least squares means based on the linear model.}
  \item{pc.means}{The observed least squares means rotated to their principal components.}
  \item{pc.data}{The observed data rotated to the principal components calculated from the 
  covariance matrix among means. In the case that trajectories are input as data, pc.data is a matrix of
   the trajectories rotated to align with principal axes in the data space.}
  \item{pc.summary}{A table summarizing the percent variation explained by each pc axis, equivalent to summary of \code{\link{prcomp}}.}
  \item{pc.trajectories}{The observed trajectories rotated to the principal components calculated from the 
  covariance matrix among means. In the case that trajectories are input as data, pc.trajectories is a list of the
  the rows of pc.data as separate matrices.}
  \item{random.means}{A list of matrices of means calculated in the random permutations.}
  \item{random.trajectories}{A list of all random means reconfigured as trajectories. The observed
  case is the first random set.}
  \item{path.distances}{The path distances of each observed trajectory.}
  \item{magnitude.diff}{A matrix of the absolute differences in pairwise path distances.}
  \item{trajectory.cor}{A matrix of pairwise vector correlations among trajectory PCs}
  \item{trajectory.angle.rad}{The vector correlations transformed into angles, in radians.}
  \item{trajectory.angle.deg}{The vector correlations transformed into angles, in degrees.}
  \item{trajectory.shape.dist}{A matrix of pairwise Procrustes distances among trajectory shapes.}
  \item{P.magnitude.diff}{P-values corresponding to trajectory magnitude differences.}
  \item{P.angle.diff}{P-values corresponding to angular differences in trajectories.}
  \item{P.shape.diff}{P-values corresponding to trajectory shape differences.}
  \item{Z.magnitude.diff}{Effect sizes of observed trajectory magnitude differences.}
  \item{Z.angle.diff}{Effect sizes of observed angular differences in trajectories.}
  \item{Z.shape.diff}{Effect sizes of observed trajectory shape differences.}
  \item{call}{The matched call.}
  \item{groups}{Factor representing group names for subsequent plotting.}
  \item{permutations}{The number of random permutations used in the RRPP applied to the ANOVA 
  and trajectory statistics.}
  \item{trajectory.type}{A value of 1 if trajectories were provided or 2 if they were estimated.}
}
\description{
Function estimates attributes of shape change trajectories or "motion" trajectories for a set of 
 Procrustes-aligned specimens and compares them statistically
}
\details{
The function quantifies phenotypic shape change trajectories from a set of specimens, and assesses variation 
 in attributes of the trajectories via permutation. A shape change trajectory is defined by a sequence 
 of shapes in tangent space. These trajectories can be quantified for various attributes (their size, orientation, 
 and shape), and comparisons of these attribute enable the statistical comparison of shape change 
 trajectories (see Collyer and Adams 2013; Collyer and Adams 2007; Adams and Collyer 2007; Adams and Collyer 2009).

Data input is specified by a two formulae (e.g., Y ~ X), where 'Y' specifies the response variables (trajectory data), 
 and 'X' contains one or more independent variables (discrete or continuous). The response matrix 'Y' can be either in the form of a two-dimensional data 
  matrix of dimension (n x [p x k]), or a 3D array (p x n x k).. The function
 \code{\link{two.d.array}} can be used to obtain a two-dimensional data matrix from a 3D array of landmark
 coordinates. It is assumed that the order of the specimens 'Y' matches the order of specimens in 'X'. 
 It is  also assumed that the landmarks have previously been aligned using Generalized Procrustes 
 Analysis (GPA) [e.g., with \code{\link{gpagen}}]. Linear model 
 fits (using the  \code{\link{lm}} function) can also be input in place of a formula.  Arguments for  \code{\link{lm}} 
 can also be passed on via this function.  The first formula, f1, must contain the independent variable on the left-hand
 side of the formula (e.g., Y ~) and either a single factor or a two factor interaction on the right-hand side.  If
 a single factor is provided, e.g., Y ~ A, it is assumed that groups to be described are the levels of factor A
 and that the data in Y comprise trajectories.  In this case, the traj.pts = NULL argument must be changed to a numeric value 
 to define the number of points in the trajectory.  It is also assumed that the data are structured as variables within points.  
 For example, y11 y21 y31 y12 y22 y32 y13 y23 y33 y14 y24 y34 would be columns of a matrix, Y, describing a 4-point trajectory
 in a data space defined by three variables.  This is the proper arrangement; the following is an improper arrangement: 
 y11 y12 y13 y14 y21 y22 y23 y24 y31 y32 y33 y34, as it groups points within variables.  This approach is typical when comparing
 motion paths (see Adams and Cerney 2007).
 
 If f1 is a two-factor factorial model, e.g., Y ~ A*B, it is assumed that the first factor defines groups, the second factor
 defines trajectory points, and that trajectories are to be estimated from the linear model.  In this case, the preceding example
 would have a Y matrix comprised only of y1, y2, and y3, but the factor B would contain levels to define the four points (see Examples).
 
 If one wishes to include other variables in the linear model, they should be indicated in the second formula, f2.
 This formula can be simply a right-hand formula, e.g., ~ x1 + x2 + x3 +...  Variables in this formula will typically
 be covariates that one wishes to include to account for extraneous sources of shape variation.  An analysis
 of variance (ANOVA) will be performed with type I sums of squares (SS) and a randomized residual permutation procedure (RRPP).
 The variables in f2 will be added prior to the trajectory defining variables in f1.

Once the function has performed the analysis, a plot can be generated of the trajectories as visualized in the 
 space of principal components (PC1 vs. PC2). The first point in each trajectory is displayed as white, the 
 last point is black, and any middle points on the trajectories are in gray.  The colors of trajectories follow
 the order in which they are found in the dataset as a default, using R's standard color palette: black, red, green,
 blue, cyan, magenta, yellow, and gray. However, one can override these colors with the argument, "group.cols" in plots 
 using the function, \code{\link{plot}}.  This will change the trajectory line colors.  
 One can also override default initial-middle-end point colors with the argument, "pt.seq.pattern".  The default is
 c("white", "gray", "black") for gray points, but with white initial points and black end points.  If changed, the 
 pt.seq.pattern argument must be a vector with three color values.
 One can also uniformly vary the size of points with the argument, "pt.scale".  Examples are provided below.
 
 The function, \code{\link{summary}} can be used to provide an ANOVA summary plus pairwise statistics of a
 an object of class "trajectory.analysis".  The argument, angle.type = c("r", "rad", "deg") can be used to
 toggle between vector correlations, vector angles in radians, or vector angles in degrees, respectively.
 
\subsection{Notes for geomorph 3.0}{ 
Previous versions of geomorph had two separate analytical approaches based on whether trajectories were estimated or
provided (as might be the case with motion trajectories; see Adams and Cerney 2007).  Starting with geomorph 3.0, 
commensurate analytical approaches are used.  This involves converting 1 x vp vectors for trajectpries, were p is the 
number of trajectory points and v is the number of variables in the data space, into p x v matrices, analogous to the
procedure for estimating trajectories.  Thus, rather than providing separate ANOVAs for size, orientation, and shape
of trajectories, a general ANOVA is provided with pairwise statistics for the same attribute differences.  This change
does not compromise any interpretations made with previous versions of geomorph, but enhances inferential capacity
by providing pairwise statistics and P-values.
}
}
\examples{
# Estimate trajectories from LS means in 2-factor model

data(plethodon) 
Y.gpa <- gpagen(plethodon$land)   
gdf <- geomorph.data.frame(Y.gpa, species = plethodon$species, site = plethodon$site)

TA <- trajectory.analysis(coords ~ species*site, data=gdf, iter=199)
summary(TA, angle.type = "deg")
plot(TA)

# Change order of groups

site <- as.factor(plethodon$site)
levels(site) <- c("Symp", "Allo")
gdf <- geomorph.data.frame(Y.gpa, species = plethodon$species, site = site)

TA <- trajectory.analysis(coords ~ species*site, data=gdf, iter=199)
summary(TA, angle.type = "deg")
plot(TA)

attributes(TA) # list of extractable parts

# Add Centroid size as a covariate

TA <- trajectory.analysis(f1 = coords ~ species*site, f2 = ~ Csize, data=gdf, iter=199)
summary(TA, angle.type = "deg")
plot(TA)

# Change trajectory colors in plot
plot(TA, group.cols = c("dark red", "dark blue"))

# Change size of points and lines
plot(TA, group.cols = c("dark red", "dark blue"), pt.scale=1.5)

# Motion paths represented by 5 time points per motion 

data(motionpaths)

gdf <- geomorph.data.frame(trajectories = motionpaths$trajectories,
groups = motionpaths$groups)
TA <- trajectory.analysis(f1 = trajectories ~ groups, 
traj.pts = 5, data=gdf, iter=199)
summary(TA)
plot(TA)
plot(TA, group.cols = c("dark red", "dark blue", "dark green", "yellow"), pt.scale = 1.3)
plot(TA, group.cols = c("dark red", "dark blue", "dark green", "yellow"), 
pt.seq.pattern = c("green", "gray30", "red"), pt.scale = 1.3)
}
\author{
Dean Adams and Michael Collyer
}
\references{
Collyer, M.L., and D.C. Adams. 2013. Phenotypic trajectory analysis: Comparison of 
 shape change patterns in evolution and ecology. Hystrix. 24:75-83.

Adams, D. C. 2010. Parallel evolution of character displacement driven by competitive 
  selection in terrestrial salamanders. BMC Evol. Biol. 10:1-10.

Adams, D. C., and M. M. Cerney. 2007. Quantifying biomechanical motion using Procrustes 
  motion analysis. J. Biomech. 40:437-444.

Adams, D. C., and M. L. Collyer. 2007. The analysis of character divergence along environmental 
  gradients and other covariates. Evolution 61:510-515.

Adams, D. C., and M. L. Collyer. 2009. A general framework for the analysis of phenotypic 
  trajectories in evolutionary studies. Evolution 63:1143-1154.

Collyer, M. L., and D. C. Adams. 2007. Analysis of two-state multivariate phenotypic change 
  in ecological studies. Ecology 88:683-692.
}
\keyword{analysis}

