% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/trajectory.analysis.r
\name{trajectory.analysis}
\alias{trajectory.analysis}
\title{Quantify and compare shape change trajectories}
\usage{
trajectory.analysis(f1, estimate.traj = TRUE, traj.pts = NULL, iter = 999,
  pca = TRUE, verbose = FALSE, group.cols = NULL, ...)
}
\arguments{
\item{f1}{A formula for the linear model (e.g., y~x1+x2)}

\item{estimate.traj}{A logical value indicating whether trajectories are estimated from original data;
described below}

\item{traj.pts}{An optional value specifying the number of points in each trajectory (if estimate.traj=FALSE)}

\item{iter}{Number of iterations for significance testing}

\item{pca}{A logical indicator if a principal component analysis should be performed on data}

\item{verbose}{A logical indicator for verbose (random) output (observed cases always first)}

\item{group.cols}{A vector of colors to use for trajectories, in the order of the levels of the grouping variable. E.g., c("red", "blue, "orange",...)}

\item{...}{Arguments passed on to procD.fit (typically associated with the lm function)}
}
\value{
If "estimate.traj=TRUE", the function returns a list with the following components:
  \item{anova.table}{Procrustes ANOVA table}
  \item{Size$Obs.dif}{A matrix of pairwise differences in trajectory size}
  \item{Size$Z}{A matrix of pairwise effect sizes for differences in trajectory size}
  \item{Size$P}{A matrix of pairwise significance levels for differences in trajectory size}
  \item{Direction$Obs.dif}{A matrix of pairwise differences in trajectory orientation}
  \item{Direction$Z}{A matrix of effect sizes for differences in trajectory orientation}
  \item{Direction$P}{A matrix of pairwise significance levels for differences in trajectory orientation}
  \item{Shape$Obs.dif}{A matrix of pairwise differences in trajectory shape (if applicable)}
  \item{Shape$Z}{A matrix of pairwise effect sizes for differences in trajectory shape (if applicable)}
  \item{Random.values}{All random values for all RRPP permutations (when {verbose=TRUE})}
  \item{plot.points}{Points for plotting (either observed or PC points)}
  \item{trajectory.points}{Trajectory points for plotting (either observed or PC points; typcially means of plot points)}

If "estimate.traj=FALSE", the function returns a list with the following components:
  \item{MANOVA.location.covariation}{Procrustes ANOVA table}
  \item{ANOVA.Size}{Results of permutational-ANOVA assessing variation in trajectory size}
  \item{ANOVA.Dir}{Results of permutational-ANOVA assessing variation in trajectory orientation}
  \item{ANOVA.Shape}{Results of permutational-ANOVA assessing variation in trajectory shape (if applicable)}
  \item{random.SS.location}{Random SS from RRPP permutations (when {verbose=TRUE})}
  \item{random.SS.size}{Random SS from RRPP permutations (when {verbose=TRUE})}
  \item{random.SS.dir}{Random SS from RRPP permutations (when {verbose=TRUE})}
  \item{random.SS.shape}{Random SS from RRPP permutations (when {verbose=TRUE})}
}
\description{
Function estimates attributes of shape change trajectories or motion trajectories for a set of
 Procrustes-aligned specimens and compares them statistically
}
\details{
The function quantifies phenotypic shape change trajectories from a set of specimens, and assesses variation
 in these parameters via permutation. A shape change trajectory is defined by a sequence
 of shapes in tangent space. These trajectories can be quantified for various attributes (their size, orientation,
 and shape), and comparisons of these attribute enable the statistical comparison of shape change
 trajectories (see Collyer and Adams 2013; Collyer and Adams 2007; Adams and Collyer 2007; Adams and Collyer 2009).

Data input is specified by a formula (e.g., Y~X), where 'Y' specifies the response variables (trajectory data),
 and 'X' contains one or more independent variables (discrete or continuous). The response matrix 'Y' can be either in the form of a two-dimensional data
  matrix of dimension (n x [p x k]), or a 3D array (p x n x k).. The function
 \code{\link{two.d.array}} can be used to obtain a two-dimensional data matrix from a 3D array of landmark
 coordinates. It is assumed that the order of the specimens 'Y' matches the order of specimens in 'X'. Linear model
 fits (using the  \code{\link{lm}} function) can also be input in place of a formula.  Arguments for  \code{\link{lm}}
 can also be passed on via this function.

There are two primary modes of analysis through this function. If "estimate.traj=TRUE" the function
 estimates shape trajectories using the least-squares means for groups, based on a two-factor model
 (e.g., Y~A+B+A:B). Under this implementation, the last factor in 'X' must be the interaction term, and
 the preceding two factors must be the effects of interest. Covariates may be included in 'X', and must
 precede the factors of interest (e.g., Y~cov+A*B). In this implementation, 'Y' contains a matrix of landmark
 coordinates. It is assumed that the landmarks have previously been aligned using Generalized Procrustes
 Analysis (GPA) [e.g., with \code{\link{gpagen}}].

If "estimate.traj=FALSE" the trajectories are assembled directly from the set of shapes provided in 'Y'.
 With this implementation, the user must specify the number of shapes that comprise each trajectory. This
 approach is useful when the set of shapes forming each trajectory have been quantified directly
 (e.g., when motion paths are compared: see Adams and Cerney 2007). With this implementation, variation in
 trajectory size, shape, and orientation are evaluated for each term in 'X'.(see Adams and Cerney 2007).
 Currently only single-factor analyses are supportd with this method.

Once the function has performed the analysis, it generates a plot of the trajectories as visualized in the
 space of principal components (PC1 vs. PC2). The first point in each trajectory is displayed as white, the
last point is black, and any middle points on the trajectories are in gray.  The colors of trajectories follow
 the order in which they are found in the dataset as a default, using R's standard color palette: black, red, green3,
 blue, cyan, magenta, yellow, and gray. However, one can override these colors with group.cols.
}
\examples{
#1: Estimate trajectories from LS means in 2-factor model
data(plethodon)
Y.gpa<-two.d.array(gpagen(plethodon$land)$coords)

trajectory.analysis(Y.gpa~plethodon$species*plethodon$site,iter=15)

# Retaining random values (first sets are always observed)
tra <- trajectory.analysis(Y.gpa~plethodon$species*plethodon$site,iter=15, verbose = TRUE)
tra$anova.table
tra$Random.values

#2: Compare motion trajectories
data(motionpaths)

#Motion paths represented by 5 time points per motion

trajectory.analysis(motionpaths$trajectories~motionpaths$groups,
estimate.traj=FALSE, traj.pts=5,iter=15)

trajectory.analysis(motionpaths$trajectories~motionpaths$groups,
estimate.traj=FALSE, traj.pts=5,iter=15, verbose=TRUE)
}
\author{
Dean Adams and Michael Collyer
}
\references{
Collyer, M.L., and D.C. Adams. 2013. Phenotypic trajectory analysis: Comparison of
 shape change patterns in evolution and ecology. Hystrix. 24:75-83.

Adams, D. C. 2010. Parallel evolution of character displacement driven by competitive
  selection in terrestrial salamanders. BMC Evol. Biol. 10:1-10.

Adams, D. C., and M. M. Cerney. 2007. Quantifying biomechanical motion using Procrustes
  motion analysis. J. Biomech. 40:437-444.

Adams, D. C., and M. L. Collyer. 2007. The analysis of character divergence along environmental
  gradients and other covariates. Evolution 61:510-515.

Adams, D. C., and M. L. Collyer. 2009. A general framework for the analysis of phenotypic
  trajectories in evolutionary studies. Evolution 63:1143-1154.

Collyer, M. L., and D. C. Adams. 2007. Analysis of two-state multivariate phenotypic change
  in ecological studies. Ecology 88:683-692.
}
\keyword{analysis}

