#' Format character columns as markdown text
#'
#' @param x A `flextable` object
#' @param j Columns to be treated as markdown texts.
#'   Selection can be done by the semantics of `dplyr::select()`.
#' @param part
#'   One of "body", "header", and "all". If "all", formatting proceeds in the
#'   order of "header" and "body".
#' @param .footnote_options
#'   Options for footnotes generated by `footnote_options()`.
#' @inheritParams as_paragraph_md
#'
#' @examples
#' if (rmarkdown::pandoc_available()) {
#'   d <- data.frame(
#'     x = c("**bold**", "*italic*"),
#'     y = c("^superscript^", "~subscript~"),
#'     z = c("***^ft^~Extra~** is*", "*Cool*")
#'   )
#'   colformat_md(flextable::flextable(d))
#' }
#' @export
colformat_md <- function(x,
                         j = where(is.character),
                         part = c("body", "header", "all"),
                         auto_color_link = "blue",
                         md_extensions = NULL,
                         pandoc_args = NULL,
                         .from = "markdown+autolink_bare_uris",
                         .footnote_options = footnote_options()
) {
  .j <- rlang::enexpr(j)
  part <- match.arg(part)
  .footnote_options$caller <- "colformat_md"

  if (part == "all") {
    for (part in c("header", "body")) {
      x <- colformat_md(x, j = !!.j, part = part,
                        auto_color_link = auto_color_link,
                        .footnote_options = .footnote_options,
                        .from = .from)
      .footnote_options$value <- list()
    }
    return(x)
  }

  dataset <- x[[part]]$dataset
  col <- names(tidyselect::eval_select(rlang::expr(c(!!.j)), dataset))

  if (length(col) == 0) {
    return(x)
  }

  # Must evaluate outside add_footnotes due to lazy evaluation of arguments
  ft <- flextable::compose(x,
                           i = seq(nrow(dataset)), j = col, part = part,
                           value = as_paragraph_md(
                             unlist(dataset[col], use.names = FALSE),
                             auto_color_link = auto_color_link,
                             .from = .from,
                             md_extensions = md_extensions,
                             pandoc_args = pandoc_args,
                             .footnote_options = .footnote_options
                           ))

  add_footnotes(ft, part, .footnote_options)
}

where <- function(...) {
  tidyselect::vars_select_helpers$where(...)
}
