% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spa_functions.R
\name{fsr_numerical_operations}
\alias{fsr_numerical_operations}
\alias{spa_avg_degree}
\alias{spa_ncomp}
\alias{spa_area}
\alias{spa_perimeter}
\alias{spa_length}
\title{Fuzzy numerical operations}
\usage{
spa_avg_degree(pgo)

spa_ncomp(pgo)

spa_area(pr)

spa_perimeter(pr)

spa_length(pl)
}
\arguments{
\item{pgo}{A \code{pgeometry} object of any type.}

\item{pr}{A \code{pgeometry} object of the type \code{PLATEAUREGION}. It throws an error if a different type is given.}

\item{pl}{A \code{pgeometry} object of the type \code{PLATEAULINE}. It throws an error if a different type is given.}
}
\value{
A numerical value.
}
\description{
Fuzzy numerical operations are given as a family of functions that implements spatial plateau metric operations.
These functions extract metric properties from spatial plateau objects,
such as the area of a plateau region object and the length of a plateau line object.
}
\details{
These functions calculate numerical properties from spatial plateau objects (i.e., \code{pgeometry} objects).
Some of them are \emph{type-independent}. This means that the parameter can be a \code{pgeometry} object of any type.
The type-independent functions are:
\itemize{
\item \code{spa_avg_degree} calculates the average membership degree of a spatial plateau object.
\item \code{spa_ncomp} returns the number of components of a spatial plateau object.
}

The remaining functions are \emph{type-dependent}. This means that the parameter have to be of a specific type.
The type-dependent functions are:
\itemize{
\item \code{spa_area} computes the area of a plateau region object. Thus, its parameter has to be a \code{PLATEAUREGION} object.
\item \code{spa_perimeter} computes the perimeter of a plateau region object. Thus, its parameter has to be a \code{PLATEAUREGION} object.
\item \code{spa_length} computes the length of a plateau line object. Thus, its parameter has to be a \code{PLATEAULINE} object.
}
}
\examples{

library(sf)
library(tibble)

pts1 <- rbind(c(1, 2), c(3, 2))
comp1 <- component_from_sfg(st_multipoint(pts1), 0.2) 
comp2 <- component_from_sfg(st_point(c(1, 5)), 0.8)  

pp <- create_pgeometry(list(comp1, comp2), "PLATEAUPOINT")

# calculating the average degree and number of components of pp

spa_avg_degree(pp)
spa_ncomp(pp)

# calculating the area and perimeter

set.seed(345)

# some random points to create plateau region objects by using the function spa_creator
tbl = tibble(x = runif(10, min= 0, max = 20), 
             y = runif(10, min = 0, max = 30), 
             z = runif(10, min = 0, max = 100))

#getting the convex hull on the points to clip the construction of plateau region objects
pts <- st_as_sf(tbl, coords = c(1, 2))
ch <- st_convex_hull(do.call(c, st_geometry(pts)))

pregions <- spa_creator(tbl, fuzz_policy = "fcp", k = 2, base_poly = ch)

spa_area(pregions$pgeometry[[1]])
spa_area(pregions$pgeometry[[2]])

spa_perimeter(pregions$pgeometry[[1]])
spa_perimeter(pregions$pgeometry[[2]])

# calculating the length of a plateau line object

lpts1 <- rbind(c(0, 0), c(1, 1))
lpts2 <- rbind(c(1, 1), c(1.2, 1.9), c(2, 1))
lpts3 <- rbind(c(2, 1), c(1.5, 0.5))

cp1 <- component_from_sfg(st_linestring(lpts1), 0.4)
cp2 <- component_from_sfg(st_linestring(lpts2), 1)
cp3 <- component_from_sfg(st_linestring(lpts3), 0.7)

pline <- create_pgeometry(list(cp1, cp2, cp3), "PLATEAULINE")

spa_length(pline)

}
\references{
\href{https://ieeexplore.ieee.org/document/8491565}{Carniel, A. C.; Schneider, M. Spatial Plateau Algebra: An Executable Type System for Fuzzy Spatial Data Types. In Proceedings of the 2018 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2018), pp. 1-8, 2018.}
}
