% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spa_functions.R
\name{fsr_geometric_operations}
\alias{fsr_geometric_operations}
\alias{spa_intersection}
\alias{spa_union}
\alias{spa_difference}
\alias{spa_common_points}
\title{Fuzzy geometric set operations}
\usage{
spa_intersection(pgo1, pgo2, itype = "min")

spa_union(pgo1, pgo2, utype = "max")

spa_difference(pgo1, pgo2, dtype = "f_diff")

spa_common_points(pline1, pline2, itype = "min")
}
\arguments{
\item{pgo1}{A \code{pgeometry} object of any type.}

\item{pgo2}{A \code{pgeometry} object of the same type of \code{pgo1}.}

\item{itype}{A character value that indicates the name of a function implementing a t-norm. The default value is \code{"min"}, which is the standard operator of the intersection.}

\item{utype}{A character value that refers to a t-conorm. The default value is \code{"max"}, which is the standard operator of the union.}

\item{dtype}{A character value that indicates the name of a difference operator. The default value is \code{"f_diff"}, which implements the standard fuzzy difference.}

\item{pline1}{A \code{pgeometry} object of the type \code{PLATEAULINE}.}

\item{pline2}{A \code{pgeometry} object of the type \code{PLATEAULINE}.}
}
\value{
A \code{pgeometry} object that is the result of the geometric manipulation between two spatial plateau objects.
}
\description{
Fuzzy geometric set operations are given as a family of functions that implements spatial plateau set operations.
These functions produces a spatial plateau object from a specific combination of other two spatial plateau objects,
such as the intersection of two plateau region objects.
}
\details{
These functions implement geometric operations of the spatial plateau algebra.
They receive two \code{pgeometry} objects of the \emph{same type} together with an operation as inputs and yield another \code{pgeometry} object as output. The output object has \emph{the same} type of the inputs.
The family of fuzzy geometric set operations consists of the following functions:
\itemize{
\item \code{spa_intersection} computes the geometric intersection of two spatial plateau objects.
The membership degree of common points are calculated by using a t-norm operator given by the parameter \code{itype}. Currently, it can assume \code{"min"} (default) or \code{"prod"}.
\item \code{spa_union} computes the geometric union of two spatial plateau objects.
The membership degree of common points are calculated by using a t-conorm operator given by the parameter \code{utype}. Currently, it can assume \code{"max"} (default).
\item \code{spa_difference} computes the geometric difference of two spatial plateau objects.
The membership degree of common points are calculated by using a diff operator given by the parameter \code{dtype}.
Currently, it can assume \code{"f_diff"} (default fuzzy difference), \code{"f_bound_diff"} (fuzzy bounded difference), \code{"f_symm_diff"} (fuzzy symmetric difference), and \code{"f_abs_diff"} (fuzzy absolute difference).
}

Another related geometric function is:
\itemize{
\item \code{spa_common_points} which gets the common points of two plateau line objects by using a t-norm to compute their membership degrees.
It is different from the other functions since it gets two plateau line objects as input and yields a plateau point object as output.
}

Other t-norms, t-conorms, and diff operators can be implemented and given as values for the \code{"itype"}, \code{"utype"}, and \code{"dtype"}, respectively.
For this, the following steps should be performed:

1 - implement your function that accepts two numeric values as inputs and yields another numeric value as output. All values should be between 0 and 1. Recall that t-norms and t-conorms must have some specific properties according to the fuzzy set theory.
2 - use the name of your function as the character value of the corresponding \code{"itype"}, \code{"utype"}, or \code{"dtype"}.

An example of operator is the source code of \code{f_bound_diff}:

\code{f_bound_diff <- function(x, y) { max(0, (x - y)) }}
}
\examples{

library(sf)

pts1 <- rbind(c(1, 2), c(3, 2))
pts2 <- rbind(c(1, 1), c(2, 3), c(2, 1))
pts3 <- rbind(c(2, 2), c(3, 3))

cp1 <- component_from_sfg(st_multipoint(pts1), 0.3)
cp2 <- component_from_sfg(st_multipoint(pts2), 0.6)
cp3 <- component_from_sfg(st_multipoint(pts3), 1)

pp1 <- create_pgeometry(list(cp1, cp2, cp3), "PLATEAUPOINT")

pts4 <- rbind(c(0, 0), c(1, 1))
pts5 <- rbind(c(2, 3), c(1.2, 1.9), c(2, 1))
pts6 <- rbind(c(3, 1), c(1.5, 0.5))

cp4 <- component_from_sfg(st_multipoint(pts4), 0.4)
cp5 <- component_from_sfg(st_multipoint(pts5), 1)
cp6 <- component_from_sfg(st_multipoint(pts6), 0.7)

pp2 <- create_pgeometry(list(cp4, cp5, cp6), "PLATEAUPOINT")

pp1
pp2

spa_intersection(pp1, pp2)
spa_intersection(pp1, pp2, itype = "prod") #changing the t-norm
spa_union(pp1, pp2)
spa_difference(pp1, pp2)

}
\references{
\href{https://ieeexplore.ieee.org/document/8491565}{Carniel, A. C.; Schneider, M. Spatial Plateau Algebra: An Executable Type System for Fuzzy Spatial Data Types. In Proceedings of the 2018 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2018), pp. 1-8, 2018.}
}
