% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Cache.R
\name{Cache}
\alias{Cache}
\title{A cache class for handling caching on the file system.}
\description{
A cache class for handling caching on the file system.

A cache class for handling caching on the file system.
}
\details{
The purpose of this class is to help managing a user cache folder for an
application.
Files can be copied or moved into the cache folder.
Character values can be saved into files and loaded from files.
Sub-folders can be defined.
Folders can be listed to see the existing files.
Files can be deleted individually or by batch,
Whole folders can be deleted, including the main cache folder.
}
\examples{
# Create a new cache instance inside a custom folder
cache <- Cache$new(tempdir())

# Create some contents for the example
contents <- c("a", "b", "c")

# Save contents
cache$saveContents(contents, c("a.txt", "b.txt", "c.txt"),
                   sub_folder = "sub1")

# Get list of files inside folder
files <- cache$listFolder("sub1")

# Delete files
cache$delPaths(c("a.txt", "c.txt"), sub_folder = "sub1")

# Delete whole sub-folder
cache$delFolder("sub1")

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$new`
## ------------------------------------------------

# Create a new cache instance.
# Note for the sake of the example we use a temporary directory specified
# as an absolute path, however the usual way to use the cache system is
# to provide a relative path, that will be placed inside the standard
# user cache folder defined by the OS.
cache <- Cache$new(tempdir())

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$isReadable`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Tests if readable (TRUE by default)
if (cache$isReadable()) {
  print("Cache is readable")
}

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$isWritable`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Tests if writeable (TRUE by default)
if (cache$isWritable()) {
  print("Cache is writeable")
}

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$setReadable`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Disallow reading
cache$setReadable(FALSE)

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$setWritable`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Disallow writing
cache$setWritable(FALSE)

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$getFolder`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the absolute path to the cache folder
folder <- cache$getFolder()

# Get the absolute path to a cache sub-folder
sub_folder <- cache$getFolder('my_sub_folder')

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$hasFolder`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Test if a sub-folder exists
if (cache$hasFolder("my_sub_folder")) {
  print("Sub-folder exists.")
}

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$getPaths`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the paths a list of filenames should have in the cache folder
paths <- cache$getPaths(c("a.csv", "b.txt"))

# Get paths using a common extension for filenames
paths <- cache$getPaths(c("a", "b"), suffix = ".csv")

# Get paths of files inside a sub-folder
paths <- cache$getPaths(c("a.csv", "b.txt"), sub_folder = "foo")

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$globPaths`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Get all existing files inside sub-folder foo
paths <- cache$globPaths(sub_folder = "foo")

# Get all existing files with extension ".txt" inside main folder
paths <- cache$globPaths(suffix = ".txt")

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$getNbItems`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the number of files inside sub-folder "foo"
n <- cache$getNbItems("foo")

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$pathsExist`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Test if some files exist in the cache
exits <- cache$pathsExist(c("a", "b"), suffix = ".txt")

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$tagExists`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Test if tag file "downloaded" exists in sub-folder "hmdb"
if (cache$tagExists("downloaded", sub_folder = "hmdb")) {
  print("Tag exists")
}

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$writeTag`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Create tag file "downloaded" in sub-folder "hmdb"
cache$writeTag("downloaded", sub_folder = "hmdb")

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$getTmp`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the cache temporary folder
tmp <- cache$getTmp()

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$getSubFolders`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the list of sub-folders
sub.folders <- cache$getSubFolders()

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$importFiles`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Create some files for the example
files <- c("k.txt", "u.csv")
file.create(files)

# Move those files into the cache
cache$importFiles(files, sub_folder = "foo", action = "copy")

# Remove original files
unlink(files)

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$saveContents`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Create some contents for the example
contents <- c("a", "b", "c")

# Save contents
cache$saveContents(contents, c("a.txt", "b.txt", "c.txt"))

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$loadContents`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Create some contents for the example
contents <- c("1", "2", "3")

# Save contents
cache$saveContents(contents, c("a", "b", "c"), suffix = ".txt",
                   sub_folder = "ex2")

# Load contents
contents <- cache$loadContents(c("a", "b", "c"), suffix = ".txt",
                               sub_folder = "ex2")

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$delPaths`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Delete some cache files
cache$delPaths(c("a.txt", "b.txt"))

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$delFolder`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Delete sub-folder
cache$delFolder("my_sub_folder")

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$listFolder`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new("my_cache_folder")

# List files in sub-folder
files <- cache$listFolder("my_sub_folder")

# Remove cache folder
cache$erase()


## ------------------------------------------------
## Method `Cache$print`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Print information
print(cache)

# Erase cache
cache$erase()


## ------------------------------------------------
## Method `Cache$erase`
## ------------------------------------------------

# Create a new cache instance
cache <- Cache$new(tempdir())

# Deletes the whole cache content
cache$erase()

# Erase cache
cache$erase()

}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Cache-new}{\code{Cache$new()}}
\item \href{#method-Cache-isReadable}{\code{Cache$isReadable()}}
\item \href{#method-Cache-isWritable}{\code{Cache$isWritable()}}
\item \href{#method-Cache-setReadable}{\code{Cache$setReadable()}}
\item \href{#method-Cache-setWritable}{\code{Cache$setWritable()}}
\item \href{#method-Cache-getFolder}{\code{Cache$getFolder()}}
\item \href{#method-Cache-hasFolder}{\code{Cache$hasFolder()}}
\item \href{#method-Cache-getPaths}{\code{Cache$getPaths()}}
\item \href{#method-Cache-globPaths}{\code{Cache$globPaths()}}
\item \href{#method-Cache-getNbItems}{\code{Cache$getNbItems()}}
\item \href{#method-Cache-pathsExist}{\code{Cache$pathsExist()}}
\item \href{#method-Cache-tagExists}{\code{Cache$tagExists()}}
\item \href{#method-Cache-writeTag}{\code{Cache$writeTag()}}
\item \href{#method-Cache-getTmp}{\code{Cache$getTmp()}}
\item \href{#method-Cache-getSubFolders}{\code{Cache$getSubFolders()}}
\item \href{#method-Cache-importFiles}{\code{Cache$importFiles()}}
\item \href{#method-Cache-saveContents}{\code{Cache$saveContents()}}
\item \href{#method-Cache-loadContents}{\code{Cache$loadContents()}}
\item \href{#method-Cache-delPaths}{\code{Cache$delPaths()}}
\item \href{#method-Cache-delFolder}{\code{Cache$delFolder()}}
\item \href{#method-Cache-listFolder}{\code{Cache$listFolder()}}
\item \href{#method-Cache-print}{\code{Cache$print()}}
\item \href{#method-Cache-erase}{\code{Cache$erase()}}
\item \href{#method-Cache-clone}{\code{Cache$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-new"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-new}{}}}
\subsection{Method \code{new()}}{
New instance initializer.

Initializes a \code{Cache} instance, using a specified folder.
Path to the folder can be absolute or relative.
When relative, the absolute root folder is either the standard user cache
folder or the current working directory, depending on \code{user}
parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$new(folder, user = TRUE, force = FALSE, create = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{folder}}{The path to the wanted cache folder. Either an absolute
path, or a relative path that will be resolved immediately
into an absolute path.}

\item{\code{user}}{If set to TRUE and the folder path is a relative path, then
the path is resolved relatively to the standard user cache
folder (i.e.: we call \code{tools::R_user_dir(folder,
            which = 'cache')}). A good and standard practice is to set
the \code{folder} parameter to your package name,
using.}

\item{\code{force}}{If the folder exists, is not empty and is not an fscache
folder, fails if force is \code{FALSE}, and use folder
anyway if force is \code{TRUE}.}

\item{\code{create}}{If \code{FALSE}, does not create the cache folder if does
not exist already. Used for testing purposes only.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance.
# Note for the sake of the example we use a temporary directory specified
# as an absolute path, however the usual way to use the cache system is
# to provide a relative path, that will be placed inside the standard
# user cache folder defined by the OS.
cache <- Cache$new(tempdir())

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-isReadable"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-isReadable}{}}}
\subsection{Method \code{isReadable()}}{
Tests if the cache system is readable.

Cache reading may be disabled and re-enabled with \code{setReadable()},
Mainly used for debug purposes.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$isReadable()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
\code{TRUE} if the cache system is readable, \code{FALSE}
otherwise.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Tests if readable (TRUE by default)
if (cache$isReadable()) {
  print("Cache is readable")
}

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-isWritable"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-isWritable}{}}}
\subsection{Method \code{isWritable()}}{
Tests if the cache system is writable.

Cache reading may be disabled and re-enabled with \code{setWritable()}.
Mainly used for debug purposes.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$isWritable()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
\code{TRUE} if the cache system is writable, \code{FALSE}
otherwise.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Tests if writeable (TRUE by default)
if (cache$isWritable()) {
  print("Cache is writeable")
}

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-setReadable"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-setReadable}{}}}
\subsection{Method \code{setReadable()}}{
Disables or enable cache reading.

Allows or disallows reading to the cache folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$setReadable(readable)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{readable}}{Set to \code{FALSE} to disallow reading and to
\code{TRUE} to allow it.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Disallow reading
cache$setReadable(FALSE)

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-setWritable"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-setWritable}{}}}
\subsection{Method \code{setWritable()}}{
Disables or enable cache writing.

Allows or disallows writing to the cache folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$setWritable(writable)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{writable}}{Set to \code{FALSE} to disallow writing and to
\code{TRUE} to allow it.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Disallow writing
cache$setWritable(FALSE)

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-getFolder"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-getFolder}{}}}
\subsection{Method \code{getFolder()}}{
Gets the path to the main cache folder or a sub-folder.

Returns the absolute path to the main cache folder or a cache sub-folder.
By default, the folder is created if it does not exist.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$getFolder(
  sub_folder = NULL,
  create = TRUE,
  fail = FALSE,
  sub.folder = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{sub_folder}}{A sub-folder.}

\item{\code{create}}{If set to \code{TRUE} and the folder does not exist,
creates it.}

\item{\code{fail}}{If set to \code{TRUE}, throws an error if the folder does not
exist.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The path to the cache folder as a character value.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the absolute path to the cache folder
folder <- cache$getFolder()

# Get the absolute path to a cache sub-folder
sub_folder <- cache$getFolder('my_sub_folder')

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-hasFolder"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-hasFolder}{}}}
\subsection{Method \code{hasFolder()}}{
Tests if the cache main folder or a cache sub-folder exists.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$hasFolder(sub_folder = NULL, sub.folder = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{sub_folder}}{The sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
TRUE if the folder exists. FALSE otherwise.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Test if a sub-folder exists
if (cache$hasFolder("my_sub_folder")) {
  print("Sub-folder exists.")
}

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-getPaths"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-getPaths}{}}}
\subsection{Method \code{getPaths()}}{
Computes paths in the cache folder or a cache sub-folder.

Takes a list of relative paths and resolves them using the cache folder
path to a list of absolute paths.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$getPaths(paths, suffix = NULL, sub_folder = NULL, sub.folder = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{paths}}{A character vector containing paths.}

\item{\code{suffix}}{A suffix to add to all paths.}

\item{\code{sub_folder}}{A sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A character vector, the same size as \code{paths},
containing the absolute paths.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the paths a list of filenames should have in the cache folder
paths <- cache$getPaths(c("a.csv", "b.txt"))

# Get paths using a common extension for filenames
paths <- cache$getPaths(c("a", "b"), suffix = ".csv")

# Get paths of files inside a sub-folder
paths <- cache$getPaths(c("a.csv", "b.txt"), sub_folder = "foo")

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-globPaths"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-globPaths}{}}}
\subsection{Method \code{globPaths()}}{
Search for files inside the cache folder or one of its subfolders.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$globPaths(
  suffix = NULL,
  sub_folder = NULL,
  tag_files = FALSE,
  folders = FALSE,
  tag.files = NULL,
  sub.folder = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{suffix}}{The suffix files must have.}

\item{\code{sub_folder}}{A sub-folder where to search.}

\item{\code{tag_files}}{If set to \code{FALSE} (default), exclude the tag files.
Otherwise include them in the output.}

\item{\code{folders}}{If set to \code{FALSE} (default), exclude the folders.
Otherwise include them in the output.}

\item{\code{tag.files}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use tag_files
instead.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A character vector containing paths to existing file matching the
criteria.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Get all existing files inside sub-folder foo
paths <- cache$globPaths(sub_folder = "foo")

# Get all existing files with extension ".txt" inside main folder
paths <- cache$globPaths(suffix = ".txt")

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-getNbItems"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-getNbItems}{}}}
\subsection{Method \code{getNbItems()}}{
Gets the number of items contained inside a cache folder.

Counts the number of items (files or folders) contained inside a cache
folder.
This method does not explore the file system recursively, but only look
at the files inside the folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$getNbItems(
  sub_folder = NULL,
  tag_files = FALSE,
  folders = FALSE,
  tag.files = NULL,
  sub.folder = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{sub_folder}}{A sub-folder.}

\item{\code{tag_files}}{If set to \code{FALSE} (default), do not count the tag
files. Otherwise count them.}

\item{\code{folders}}{If set to \code{FALSE} (default), do not count the
folders. Otherwise count them.}

\item{\code{tag.files}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use tag_files
instead.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The number of items.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the number of files inside sub-folder "foo"
n <- cache$getNbItems("foo")

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-pathsExist"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-pathsExist}{}}}
\subsection{Method \code{pathsExist()}}{
Tests if paths exist inside a cache folder.

Takes a list of relative paths and resolves them using the cache folder
path to a list of absolute paths. Tests then if each path points to real
object on the file system.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$pathsExist(paths, suffix = NULL, sub_folder = NULL, sub.folder = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{paths}}{A character vector containing paths.}

\item{\code{suffix}}{A suffix to add to all paths.}

\item{\code{sub_folder}}{A sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A logical vector, the same size as \code{paths}, with \code{TRUE}
value if the file exists in the cache, or \code{FALSE} otherwise.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Test if some files exist in the cache
exits <- cache$pathsExist(c("a", "b"), suffix = ".txt")

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-tagExists"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-tagExists}{}}}
\subsection{Method \code{tagExists()}}{
Tests if a tag exists in the cache.

Tags are empty files, without extension, whose name starts and ends with
\code{"__"}.
This method tests if some tag file already exist in a cache folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$tagExists(name, sub_folder = NULL, sub.folder = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{The name of the tag, without the prefix \code{"__"} and the
suffix \code{"__"}. It will be automatically converted in uppercase. It
can only contains digits, letters and underscore characters.}

\item{\code{sub_folder}}{A sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\code{TRUE} if the tag exists in the cache. \code{FALSE}
otherwise.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Test if tag file "downloaded" exists in sub-folder "hmdb"
if (cache$tagExists("downloaded", sub_folder = "hmdb")) {
  print("Tag exists")
}

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-writeTag"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-writeTag}{}}}
\subsection{Method \code{writeTag()}}{
Sets a tag into the cache.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$writeTag(name, sub_folder = NULL, sub.folder = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{The name of the tag, without the prefix \code{"__"} and the
suffix \code{"__"}. It will be automatically converted in uppercase. It
can only contains digits, letters and underscore characters.}

\item{\code{sub_folder}}{A sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Create tag file "downloaded" in sub-folder "hmdb"
cache$writeTag("downloaded", sub_folder = "hmdb")

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-getTmp"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-getTmp}{}}}
\subsection{Method \code{getTmp()}}{
Gets path to the cache system temporary folder.

This temporary folder located inside the cache folder is needed in order
to be able to move/rename files into the right cache location.
When creating files in the system temporary folder, which may reside on a
different partition, moving a file could fail as in the following error:
cannot rename file "/tmp/Rtmpld18y7/10182e3a086e7b8a7.tsv" to
"/home/pr228844/dev/biodb/cache/comp.csv.file-58e...c4/2e3...a7.tsv",
reason "Invalid cross-device link".

When you download a file directly to the disk using for instance
\code{download.file()}, write the destination into this destination
folder.  When downloaded is complete, move the file using the method
\code{importFiles()}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$getTmp()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A string containing the path to the folder.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the cache temporary folder
tmp <- cache$getTmp()

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-getSubFolders"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-getSubFolders}{}}}
\subsection{Method \code{getSubFolders()}}{
Returns all existing sub-folders.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$getSubFolders()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A character vector containing all the sub-folders.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Get the list of sub-folders
sub.folders <- cache$getSubFolders()

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-importFiles"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-importFiles}{}}}
\subsection{Method \code{importFiles()}}{
Imports existing files into the cache.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$importFiles(
  src,
  dst = NULL,
  suffix = NULL,
  sub_folder = NULL,
  action = c("copy", "move"),
  sub.folder = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{src}}{A character vector containing paths to files to import.}

\item{\code{dst}}{A character vector containing destination filenames. The
vector must have the length as the \code{src} vector. If
\code{NULL}, the filenames in \code{src} will be used.}

\item{\code{suffix}}{A suffix to add to all destination paths.}

\item{\code{sub_folder}}{A sub-folder. All files will copied or moved to this
sub-folder.}

\item{\code{action}}{Specifies if files have to be moved or copied into the
cache.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Create some files for the example
files <- c("k.txt", "u.csv")
file.create(files)

# Move those files into the cache
cache$importFiles(files, sub_folder = "foo", action = "copy")

# Remove original files
unlink(files)

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-saveContents"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-saveContents}{}}}
\subsection{Method \code{saveContents()}}{
Saves contents to files into the cache.

Saves character values into files inside a cache folder.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$saveContents(
  contents,
  dst,
  suffix = NULL,
  sub_folder = NULL,
  sub.folder = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{contents}}{A character vector containing the contents to write.}

\item{\code{dst}}{A character vector containing destination filenames. The
vector must have the length as the \code{contents} vector.}

\item{\code{suffix}}{A suffix to add to all destination paths.}

\item{\code{sub_folder}}{A sub-folder. All files will copied or moved to this
sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Create some contents for the example
contents <- c("a", "b", "c")

# Save contents
cache$saveContents(contents, c("a.txt", "b.txt", "c.txt"))

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-loadContents"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-loadContents}{}}}
\subsection{Method \code{loadContents()}}{
Loads contents from files stored into the cache.

Loads character values from cache files.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$loadContents(paths, suffix = NULL, sub_folder = NULL, sub.folder = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{paths}}{A character vector containing destination filenames. The
vector must have the length as the \code{contents} vector.}

\item{\code{suffix}}{A suffix to add to all destination paths.}

\item{\code{sub_folder}}{A sub-folder. All files will copied or moved to this
sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A character vector , the same size as \code{paths}, containing
the contents of the files. If some file does not exist, a \code{NA} value
is returned.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Create some contents for the example
contents <- c("1", "2", "3")

# Save contents
cache$saveContents(contents, c("a", "b", "c"), suffix = ".txt",
                   sub_folder = "ex2")

# Load contents
contents <- cache$loadContents(c("a", "b", "c"), suffix = ".txt",
                               sub_folder = "ex2")

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-delPaths"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-delPaths}{}}}
\subsection{Method \code{delPaths()}}{
Deletes a list of paths inside the cache system.

Takes a list of relative paths, resolves them using the cache folder path
to a list of absolute paths, and deletes the corresponding files.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$delPaths(
  paths = NULL,
  suffix = NULL,
  sub_folder = NULL,
  sub.folder = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{paths}}{A character vector containing paths.}

\item{\code{suffix}}{A suffix to add to all paths.}

\item{\code{sub_folder}}{A sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Delete some cache files
cache$delPaths(c("a.txt", "b.txt"))

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-delFolder"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-delFolder}{}}}
\subsection{Method \code{delFolder()}}{
Deletes all files in a sub-folder.

Deletes a sub-folder and all its content.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$delFolder(sub_folder, sub.folder = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{sub_folder}}{A sub-folder.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Delete sub-folder
cache$delFolder("my_sub_folder")

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-listFolder"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-listFolder}{}}}
\subsection{Method \code{listFolder()}}{
Lists files present inside a cache folder.

Lists files that exist inside a cache folder.
Returns by default the full paths of the found files.
It is possible to filter on files suffix, and to extract the basename.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$listFolder(
  sub_folder = NULL,
  suffix = NULL,
  extract_name = FALSE,
  remove_suffix = FALSE,
  tag_files = FALSE,
  folders = FALSE,
  extract.name = NULL,
  remove.suffix = NULL,
  tag.files = NULL,
  sub.folder = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{sub_folder}}{A sub-folder, or \code{NULL} for the main folder.}

\item{\code{suffix}}{A file suffix on which to filter.}

\item{\code{extract_name}}{If set to \code{TRUE}, instead of returning the full
paths of the files, returns their basenames.}

\item{\code{remove_suffix}}{When set to \code{TRUE} and \code{extract.name} is
\code{TRUE} and \code{suffix} is not \code{NULL},
remove the suffix from the returned basenames.}

\item{\code{tag_files}}{If set to \code{FALSE} (default), exclude the tag files.
Otherwise include them in the output.}

\item{\code{folders}}{If set to \code{FALSE} (default), exclude the folders.
Otherwise include them in the output.}

\item{\code{extract.name}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use extract_name
instead.}

\item{\code{remove.suffix}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use remove_suffix
instead.}

\item{\code{tag.files}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use tag_files
instead.}

\item{\code{sub.folder}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use sub_folder
instead.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The paths to the found files, or the names of the files if
\code{extract.name} is set to \code{TRUE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new("my_cache_folder")

# List files in sub-folder
files <- cache$listFolder("my_sub_folder")

# Remove cache folder
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-print"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-print}{}}}
\subsection{Method \code{print()}}{
Displays information about this object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$print()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Print information
print(cache)

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-erase"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-erase}{}}}
\subsection{Method \code{erase()}}{
Erases the whole cache folder.

Deletes the main cache folder and all its files and sub-folders.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$erase()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
Nothing.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Create a new cache instance
cache <- Cache$new(tempdir())

# Deletes the whole cache content
cache$erase()

# Erase cache
cache$erase()

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Cache-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Cache-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Cache$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
