% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fmsm.R
\name{fmsm}
\alias{fmsm}
\title{Flexible transition intensity based models for univariate multistate processes}
\usage{
fmsm(formula, data, id, state, death, pmethod = 'eigendecomp',
       aggregate = TRUE, params.0 = NULL, sp.0 = NULL,
       constraint = NULL, sp.method = 'perf', iterlimsp = 50,
       Q.diagnostics = TRUE, fit = TRUE, iterlim = 100,
       tolsp = 1e-7, tolsp.EFS = 0.1, parallel = FALSE, no_cores = 2,
       cens.state = NULL, living.exact = NULL, verbose = FALSE)
}
\arguments{
\item{formula}{Model specification for the transition intensities.}

\item{data}{Dataset.}

\item{id}{Name of the variable in the dataset representing the unique code associated with each patient.}

\item{state}{Name of the variable in the dataset representing the state occupied by the patient at the given time.}

\item{death}{\code{TRUE} if the last state is an absorbing state, \code{FALSE} otherwise.}

\item{pmethod}{Which method should be used for the computation of the transition probability matrix. Available options are
\itemize{
\item \code{'eigendecomp'} (default): this method is based on the eigendecomposition of the transition intensity matrix (from Kalbfleisch & Lawless 1985);
\item \code{'analytic'}: uses analytic expressions of the transition probabilities, obtained by solving the Kolmogorov forward differential equation, only implemented for IDMs for now;
\item \code{'scaling&squaring'}: this is the scaling and squaring method implemented as proposed in Fung (2004).This is inefficient, so its use is not recommended. Can be used to investigate convergence errors.
}}

\item{aggregate}{Whether or not data should be aggregated (this slightly improves efficiency as redundancies in the data are eliminated). The default is \code{TRUE}.}

\item{params.0}{Starting values for the model parameters. Defaults to \code{NULL}, i.e. they are computed internally.}

\item{sp.0}{Starting values for the smoothing parameters. Defaults to \code{NULL}, i.e. they are computed internally.}

\item{constraint}{A list containing the constraints to be applied to the model parameters. This follows the form \code{list(x1 = (1,1,1), x2 = c())}}

\item{sp.method}{Method to be used for smoothing parameter estimation. The default is \code{magic}, the automatic multiple smoothing parameter selection algorithm. Alternatively, \code{efs} can be used for the Fellner-Schall method.}

\item{iterlimsp}{Maximum allowed iterations for smoothing parameter estimation.}

\item{Q.diagnostics}{If \code{TRUE}, diagnostics information on the Q matrix are saved. The default \code{TRUE}.}

\item{fit}{If \code{FALSE}, fitting is not carried. May be useful to extract model setup quantities.}

\item{iterlim}{Maximum allowed iterations for trust region algorithm.}

\item{tolsp}{Convergence criterion used in \code{magic} based smoothing parameter estimation.}

\item{tolsp.EFS}{Convergence criterion used in \code{efs} based smoothing parameter estimation.}

\item{parallel}{If \code{TRUE} parallel computing is used during estimation. This can only be used by Windows users for now.}

\item{no_cores}{Number of cores used if parallel computing chosen. The default is 2. If \code{NULL}, all available cores are used.}

\item{cens.state}{Code used in the dataset to indicate the censored states.}

\item{living.exact}{Name of the variable in the dataset indicating whether an observation is exactly observed or not.}

\item{verbose}{If \code{TRUE}, prints the convergence criterion obtained at each iteration of the full algorithm. The default is \code{FALSE}.}
}
\value{
The function returns an object of class \code{fmsm} as described in \code{fmsmObject}.
}
\description{
Fits a flexible multistate survival model. Any type of process is supported,
including both forward and backward transitions, and must be specified by providing a list
of equations, one for each transition intensity allowed. Any type of observation scheme is
allowed: the process can be observed in continuous time, intermittently at fixed times, there
can be an absorbing state as well as censored states.
 Virtually any type of covariate
effects are supported and can be specified by means of splines, with the same syntax used
to specify Generalised Additive Models (GAMs) in \code{R}.
}
\examples{
\dontrun{

##################################################
# MULTISTATE SURVIVAL MODELLING with CAV DATA ####
##################################################

library(flexmsm)

# Import data
Data <- IDM_cav

# MODEL SPECIFICATION ####
formula <- list(years ~ s(years, bs = 'cr', k = 10) + dage + pdiag, # 1-2
                years ~ s(years, bs = 'cr', k = 10) + dage + pdiag, # 1-3
                0,                                                  # 2-1
                years ~ s(years, bs = 'cr', k = 10) + dage + pdiag, # 2-3
                0,                                                  # 3-1
                0                                                   # 3-2
)

# Counts of pairs of consecutive states observed (C = counts, T = times)
counts.CT <- state.pairs.CT(formula = formula, data = Data, time = 'years',
                            state = 'state', id = 'PTNUM')
counts.CT$counts


# MODEL FITTING ###
# (Warning: takes about 18 minutes on a machine with Windows 10,
#  Intel 2.20 GHz core, 16 GB of RAM and 8 cores. The default is
# to use 2 cores, this takes about 26 minutes.)
fmsm.out <- fmsm(formula = formula, data = Data,
                 id = PTNUM, state = state, death = TRUE,
                 fit = TRUE, parallel = TRUE, no_cores = 2,
                 pmethod = 'analytic')

print(fmsm.out)

AIC(fmsm.out)
BIC(fmsm.out)

# FITTING SUMMARY ####
summary(fmsm.out)
conv.check(fmsm.out)

####################
# VISUALISATION ####
####################

# PLOT THE SMOOTHS OF TIME FOR EACH TRANSITION ####
# par(mfrow = c(1,3))
plot(fmsm.out)


# Consider a patient with:
dage.pred <- 16      # - 16 year old donor
pdiag.pred <- 0      # - IDC as principal diagnosis
start.pred <- 0      # - start observation at time t = 0
stop.pred <- 15      # - t = 15 years for time horizon
n.pred <- 21         # - 21 time points
no.state.pred <- -13 # - (because we don't need this, so anything is fine)

newdata <- data.frame(PTNUM = rep(1, n.pred),
                      years = seq(start.pred, stop.pred, length.out = n.pred),
                      state = rep(no.state.pred, n.pred),
                      dage = rep(dage.pred, n.pred), pdiag = rep(pdiag.pred, n.pred))



# ESTMATED TRANSITION INTENSITIES ####

# Plot of estimated transition intensities
# par(mfrow = c(1,3))
Q.hat <- Q.pred(fmsm.out, newdata = newdata, get.CI = TRUE, plot.Q = TRUE, rug = TRUE,
                ylim = c(0, 1.5))

# Estimated transition intensity matrix at, e.g., t = 0
round(Q.hat$Q.hist[,,1], 3)



# ESTMATED TRANSITION PROBABILITIES ####

# Plot of estimated transition probabilities
# par(mfrow = c(2,3))
P.hat <- P.pred(fmsm.out, newdata = newdata, get.CI = TRUE, plot.P = TRUE, rug = TRUE)

# Estimated 15 year transition probability matrix
round(P.hat$P.pred, 3)
# e.g., there is a 6.2\% chance of observing CAV onset 15 years after transplant


}

}
