% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smoothFPCA.R
\name{FPCA.FEM}
\alias{FPCA.FEM}
\title{Smooth Functional Principal Component Analysis}
\usage{
FPCA.FEM(locations = NULL, datamatrix, FEMbasis, lambda, nPC = 1, validation = NULL,
                NFolds = 5,GCVmethod = "Stochastic", nrealizations = 100, search = "tree",
                bary.locations = NULL)
}
\arguments{
\item{locations}{A #observations-by-2 matrix in the 2D case and #observations-by-3 matrix in the 2.5D and 3D case, where
each row specifies the spatial coordinates \code{x} and \code{y} (and \code{z} in 2.5D and 3D) of the corresponding
observation in the \code{datamatrix}.
If the locations of the observations coincide with (or are a subset of) the nodes of the mesh in the \code{FEMbasis},
leave the parameter \code{locations = NULL} for a faster implementation.}

\item{datamatrix}{A matrix of dimensions #samples-by-#locations with the observed data values over the domain
for each sample. The datamatrix needs to have zero mean.
If the \code{locations} argument is left \code{NULL} the datamatrix has to be dimensions #samples-by-#nodes where #nodes
is the number of nodes of the mesh in the FEMbasis. In this case, each observation is associated to the corresponding
node in the mesh.
If the data are observed only on a subset of the mesh nodes, fill with \code{NA} the values of the
\code{datamatrix} in correspondence of unobserved data.}

\item{FEMbasis}{A \code{FEMbasis} object describing the Finite Element basis, as created by \code{\link{create.FEM.basis}}.}

\item{lambda}{A scalar or vector of smoothing parameters.}

\item{nPC}{An integer specifying the number of Principal Components to compute.}

\item{validation}{A string specifying the type of validation to perform. If \code{lambda} is a vector, it has to
be specified as \code{"GCV"} or \code{"KFold"}. This parameter specify which method of cross-validation is used
to select the best parameter \code{lambda} among those values of the smoothing parameter specified in \code{lambda}
for each Principal Component.}

\item{NFolds}{This parameter is used only in case \code{validation = "KFold"}. It is an integer specifying
the number of folds to use if the KFold cross-validation method for the
selection of the best parameter \code{lambda} is chosen. Default value is 5.}

\item{GCVmethod}{This parameter is considered only when \code{validation = "GCV"}. It can be either "Exact" or
"Stochastic". If set to "Exact" the algoritm performs an exact (but possibly slow) computation
of the GCV index. If set to "Stochastic" the GCV is approximated by a stochastic algorithm.}

\item{nrealizations}{The number of realizations to be used in the stochastic algorithm for the estimation of GCV.}

\item{search}{a flag to decide the search algorithm type (tree or naive or walking search algorithm).}

\item{bary.locations}{A list with three vectors:
\code{locations}, location points which are same as the given locations options. (checks whether both locations are the same);
\code{element ids}, a vector of element id of the points from the mesh where they are located;
\code{barycenters}, a vector of barycenter of points from the located element.}
}
\value{
A list with the following variables:
\describe{
\item{\code{loadings.FEM}}{A \code{FEM} object that represents the L^2-normalized functional loadings for each
Principal Component computed.}
\item{\code{scores}}{A #samples-by-#PrincipalComponents matrix that represents the unnormalized scores or PC vectors.}
\item{\code{lambda}}{A vector of length #PrincipalComponents with the values of the smoothing parameter \code{lambda}
chosen for that Principal Component.}
\item{\code{variance_explained}}{A vector of length #PrincipalComponents where each value represent the variance explained by that component.}
\item{\code{cumsum_percentage}}{A vector of length #PrincipalComponents containing the cumulative percentage of the variance explained by the first components.}
\item{\code{bary.locations}}{A barycenter information of the given locations if the locations are not mesh nodes.}
}
}
\description{
This function implements a smooth functional principal component analysis over a planar mesh,
a smooth manifold or a volume.
}
\examples{
library(fdaPDE)

## Load the hub data
data(hub2.5D)
hub2.5D.nodes = hub2.5D$hub2.5D.nodes
hub2.5D.triangles = hub2.5D$hub2.5D.triangles

mesh = create.mesh.2.5D(nodes = hub2.5D.nodes, triangles = hub2.5D.triangles)
## Create the Finite Element basis
FEMbasis = create.FEM.basis(mesh)
## Create a datamatrix
datamatrix = NULL
for(ii in 1:50){
  a1 = rnorm(1, mean = 1, sd = 1)
  a2 = rnorm(1, mean = 1, sd = 1)
  a3 = rnorm(1, mean = 1, sd = 1)

  func_evaluation = numeric(nrow(mesh$nodes))
  for (i in 0:(nrow(mesh$nodes)-1)){
    func_evaluation[i+1] = a1* sin(2*pi*mesh$nodes[i+1,1]) +
                           a2* sin(2*pi*mesh$nodes[i+1,2]) +
                           a3*sin(2*pi*mesh$nodes[i+1,3]) + 1
  }
  data = func_evaluation + rnorm(nrow(mesh$nodes), mean = 0, sd = 0.5)
  datamatrix = rbind(datamatrix, data)
}
## Compute the mean of the datamatrix and subtract it to the data
data_bar = colMeans(datamatrix)
data_demean = matrix(rep(data_bar,50), nrow=50, byrow=TRUE)

datamatrix_demeaned = datamatrix - data_demean
## Set the smoothing parameter lambda
lambda = 0.00375
## Estimate the first 2 Principal Components
FPCA_solution = FPCA.FEM(datamatrix = datamatrix_demeaned,
                      FEMbasis = FEMbasis, lambda = lambda, nPC = 2)

## Plot the functional loadings of the estimated Principal Components
plot(FPCA_solution$loadings.FEM)
}
\references{
Lila, E., Aston, J.A.D.,  Sangalli, L.M., 2016a. Smooth Principal Component Analysis over two-dimensional
manifolds with an application to neuroimaging. Ann. Appl. Stat., 10(4), pp. 1854-1879.
}
