% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/fdaPDE.smoothing.R
\name{smooth.FEM.basis}
\alias{smooth.FEM.basis}
\title{Spatial regression with differential regularization: stationary and isotropic case (Laplacian)}
\usage{
smooth.FEM.basis(locations = NULL, observations, FEMbasis, lambda,
       covariates = NULL, BC = NULL, GCV = FALSE, CPP_CODE = TRUE)
}
\arguments{
\item{locations}{A #observations-by-2 matrix where each row specifies the spatial coordinates \code{x} and \code{y} of the corresponding observations in the vector \code{observations}.
This parameter can be \code{NULL}. In this case the spatial coordinates of the corresponding observations are assigned as specified in \code{observations}.}

\item{observations}{A vector of length #observations with the observed data values over the domain.
The locations of the observations can be specified with the \code{locations} argument.
Otherwise if only the vector of observations is given, these are consider to be located in the corresponding node in the table
\code{nodes} of the mesh. In this last case, an \code{NA} value in the \code{observations} vector indicates that there is no observation associated to the corresponding
 node.}

\item{FEMbasis}{A \code{FEMbasis} object describing the Finite Element basis, as created by \code{\link{create.FEM.basis}}.}

\item{lambda}{A scalar or vector of smoothing parameters.}

\item{covariates}{A #observations-by-#covariates matrix where each row represents the covariates associated with the corresponding observed data value in \code{observations}.}

\item{BC}{A list with two vectors:
\code{BC_indices}, a vector with the indices in \code{nodes} of boundary nodes where a Dirichlet Boundary Condition should be applied;
\code{BC_values}, a vector with the values that the spatial field must take at the nodes indicated in \code{BC_indices}.}

\item{GCV}{Boolean. If \code{TRUE} the following quantities are computed: the trace of the smoothing matrix, the estimated error standard deviation,  and
the Generalized Cross Validation criterion, for each value of the smoothing parameter specified in \code{lambda}.}

\item{CPP_CODE}{Boolean. If \code{TRUE} the computation relies on the C++ implementation of the algorithm. This usually ensures a much faster computation.}
}
\value{
A list with the following variables:
\item{\code{fit.FEM}}{A \code{FEM} object that represents the fitted spatial field.}
\item{\code{PDEmisfit.FEM}}{A \code{FEM} object that represents the Laplacian of the estimated spatial field.}
\item{\code{beta}}{If covariates is not \code{NULL}, a vector of length #covariates with the regression coefficients associated with each covariate.}
\item{\code{edf}}{If GCV is \code{TRUE}, a scalar or vector with the trace of the smoothing matrix for each value of the smoothing parameter specified in \code{lambda}.}
\item{\code{stderr}}{If GCV is \code{TRUE}, a scalar or vector with the estimate of the standard deviation of the error for each value of the smoothing parameter specified in \code{lambda}.}
\item{\code{GCV}}{If GCV is \code{TRUE}, a  scalar or vector with the value of the GCV criterion for each value of the smoothing parameter specified in \code{lambda}.}
}
\description{
This function implements a spatial regression model with differential regularization; isotropic and stationary case. In particular, the regularizing term involves the Laplacian of the spatial field. Space-varying covariates can be included in the model. The technique accurately handle data distributed over irregularly shaped domains. Moreover, various conditions can be imposed at the domain boundaries.
}
\examples{
library(fdaPDE)
## Load the Meuse data and a domain boundary for these data
data(MeuseData)
data(MeuseBorder)
## Create a triangular mesh for these data with the provided boundary and plot it
order=1
mesh <- create.MESH.2D(nodes = MeuseData[,c(2,3)], segments = MeuseBorder, order = order)
plot(mesh)
## Create the Finite Element basis
FEMbasis = create.FEM.basis(mesh, order)
## Estimate zync field without using covariates, setting the smoothing parameter to 10^3.5
data = log(MeuseData[,"zinc"])
lambda = 10^3.5
ZincMeuse = smooth.FEM.basis(observations = data,
                             FEMbasis = FEMbasis, lambda = lambda)
## Plot the estimated spatial field
plot(ZincMeuse$fit.FEM)
# Now repeat the analysis using as covariates the square root of the log-distance
# from river \\code{sqrt(dist.log(m))} and the altitude \\code{elev}
desmat = matrix(1,nrow=nrow(MeuseData),ncol=2)
desmat[,1] = sqrt(MeuseData[,"dist.log(m)"])
desmat[,2] = MeuseData[,"elev"]
ZincMeuseCovar = smooth.FEM.basis(observations = data, covariates = desmat,
                                   FEMbasis = FEMbasis, lambda = lambda)
# Plot of the non parametric part (f) of the regression model y_i = beta_1 x_i1 + beta_2 x_i2 + f
plot(ZincMeuseCovar$fit.FEM)
# Print covariates' regression coefficients
print(ZincMeuseCovar$beta)
}
\references{
Sangalli, L.M., Ramsay, J.O. & Ramsay, T.O., 2013. Spatial spline regression models. Journal of the Royal Statistical Society. Series B: Statistical Methodology, 75(4), pp. 681-703.
}
\seealso{
\code{\link{smooth.FEM.PDE.basis}}, \code{\link{smooth.FEM.PDE.sv.basis}}
}

