% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/expirest_osle.R
\name{expirest_osle}
\alias{expirest_osle}
\title{Ordinary shelf life estimation (osle)}
\usage{
expirest_osle(
  data,
  response_vbl,
  time_vbl,
  batch_vbl,
  sl,
  sl_sf,
  srch_range,
  alpha = 0.05,
  alpha_pool = 0.25,
  xform = c("no", "no"),
  shift = c(0, 0),
  sf_option = "loose",
  ivl = "confidence",
  ivl_type = "one.sided",
  ivl_side = "lower",
  ...
)
}
\arguments{
\item{data}{A data frame with the columns specified by \code{response_vbl},
\code{time_vbl} and \code{batch_vbl}.}

\item{response_vbl}{A character string specifying the response variable name
that must be a column of \code{data}.}

\item{time_vbl}{A character string specifying the time variable name that
must be a column of \code{data}.}

\item{batch_vbl}{A character string specifying the column in \code{data}
with the grouping information (i.e. a factorial variable) for the
differentiation of the observations of the different batches.}

\item{sl}{A numeric value or a numeric vector of length \code{2} specifying
the specification limit or limits. If a vector is provided it must be of
the form \code{c(lower limit, upper limit)}.}

\item{sl_sf}{A positive integer or a vector of positive integers specifying
the number of \dQuote{significant figures} (sf) of \code{sl}. It must have
the same length as \code{sl}.}

\item{srch_range}{A vector of length \code{2} specifying the end-points of
the (time) range that is supposed to contain the shelf life or retest
period.}

\item{alpha}{A numeric value between 0 and 1 specifying the significance
level for the calculation of confidence or prediction intervals. The
default is \code{0.05}.}

\item{alpha_pool}{A numeric value between 0 and 1 specifying the type I
error rate for the test of the poolability of the batches. The default
is \code{0.25}, following ICH Q1E guideline. The value should not be
changed unless supported by well-founded reasons.}

\item{xform}{A vector of two character strings specifying the transformation
of the response and the time variable. The default is \dQuote{no}
transformation, i.e. \code{c("no", "no")}, where the first element
specifies the transformation of the \eqn{x} variable and the second
element the transformation of the \eqn{y} variable. Valid alternatives
for \eqn{x} and/or \eqn{y} variable transformation are \code{"log"}
(natural logarithm), \code{"sqrt"} (square root) and \code{"sq"} (square).}

\item{shift}{A vector of two values which will be added to the variables
\eqn{x} and/or \eqn{y} before they are transformed as specified by the
\code{xform} parameter, where the first element will be added to the
\eqn{x} variable and the second element to the \eqn{y} variable. The
purpose is to prevent an undefined state which could arise when variables
with values of \eqn{\leq 0} are log or square root transformed. The
default is \code{c(0, 0)}.}

\item{sf_option}{A character string specifying if the limits (\code{rl}
or \code{sl}) should be regarded as \dQuote{tight} or \dQuote{loose}, i.e.
either \code{"tight"} or \code{"loose"}, respectively. The option
\code{"tight"} means that the limits are rounded to the specified number
of significant figures specified by the parameters \code{rl_sf} and
\code{sl_sf}. In case of the option \code{"loose"} the limits are rounded
to the specified number of significant figures (\eqn{n}), followed by the
subtraction of \eqn{1} from the \eqn{n^{th}} digit and addition of
\eqn{5} to the \eqn{(n + 1)^{th}} digit if \code{ivl_side} is
\code{"lower"}, or followed by the addition of \eqn{4} to the
\eqn{(n + 1)^{th}} digit if \code{ivl_side} is \code{"upper"}.}

\item{ivl}{A character string of either \code{"confidence"} or
\code{"prediction"}, i.e. specifying the type of interval of interest.
The default is \code{"confidence"}.}

\item{ivl_type}{A character string specifying if a \dQuote{one sided} or
a \dQuote{two sided} interval should be calculated, i.e. either
\code{"one.sided"} or \code{"two.sided"}, respectively. The default is
\code{"one.sided"}.}

\item{ivl_side}{A character string specifying if the \dQuote{upper} or the
\dQuote{lower} limit is the relevant limit, i.e. either \code{"upper"} or
\code{"lower"}, respectively. The default is \code{"lower"}.}

\item{...}{Additional named or unnamed arguments passed on to
\code{uniroot()}.}
}
\value{
An object of class \sQuote{\code{expirest_osle}} is returned,
containing the following elements:
\item{Data}{Data frame of the original data including new columns with
  transformed variables, if applicable.}
\item{Parameters}{A list of the parameters with the elements \code{alpha},
  \code{alpha.pool}, \code{ivl}, \code{ivl.type} and \code{ivl.side}.}
\item{Variables}{A list of the variable names, i.e. the original names of
  \code{batch_vbl}, \code{time_vbl} and \code{response_vbl} and, if
  applicable, of the transformed variables.}
\item{Model.Type}{A list of five elements specifying which model, based on
  the ANCOVA analysis, suits best. The first element (\code{type.spec})
  is a numeric vector of length 2 specifying the best model accepted at the
  significance level of 0.25. The first number represents the decision on
  the intercept and the second on the slope, where \code{1} stands for
  \dQuote{common} and \code{2} stands for \dQuote{different}. The second
  element (\code{type.acronym}) is an acronym representing the first item.}
\item{Models}{A list of four elements named \code{cics}, \code{dics},
  \code{dids} and \code{individual}. The first three elements contain the
  \sQuote{\code{lm}} objects of the \dQuote{common intercept / common slope}
  (\code{cics}), \dQuote{different intercept / common slope} (\code{dics})
  and \dQuote{different intercept / different slope} (\code{dids}) models.
  The fourth element is a list of the \sQuote{\code{lm}} objects of the
  models obtained from fitting the data of each batch individually.}
\item{AIC}{A numeric named vector of the Akaike Information Criterion (AIC)
  values of each of the three fitted models.}
\item{BIC}{A numeric named vector of the Bayesian Information Criterion (BIC)
  values of each of the three fitted models.}
\item{wc.icpt}{A numeric value of the worst case intercept.}
\item{wc.batch}{A numeric value of the worst case batch. In case of the
 \code{dids} model type, the estimation is done using the models obtained
 from fitting the data of each batch individually.}
\item{Limits}{A list of all limits.}
\item{Intercepts}{A list of the intercepts of the three fitted models.}
\item{POI}{A numeric named vector of the POI values of each of the three
  fitted models. In case of the \code{dids} model type, the estimation is
  done using the models obtained from fitting the data of each
  batch individually.}
}
\description{
Based on a linear regression model fitted to a stability data set the
function \code{expirest_osle()} estimates the shelf life, or retest period,
following the ICH Q1E guideline. The abbreviation \dQuote{osle} stands for
\dQuote{ordinary shelf life estimation}.
}
\details{
According to ICH Q1E guideline, \dQuote{\emph{an appropriate
approach to retest period or shelf life estimation is to analyse a
quantitative  attribute by determining the earliest time at which the 95
percent confidence limit for the mean intersects the proposed acceptance
criterion}} (in this package, this point is called the \dQuote{point of
intersection} (POI)). Furthermore, it says that \dQuote{\emph{for an
attribute known to increase with time, the upper one-sided 95 percent
confidence limit should be compared to the acceptance criterion. For an
attribute that can either increase or decrease, or whose direction of change
is not known, two-sided 95 percent confidence limits should be calculated
and compared to the upper and lower acceptance criteria.}} With respect to
the number of batches to be included in the analysis it says that
\dQuote{\emph{the retest period or shelf life is generally estimated based
on the stability data from a minimum of three batches.}}

Before performing the retest period or shelf life estimation the most
suitable model should be determined. It should particularly be verified
if data of all test batches are poolable or not. Details on this are
described in section \dQuote{Checking batch poolability} below.
}
\section{Checking batch poolability}{

According to ICH Q1E guideline, construction of the 95\% confidence interval
on the basis of the combined data of all test batches is allowed only if it
has been confirmed by aid of a statistical test whether the regression lines
from the different batches have a common slope and a common intercept. A
significance level of \code{alpha_pool = 0.25} has to be used for both
batch-related terms, and the test of the slopes has to precede the test of
the intercepts. From these tests, three possible models may be appropriate,
i.e.
\itemize{
 \item a \emph{common intercept / common slope} model (cics),
 \item a \emph{different intercept / common slope} model (dics) or
 \item a \emph{different intercept / different slope} model (dids).
}
The \emph{common intercept / different slope} model is not of practical
relevance because the corresponding model is missing an effect. If the slopes
are significantly different, there is no point comparing intercepts.

These requirements can be checked by aid of an \dQuote{ANalysis of
COVAriance} (ANCOVA) including the batch variable as interaction term. The
full ANCOVA model simultaneously tests all the effects, and non-significant
effects can be identified and removed for fitting of the final regression
model that is used for the estimation of the shelf life or retest period.

The significance level (\code{alpha_pool = 0.25}, Type I error) is used to
increase the power of the test to detect cases where the data should not be
pooled. Setting \code{alpha_pool = 0.25} decreases the probability of
incorrectly concluding that stability data from multiple batches can be
pooled. On the other hand, though, it increases the probability of using a
single batch to determine expiry when pooling batches would be more
appropriate.
}

\examples{
# Potency stability data (in \% of label claim (LC)) of five batches of a drug
# product obtained over a 24 months period:
str(exp1)

# 'data.frame':	53 obs. of  3 variables:
# $ Batch  : Factor w/ 6 levels "b2","b3","b4",..: 1 1 1 1 1 1 1 1 1 1 ...
# $ Month  : num  0 1 3 3 6 6 12 12 24 24 ...
# $ Potency: num  101 101.3 99.8 99.2 99.5 ...

# Successful estimations
res1 <-
  expirest_osle(data = exp1[exp1$Batch \%in\% c("b2", "b5", "b7"), ],
                response_vbl = "Potency", time_vbl = "Month",
                batch_vbl = "Batch", sl = 95, sl_sf = 3, srch_range = c(0, 500))
res1$Model.Type
res1$POI

# Expected results in res1$Model.Type
# $type.spec
# common.icpt  common.slp
#           1           1
#
# $type.acronym
# [1] "cics"

# Expected results in res1$POI
#     cics     dics     dids
# 26.22410 24.80030 23.34184

res2 <-
  expirest_osle(data = exp1[exp1$Batch \%in\% c("b4", "b5", "b8"), ],
                response_vbl = "Potency", time_vbl = "Month",
                batch_vbl = "Batch", sl = 95, sl_sf = 3, srch_range = c(0, 500))
res2$Model.Type
res2$POI

# Expected results in res2$Model.Type
# $type.spec
# common.icpt  common.slp
#           0           0
#
# $type.acronym
# [1] "dids"

# Expected results in res2$POI
#     cics     dics     dids
# 28.12518 22.47939 15.96453

# Moisture stability data (\% (w/w)) of three batches of a drug product obtained
# over a 24 months period:
str(exp3)

# 'data.frame':	33 obs. of  3 variables:
# $ Batch   : Factor w/ 3 levels "b1","b2","b3": 1 1 1 1 1 1 1 1 1 1 ...
# $ Month   : num  0 1 2 3 3 6 6 12 12 24 ...
# $ Moisture: num  2.2 1.7 3.32 2.76 2.43 ...

# Unsuccessful estimations
# Intervals are wider than the specification limits (no intersection)
res3 <-
  expirest_osle(
    data = exp3, response_vbl = "Moisture", time_vbl = "Month",
    batch_vbl = "Batch", sl = 1.5, sl_sf = 2, srch_range = c(0, 500),
    sf_option = "tight", ivl = "prediction", ivl_type = "two.sided",
    ivl_side = "lower")
res3$POI

# (Expected) results in res3$POI
# cics dics dids
#   NA   NA   NA

# Estimation may also fail because of an inappropriate 'srch-range' setting.
res4 <-
  expirest_osle(data = exp1[exp1$Batch \%in\% c("b2", "b5", "b7"), ],
                response_vbl = "Potency", time_vbl = "Month",
                batch_vbl = "Batch", sl = 95, sl_sf = 3, srch_range = c(0, 5))
res4$POI

# (Expected) results in res4$POI
# cics dics dids
#   NA   NA   NA
}
\references{
International Council for Harmonisation of Technical Requirements for
Registration of Pharmaceuticals for Human (ICH), Harmonised Tripartite
Guideline, Evaluation of Stability Data Q1E, step 4, February 2003
(CPMP/ICH/420/02).\cr
\url{https://www.ich.org/page/quality-guidelines}
}
\seealso{
\code{\link{expirest_wisle}}, \code{\link{find_poi}},
\code{\link[stats]{uniroot}}, \code{\link[stats]{lm}},
\code{\link[stats]{AIC}}, \code{\link[stats]{BIC}}.
}
