#' Bootstrap analysis via resampling
#'
#'
#'   Calculates the repeatability of the covariance matrix of the suplied data
#'   via bootstrap resampling
#'
#'   Samples with replacement are taken from the full population, a statistic calculated
#'   and compared to the full population statistic. 
#'
#' @param ind.data Matrix of residuals or indiviual measurments
#' @param ComparisonFunc comparison function
#' @param ... Aditional arguments passed to ComparisonFunc
#' @param iterations Number of resamples to take
#' @param correlation If TRUE, correlation matrix is used, else covariance matrix. 
#' @param parallel if TRUE computations are done in parallel. Some foreach backend must be registered, like doParallel or doMC.
#' @return returns the mean repeatability, that is, the mean value of comparisons from samples to original statistic.
#' @author Diogo Melo, Guilherme Garcia
#' @seealso \code{\link{MonteCarloStat}}, \code{\link{AlphaRep}}
#' @export
#' @examples
#' BootstrapRep(iris[,1:4], MantelCor, iterations = 5, correlation = TRUE)
#'              
#' BootstrapRep(iris[,1:4], RandomSkewers, iterations = 50)
#' 
#' BootstrapRep(iris[,1:4], KrzCor, iterations = 50, correlation = TRUE)
#' 
#' BootstrapRep(iris[,1:4], PCAsimilarity, iterations = 50)
#' 
#' #Multiple threads can be used with some foreach backend library, like doMC or doParallel
#' #library(doParallel)
#' ##Windows:
#' #cl <- makeCluster(2)
#' #registerDoParallel(cl)
#' ##Mac and Linux:
#' #registerDoParallel(cores = 2)
#' #BootstrapRep(iris[,1:4], PCAsimilarity,
#' #             iterations = 5,
#' #             parallel = TRUE)
#' @keywords bootstrap
#' @keywords repetabilities

BootstrapRep <- function(ind.data,
                         ComparisonFunc,
                         ...,
                         iterations = 1000, 
                         correlation = FALSE, 
                         parallel = FALSE){
  if(correlation)  {StatFunc <- cor; c2v <- cov2cor
  } else {StatFunc <- cov; c2v <- function(x) x}
  repeatability <- BootstrapRep_primitive(ind.data, iterations,
                                          ComparisonFunc = function(x, y) ComparisonFunc(c2v(x), 
                                                                                         c2v(y), ...),
                                          StatFunc = StatFunc,
                                          parallel = parallel)
  return(mean(repeatability[,2]))
}

BootstrapRep_primitive <- function (ind.data, iterations,
                                    ComparisonFunc, StatFunc,
                                    parallel = FALSE){
  if(isSymmetric(as.matrix(ind.data))) stop("input appears to be a matrix, use residuals.")
  sample.size <-  dim (ind.data) [1]
  c.matrix <- StatFunc(ind.data)
  populations  <- alply(1:iterations, 1,
                        function(x) ind.data[sample (1:sample.size, sample.size, TRUE),],
                        .parallel = parallel)
  comparisons <- ldply (populations, function (x) ComparisonFunc (c.matrix, StatFunc(x)),
                        .parallel = parallel)
  return (comparisons)
}

#' R2 confidence intervals by bootstrap resampling
#'
#' Random populations are generated by  resampling 
#' the suplied data or residuals. R2 is calculated on all the
#' random population's correlation matrices, provinding a distribution based on the original data.
#'
#' @param ind.data Matrix of residuals or indiviual measurments
#' @param iterations Number of resamples to take
#' @param parallel if TRUE computations are done in parallel. Some foreach backend must be registered, like doParallel or doMC.
#' @return returns a vector with the R2 for all populations
#' @author Diogo Melo Guilherme Garcia
#' @seealso \code{\link{BootstrapRep}}, \code{\link{AlphaRep}}
#' @export
#' @import plyr
#' @importFrom mvtnorm rmvnorm
#' @examples
#' r2.dist <- BootstrapR2(iris[,1:4], 30)
#' quantile(r2.dist)
#' @keywords bootstrap
#' @keywords integration
#' @keywords repeatability
BootstrapR2 <- function (ind.data, iterations = 1000, parallel = FALSE) {
  it.r2 <- BootstrapRep_primitive(ind.data, iterations,
                          ComparisonFunc = function(x, y) y,
                          StatFunc = function(x) CalcR2(cor(x)),
                          parallel = parallel)
  return (it.r2[,2])
}
