% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mle-factory.R
\name{smle}
\alias{smle}
\alias{smle.default}
\alias{smle.intensity}
\title{Simple maximum likelihood estimation}
\usage{
smle(data, ...)

\method{smle}{default}(data, f, param_init, max = TRUE, ...)

\method{smle}{intensity}(data, f, param_init, max = TRUE, ...)
}
\arguments{
\item{data}{The data set to work with. It can be a vector (if there is only
one variable), a data frame (if there is one or more variables) or an
\code{\link{intensity}} object.}

\item{...}{Additional arguments to be passed to \code{\link[stats]{optim}}.}

\item{f}{A function to be maximized, typically a log-likelihood function.
This function must have only two arguments: \code{data} and \code{param},
which must correspond to the \code{data} argument of \code{smle} and a
named vector of the parameter(s) to be estimated.}

\item{param_init}{Either a named vector with proposed initial values of the
parameter(s), or a function that returns such a vector. This parameter
is not needed if the parameter \code{param} of \code{f} is already
provided with such a named vector.}

\item{max}{Does \code{f} need to be maximized? Set to \code{FALSE} to require
a minimization of \code{f}.}
}
\description{
By default, this function performs a maximum likelihood estimation for one or
several parameters, but it can be used for any other optimization problems.
The interface is intented to be rather simple while allowing more advanced
parametrizations.
}
\details{
The \code{\link[stats]{optim}} tool does the hard work under the hood. Extra
arguments (e.g. method of optimization to be used) can be passed to
\code{\link[stats]{optim}} through the \code{...} argument. Note that
contrary to the default \code{\link[stats]{optim}} arguments, \code{smle}
tries to solve a maximization problem using the method "L-BFGS-B" by default
(see \code{\link[stats]{optim}} documentation for more information).
}
\examples{
set.seed(12345)
data <- rlogis(100, location = 5, scale = 2)
ll_logis <- function(data, param = c(location = 0, scale = 1)) {
    sum(dlogis(x = data, location = param[["location"]],
               scale = param[["scale"]], log = TRUE))
}
res <- smle(data, ll_logis)
res
summary(res)

# Using the magrittr syntax:
require(magrittr)
data \%>\% smle(ll_logis)

# Comparision with the output of fitdistr (MASS package), which works for a
# limited number of predefined distributions:
require(MASS)
fitdistr(data, "logistic")

# Example with an intensity object:
require(magrittr)
require(dplyr)
data <- tomato_tswv$field_1929 \%>\%
    filter(t == 1) \%>\%
    incidence() \%>\%
    clump(unit_size = c(x = 3, y = 3))
ll_betabinom <- function(data, param) {
    sum(dbetabinom(x = data[["i"]], size = data[["n"]],
                   prob = param[["prob"]], theta = param[["theta"]],
                   log = TRUE))
}
epsilon <- 1e-7
res <- smle(data, ll_betabinom, param_init = c(prob = 0.5, theta = 0.5),
            lower = c(prob  = 0 + epsilon,
                      theta = 0 + epsilon),
            upper = c(prob = 1 - epsilon,
                      theta = Inf))
res
summary(res)

param_init <- data.frame(lower = c(0 + epsilon, 0 + epsilon),
                         start = c(0.5, 0.5),
                         upper = c(1 - epsilon, Inf))
rownames(param_init) <- c("prob", "theta")
res <- smle(data, ll_betabinom, param_init)
res
summary(res)

}
\keyword{internal}
