% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/factors.R
\name{comb_facs}
\alias{comb_facs}
\alias{split_fac}
\alias{add_grouping}
\title{Manipulate factors in a reference grid}
\usage{
comb_facs(object, facs, newname = paste(facs, collapse = "."), drop = FALSE)

split_fac(object, fac, newfacs)

add_grouping(object, newname, refname, newlevs)
}
\arguments{
\item{object}{An object of class \code{emmGrid}}

\item{facs}{Character vector. The names of the factors to combine}

\item{newname}{Character name of grouping factor to add (different from any
existing factor in the grid)}

\item{drop}{Logical value. If \code{TRUE}, any levels of the new factor that 
are dropped if all occurrences in the newly reconstructed object have 
weight zero. If \code{FALSE}, all levels are retained. 
(This argument is ignored if there is no \code{.wgt.} column
in \code{object@grid}.)}

\item{fac}{The name of a factor that is part of the grid in \code{object}}

\item{newfacs}{A named list with the names of new factors
and their levels. The names must not already exist in the object,
and the product of the lengths of the levels must equal the number
of levels of \code{fac}.}

\item{refname}{Character name(s) of the reference factor(s)}

\item{newlevs}{Character vector or factor of the same length as that of the (combined) levels for 
\code{refname}. The grouping factor \code{newname} will have the unique
values of \code{newlevs} as its levels. The order of levels in \code{newlevs}
is the same as the order of the level combinations produced by 
\code{\link{expand.grid}} applied to the levels of \code{refname} -- that is, the
first factor's levels change the fastest and the last one's vary the slowest.}
}
\value{
A modified object of class \code{emmGrid}
}
\description{
These functions manipulate the velels of factors comprising a reference
grid by combining factor levels, splitting a factor's levels into 
combinations of newly-defined factors, or by creating a grouping factor in which 
factor(s) levels are nested.
}
\section{The \code{comb_facs} function}{

\code{comb_facs} combines the levels of factors into a single factor
in the reference grid (similar to \code{\link{interaction}}). This new factor
replaces the factors that comprise it.

\emph{Additional note:}
The choice of whether to drop levels or not can make a profound difference.
If the goal is to combine factors for use in \code{joint_tests}, we advise \emph{against} 
\code{drop = TRUE} because that might change the weights used in deriving marginal means.
If combining factors in a nested structure, dropping unused cases can considerably reduce 
the storage required.
}

\section{The \code{split_fac} function}{

The levels in \code{newfacs} are expanded via \code{\link{expand.grid}} into
combinations of levels, and the factor \code{fac} is replaced by those 
factor combinations. Unlike \code{add_grouping}, this creates a crossed, 
rather than a nested structure. Note that the order of factor combinations
is systematic with the levels of first factor in \code{newfacs} varying 
the fastest; and those factor combinations are assigned respectively
to the levels of \code{fac} as displayed in \code{str(object)}.
}

\section{The \code{add_grouping} function}{

This function adds a grouping factor to an existing reference grid or other 
\code{emmGrid} object, such that the levels of one or more existing factors (call them the
reference factors) are mapped to a smaller number of levels of the new
grouping factor. The reference factors are then nested in a 
new grouping factor named \code{newname}, and a new nesting structure
\code{refname \%in\% newname}.
This facilitates obtaining marginal means of the grouping factor, and 
contrasts thereof.


\emph{Additional notes:} By default, the levels of \code{newname} will be ordered
  alphabetically. To dictate a different ordering of levels, supply 
  \code{newlevs} as a \code{factor} having its levels in the desired order.
  
When \code{refname} specifies more than one factor, this can
  fundamentally (and permanently) change what is meant by the levels of those
  individual factors. For instance, in the \code{gwrg} example below, there
  are two levels of \code{wool} nested in each \code{prod}; and that implies
  that we now regard these as four different kinds of wool. Similarly, there
  are five different tensions (L, M, H in prod 1, and L, M in prod 2).
}

\examples{
mtcars.lm <- lm(mpg ~ factor(vs)+factor(cyl)*factor(gear), data = mtcars)
(v.c.g <- ref_grid(mtcars.lm))
(v.cg <- comb_facs(v.c.g, c("cyl", "gear")))
  
# One use is obtaining a single test for the joint contributions of two factors:
joint_tests(v.c.g)

joint_tests(v.cg)

# undo the 'comb_facs' operation:
split_fac(v.cg, "cyl.gear", list(cyl = c(4, 6, 8), gear = 3:5))

IS.glm <- glm(count ~ spray, data = InsectSprays, family = poisson)
IS.emm <- emmeans(IS.glm, "spray")
IS.new <- split_fac(IS.emm, "spray", list(A = 1:2, B = c("low", "med", "hi")))
str(IS.new)

fiber.lm <- lm(strength ~ diameter + machine, data = fiber)
( frg <- ref_grid(fiber.lm) )

# Suppose the machines are two different brands
brands <- factor(c("FiberPro", "FiberPro", "Acme"), levels = c("FiberPro", "Acme"))
( gfrg <- add_grouping(frg, "brand", "machine", brands) )

emmeans(gfrg, "machine")

emmeans(gfrg, "brand")

### More than one reference factor
warp.lm <- lm(breaks ~ wool * tension, data = warpbreaks)
gwrg <- add_grouping(ref_grid(warp.lm), 
    "prod",  c("tension", "wool"),  c(2, 1, 1,  1, 2, 1))
        # level combinations:         LA MA HA  LB MB HB

emmeans(gwrg, ~ wool * tension)   # some NAs due to impossible combinations

emmeans(gwrg, "prod")

}
