% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/doTheEvolution.R
\name{doTheEvolution}
\alias{doTheEvolution}
\title{Working horse of the ecr package.}
\usage{
doTheEvolution(task, control, initial.population = NULL, more.args = list())
}
\arguments{
\item{task}{[\code{ecr_optimization_task}]\cr
Optimization task. If a \code{smoof_function} is passed it is automatically
converted into a task.}

\item{control}{[\code{ecr_control}]\cr
ECR control object generated via \code{\link{setupECRControl}}.}

\item{initial.population}{[\code{list}]\cr
List of individuals which should be placed in the initial population.
The function will stop with an error message if the number of passed individuals
is larger than \code{control$n.population}. If the number of passed individuals
is lower than \code{control$n.population}, the population will be filled up
by individuals generated by the corresponding generator.
Default is \code{NULL}, i.e., the entire population is generated by the
population generator.}

\item{more.args}{[\code{list}]\cr
Additional arguments passed to objective function.}
}
\value{
[\code{\link{ecr_result}}]
}
\description{
Takes a function and searches for a global optimum with an evolutionary approach.
}
\examples{
library(smoof)
library(ParamHelpers)
library(ggplot2)

# We want to find the minimum of the function f(x) = x sin(2x) on the interval
# [0, 2pi]. The optimal value is about -5.5 for x = 5.54.
# First we wrap the function with the smoof package:
obj.fn = makeSingleObjectiveFunction(
  name = "My obj. function",
  fn = function(x) x * sin(2 * x),
  par.set = makeParamSet(makeNumericParam("x", lower = 0, upper = 2 * pi))
)

# We want to solve this with a (10 + 10) evolutionary strategy based on
# the floating point representation of the input vectors with the default
# operators: intermediate recombinator and Gaussian mutation
ctrl = setupECRControl(
  n.population = 10L,
  n.offspring = 10L,
  survival.strategy = "plus",
  representation = "float",
  stopping.conditions = setupTerminators(max.iter = 100L)
)
# use the default operators for representation "float"
ctrl = setupEvolutionaryOperators(ctrl)

res = doTheEvolution(obj.fn, ctrl)
print(res)

# Now let us choose a (10, 10) strategy with intermediate recombination. Moreover,
# we want the "fittest" individual to surive each time and therefore set n.elite
# to 1.
ctrl = setupECRControl(
  n.population = 10L,
  n.offspring = 10L,
  survival.strategy = "comma",
  n.elite = 1L,
  representation = "float",
  stopping.conditions = setupTerminators(max.iter = 100L)
)
ctrl = setupEvolutionaryOperators(ctrl)

res = doTheEvolution(obj.fn, ctrl)
print(res)

# Now let us tackle a two-dimensional problem
# Here we aim to find the global optimum of the 2D-Rastrigin function

set.seed(1234)

obj.fn = makeRastriginFunction(2L)
autoplot(obj.fn, show.optimum = TRUE)
# The global optimum is located in x1 = x2 = 0 with a function value of 0.

# We choose a (100 + 10) strategy with crossover recombination here and save
# each population. This way we can visualize the population later on.
ctrl = setupECRControl(
  n.population = 100L,
  n.offspring = 10L,
  survival.strategy = "plus",
  representation = "float",
  logger = setupOptPathLoggingMonitor(),
  stopping.conditions = setupTerminators(max.iter = 50L)
)

ctrl = setupEvolutionaryOperators(
 ctrl,
 parent.selector = setupRouletteWheelSelector(),
 recombinator = setupCrossoverRecombinator(),
 survival.selector = setupGreedySelector()
)
\dontrun{
  res = doTheEvolution(obj.fn, ctrl)
  print(res)
  autoplot(res, show.process = TRUE)
}
}
\seealso{
\code{\link{setupECRControl}}
}
\keyword{optimize}

