% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_eml.R
\name{create_eml}
\alias{create_eml}
\title{Create EML metadata for an ecocomDP dataset}
\usage{
create_eml(
  path,
  source_id,
  derived_id,
  script,
  script_description,
  is_about = NULL,
  contact,
  user_id,
  user_domain,
  basis_of_record = NULL,
  url = NULL
)
}
\arguments{
\item{path}{(character) Path to the directory containing ecocomDP tables, conversion script, and where EML metadata will be written.}

\item{source_id}{(character) Identifier of a data package published in a supported repository. Currently, the EDI Data Repository is supported.}

\item{derived_id}{(character) Identifier of the dataset being created.}

\item{script}{(character) Name of file used to convert \code{source_id} to \code{derived_id}.}

\item{script_description}{(character) Description of \code{script}.}

\item{is_about}{(named character) An optional argument for specifying dataset level annotations describing what this dataset "is about".}

\item{contact}{(data.frame) Contact information for the person that created this ecocomDP dataset, containing these columns:
\itemize{
    \item givenName
    \item surName
    \item organizationName
    \item electronicMailAddress
}}

\item{user_id}{(character) Identifier of user associated with \code{user_domain}.}

\item{user_domain}{(character) Domain (data repository) the \code{user_id} belongs to. Currently, EDI is supported.}

\item{basis_of_record}{(character) An optional argument to facilitate creation of a Darwin Core record from this dataset using \code{convert_to_dwca()}. Use this to define the Darwin Core property \href{https://dwc.tdwg.org/terms/#dwc:basisOfRecord}{basisOfRecord} as \href{http://rs.tdwg.org/dwc/terms/HumanObservation}{HumanObservation} or \href{http://rs.tdwg.org/dwc/terms/MachineObservation}{MachineObservation}.}

\item{url}{(character) URL to the publicly accessible directory containing ecocomDP tables, conversion script, and EML metadata. This argument supports direct download of the data entities by a data repository and is used for automated revisioning and publication.}
}
\value{
An EML metadata file.
}
\description{
Create EML metadata for an ecocomDP dataset
}
\details{
This function creates an EML record for an ecocomDP by combining metadata from \code{source_id} with boiler-plate metadata describing the ecocomDP model. Changes to the \code{source_id} EML include:
    \itemize{
        \item \strong{<access>} Adds \code{user_id} to the list of principals granted read and write access to the ecocomDP data package this EML describes.
        \item \strong{<title>} Adds a note that this is a derived data package in the ecocomDP format.
        \item \strong{<pubDate>} Adds the date this EML was created.
        \item \strong{<abstract>} Adds a note that this is a derived data package in the ecocomDP format.
        \item \strong{<keywordSet} Adds the "ecocomDP" keyword to enable search and discovery of all ecocomDP data packages in the data repository it is published, and 7 terms from the LTER Controlled vocabulary: "communities", "community composition", "community dynamics", "community patterns", "species composition", "species diversity", and "species richness". Darwin Core Terms listed under \code{basis_of_record} are listed and used by \code{convert_to_dwca()} to create a Darwin Core Archive of this ecocomDP data package.
        \item \strong{<intellectualRights>} Keeps intact the original intellectual rights license \code{source_id} was released under, or uses \href{https://creativecommons.org/publicdomain/zero/1.0/legalcode}{CCO} if missing.
        \item \strong{<taxonomicCoverage>} Appends to the taxonomic coverage element with data supplied in the ecocomDP taxon table.
        \item \strong{<contact>} Adds the ecocomDP creator as a point of contact.
        \item \strong{<methodStep>} Adds a note that this data package was created by the \code{script}, and adds provenance metadata noting that this is a derived dataset and describes where the \code{source_id} can be accessed.
        \item \strong{<dataTables>} Replaces the \code{source_id} table metadata with descriptions of the the ecocomDP tables.
        \item \strong{<otherEntity>} Adds \code{script} and \code{script_description}. otherEntities of \code{source_id} are removed.
        \item \strong{<annotations>} Adds boilerplate annotations describing the ecocomDP at the dataset, entity, and entity attribute levels.
    }
    
    Taxa listed in the taxon table, and resolved to one of the supported authority systems (i.e. \href{https://www.itis.gov/}{ITIS}, \href{http://www.marinespecies.org/}{WORMS}, or \href{https://gbif.org}{GBIF}), will have their full taxonomic hierarchy expanded, including any common names for each level.
}
\examples{
\dontrun{
# Create directory with ecocomDP tables for create_eml()
mypath <- paste0(tempdir(), "/data")
dir.create(mypath)
inpts <- c(ants_L1$edi.193.4$tables, path = mypath)
do.call(write_tables, inpts)
file.copy(system.file("extdata", "create_ecocomDP.R", package = "ecocomDP"), mypath)

dir(mypath)

# Describe, with annotations, what the source L0 dataset "is about"
dataset_annotations <- c(
  `species abundance` = "http://purl.dataone.org/odo/ECSO_00001688",
  Population = "http://purl.dataone.org/odo/ECSO_00000311",
  `level of ecological disturbance` = "http://purl.dataone.org/odo/ECSO_00002588",
  `type of ecological disturbance` = "http://purl.dataone.org/odo/ECSO_00002589")

# Add self as contact information incase questions arise
additional_contact <- data.frame(
  givenName = 'Colin',
  surName = 'Smith',
  organizationName = 'Environmental Data Initiative',
  electronicMailAddress = 'ecocomdp@gmail.com',
  stringsAsFactors = FALSE)

# Create EML
eml <- create_eml(
  path = mypath,
  source_id = "knb-lter-hfr.118.32",
  derived_id = "edi.193.4",
  is_about = dataset_annotations,
  script = "create_ecocomDP.R",
  script_description = "A function for converting knb-lter-hrf.118 to ecocomDP",
  contact = additional_contact,
  user_id = 'ecocomdp',
  user_domain = 'EDI',
  basis_of_record = "HumanObservation")

dir(mypath)
# View(eml)

# Clean up
unlink(mypath, recursive = TRUE)
}

}
