% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitModel.R
\name{fitModel}
\alias{fitModel}
\title{Fit drug interaction index model according to Harbron's framework}
\usage{
fitModel(data, mono = NULL, model = NULL, tauFormula = NULL,
  tauLog = FALSE, tauStart = 1 * (!tauLog), stage = 1,
  fixed = NULL, inactiveIn = 0, verbose = FALSE, ...)
}
\arguments{
\item{data}{A (long) data frame to fit the model to. Required columns are 
"d1", "d2" and "effect". Other variables are allowed and can be used in 
\code{tauFormula}.}

\item{mono}{An optional "MarginalFit" object, obtained from 
\code{\link{fitMarginals}}.}

\item{model}{A pre-defined model to use for the interaction index tau. One of 
"additive", "uniform", "linear1", "separate1", "linear2", "separate2", 
"separate12" or "zhao". See details.}

\item{tauFormula}{A formula to define the interaction index tau, using either 
'literal' (as in \code{nls}) or 'symbolic' (as in \code{lm}) specification.}

\item{tauLog}{Whether to fit the model using log-transformed tau parameters.
This is mostly useful for "separate"-type tau models for better convergence.
Note that if TRUE, tau cannot be negative, which may be not approriate
for some models, such as "linear1" and "linear2".
Note that this affects the coefficient names in the result 
("logtau1", "logtau2", ... instead of "tau1", "tau2", ...), so if 
\code{fixed} argument is used, this should be taken into account.}

\item{tauStart}{Vector of starting values for tau parameters, either of 
length 1 or of the same length as the total number of tau parameters.}

\item{stage}{Whether to run a 1-stage or 2-stage estimation.}

\item{fixed}{Constraints on monotherapy and/or tau parameters as a vector 
of the form `name = value`, if NULL (default), taken from \code{mono}. Note 
that the tau parameters should be named "tau1", "tau2", ... if 
\code{tauLog = FALSE} (default), or "logtau1", "logtau2", ... if 
\code{tauLog = TRUE}.}

\item{inactiveIn}{which compound is inactive (1 or 2), or 0 (default) when 
both compounds are active.}

\item{verbose}{Whether to show extra information useful for debugging.}

\item{...}{Further arguments passed to the \code{\link[minpack.lm]{nlsLM}} 
call. For example, \code{trace = TRUE} is useful to see the trace of the 
fitting process and may help identify issues with convergence.}
}
\value{
Fitted object of class "HarbronFit" which is an \code{nls}-like 
object with extra elements.
}
\description{
This is the main function to fit an interaction index model to 
drug combination data based on the Loewe additivity model. The interaction 
index can be specified in a flexible way as a function of doses and other 
variables.
}
\details{
There are different ways to specify a model for the interaction index tau:
\itemize{
\item Using one of the pre-defined models as specified in the \code{model} 
argument:
	\itemize{
		\item "additive", for additivity model,
		\item "uniform", one overall value for tau,
		\item "linear1", linear dependency on log10 dose of the first compound,
		\item "linear2", linear dependency on log10 dose of the second compound,
		\item "separate1", different tau for each dose of the first compound,
		\item "separate2", different tau for each dose of the second compound, 
		\item "separate12", different tau for each combination of doses of the two compounds,
		\item "zhao", quadratic response surface model following Zhao et al 2012.
	}
\item Using a literal or symbolic formula. Note that for the monotherapies, 
tau is assumed to be equal to 1. Therefore, continuous models may entail 
discontinuities in the interaction index when d1 and d2 approach 0.
}
}
\examples{
\donttest{
data("checkerboardData", package = "drugCombo")
data1 <- checkerboardData[checkerboardData$exp == 1, ]
mono1 <- fitMarginals(data1, fixed = c(b = 1))
# all three ways below are equivalent
fitLin1 <- fitModel(data = data1, mono = mono1, model = "linear1")
fitLin1b <- fitModel(data1, mono1, tauFormula = ~ log10(d1))
fitLin1c <- fitModel(data1, mono1, tauFormula = ~ tau1+tau2*log10(d1))
}
}
\author{
Maxim Nazarov
}
