#' @title Function \code{plot_graph}
#' @description Plot the dependency structure of your workflow.
#' IMPORTANT: you must be in the root directory of your project.
#' To save time for repeated plotting, this function is divided into
#' \code{\link{dataframes_graph}()} and \code{\link{render_graph}()}.
#' @export
#' @seealso \code{\link{build_graph}}
#' @param plan workflow plan data frame, same as for function 
#' \code{\link{make}()}.
#' @param targets names of targets to bulid, same as for function
#' \code{\link{make}()}.
#' @param envir environment to import from, same as for function
#' \code{\link{make}()}.
#' @param verbose logical, whether to output messages to the console.
#' @param jobs The \code{outdated()} function is called internally, 
#' and it needs to import objects and examine your 
#' input files to see what has been updated. This could take some time,
#' and parallel computing may be needed
#' to speed up the process. The \code{jobs} argument is number of parallel jobs 
#' to use for faster computation.
#' @param parallelism Choice of parallel backend to speed up the computation.
#' See \code{?parallelism_choices} for details. The Makefile option is not available
#' here. Drake will try to pick the best option for your system by default.
#' @param packages same as for \code{\link{make}()}.
#' @param prework same as for \code{\link{make}()}.
#' @param file Name of HTML file to save the graph.
#' If \code{NULL} or \code{character(0)},
#' no file is saved and the graph is rendered and displayed within R.
#' @param selfcontained logical, whether to save the \code{file} as a self-contained
#' HTML file (with external resources base64 encoded) or a file with
#' external resources placed in an adjacent directory. If \code{TRUE},
#' pandoc is required.
#' @param targets_only logical, whether to skip the imports and only show the 
#' targets in the workflow plan.
#' @param font_size numeric, font size of the node labels in the graph
#' @param layout name of an igraph layout to use, such as "layout_with_sugiyama"
#' or "layout_as_tree". Be careful with "layout_as_tree": the graph is a directed 
#' acyclic graph, but not necessarily a tree.
#' @param direction an argument to \code{visNetwork::visHierarchicalLayout()}
#' indicating the direction of the graph. Options include "LR", "RL", "DU", and "UD".
#' At the time of writing this, the letters must be capitalized, 
#' but this may not always be the case ;) in the future.
#' @param navigationButtons logical, whether to add navigation buttons with 
#' \code{visNetwork::visInteraction(navigationButtons = TRUE)}
#' @param hover logical, whether to show the command that generated the target
#' when you hover over a node with the mouse. For imports, the label does not
#' change with hovering.
#' @param config option internal runtime parameter list of 
#' \code{\link{make}(...)},
#' produced with \code{\link{config}()}.
#' Computing this
#' in advance could save time if you plan multiple calls to 
#' \code{outdated()}.
#' @param ... other arguments passed to \code{visNetwork::visNetwork()} to plot the graph.
#' @examples
#' \dontrun{
#' load_basic_example()
#' plot_graph(my_plan, width = "100%") # The width is passed to visNetwork().
#' make(my_plan)
#' plot_graph(my_plan) # The red nodes from before are now green.
#' }
plot_graph = function(plan, targets = drake::possible_targets(plan), 
  envir = parent.frame(), verbose = TRUE, jobs = 1, 
  parallelism = drake::default_parallelism(), 
  packages = (.packages()), prework = character(0),
  file = character(0), selfcontained = FALSE, targets_only = FALSE, config = NULL,
  font_size = 20, layout = "layout_with_sugiyama", direction = "LR",
  navigationButtons = TRUE, hover = TRUE, ...){
  
  force(envir)
  raw_graph = dataframes_graph(plan = plan, targets = targets, 
     envir = envir, verbose = verbose, jobs = jobs, parallelism = parallelism,
     packages = packages, prework = prework,targets_only = targets_only,
     config = config, font_size = font_size)
  render_graph(raw_graph, file = file, selfcontained = selfcontained,
     layout = layout, direction = direction,
     navigationButtons = navigationButtons, hover = hover, ...)
}

#' @title Function \code{render_graph}
#' @description render a graph from the data frames 
#' generated by \code{\link{dataframes_graph}()}
#' @export
#' @param graph list of data frames generated by \code{\link{dataframes_graph}()}.
#' There should be 3 data frames: \code{nodes}, \code{edges}, 
#' and \code{legend_nodes}.
#' @param file Name of HTML file to save the graph. If \code{NULL} or \code{character(0)},
#' no file is saved and the graph is rendered and displayed within R.
#' @param layout name of an igraph layout to use, such as "layout_with_sugiyama"
#' or "layout_as_tree". Be careful with "layout_as_tree": the graph is a directed 
#' acyclic graph, but not necessarily a tree.
#' @param selfcontained logical, whether to save the \code{file} as a self-contained
#' HTML file (with external resources base64 encoded) or a file with
#' external resources placed in an adjacent directory. If \code{TRUE},
#' pandoc is required.
#' @param direction an argument to \code{visNetwork::visHierarchicalLayout()}
#' indicating the direction of the graph. Options include "LR", "RL", "DU", and "UD".
#' At the time of writing this, the letters must be capitalized, 
#' but this may not always be the case ;) in the future.
#' @param navigationButtons logical, whether to add navigation buttons with 
#' \code{visNetwork::visInteraction(navigationButtons = TRUE)}
#' @param hover logical, whether to show the command that generated the target
#' when you hover over a node with the mouse. For imports, the label does not
#' change with hovering.
#' @param ... arguments passed to \code{visNetwork()}.
#' @examples
#' \dontrun{
#' load_basic_example()
#' graph = dataframes_graph(my_plan)
#' render_graph(graph, width = "100%") # The width is passed to visNetwork().
#' }
render_graph = function(graph, file = character(0), layout = "layout_with_sugiyama", direction = "LR",
  navigationButtons = TRUE, hover = TRUE, selfcontained = FALSE, ...){
  out = visNetwork(nodes = graph$nodes, edges = graph$edges, ...) %>%
    visLegend(useGroups = FALSE, addNodes = graph$legend_nodes) %>% 
    visHierarchicalLayout(direction = direction) %>%
    visIgraphLayout(physics = FALSE, randomSeed = 2017, # increases performance
      layout = layout) 
  if(navigationButtons)
    out = visInteraction(out, navigationButtons = TRUE)
  if(hover)
    out = with_hover(out)
  if(length(file)){
    visSave(graph = out, file = file, selfcontained = selfcontained)
    return(invisible())
  }
  out
}

with_hover = function(x){
  visInteraction(x, hover = T) %>%
    visEvents(hoverNode  = "function(e){
              var label_info = this.body.data.nodes.get({
              fields: ['label', 'hover_label'],
              filter: function (item) {
              return item.id === e.node
              },
              returnType :'Array'
              });
              this.body.data.nodes.update({id: e.node, label : label_info[0].hover_label, hover_label : label_info[0].label});
  }") %>% 
  visEvents(blurNode  = "function(e){
            var label_info = this.body.data.nodes.get({
            fields: ['label', 'hover_label'],
            filter: function (item) {
            return item.id === e.node
            },
            returnType :'Array'
            });
            this.body.data.nodes.update({id: e.node, label : label_info[0].hover_label, hover_label : label_info[0].label});
  }")
}
