% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/space.maker.R
\name{space.maker}
\alias{space.maker}
\title{Creating multidimensional spaces}
\usage{
space.maker(elements, dimensions, distribution, arguments = NULL,
  cor.matrix = NULL, scree = NULL)
}
\arguments{
\item{elements}{An \code{numeric} value.}

\item{dimensions}{An \code{numeric} value smaller than \code{elements}.}

\item{distribution}{One or more \code{functions} to determine the distribution of the \code{elements} along each \code{dimension}. The function must have a single input: \code{elements}.}

\item{arguments}{Optional \code{list} of arguments to be passed to the distributions functions in the order they appear (\code{default = NULL}, see details).}

\item{cor.matrix}{An optional correlation \code{matrix} of size \code{dimensions * dimensions} (\code{default = NULL}, see details).}

\item{scree}{An optional proportional \code{numeric} vector for approximating the \code{dimensions} variance (\code{default = NULL}, see details).}
}
\description{
Creates a multidimensional space with a given number of elements and dimensions
}
\details{
When passing additional arguments to different distributions, these must be given as a \code{list} to each function in the order they appear.
For example if \code{distribution = c(runif, rnorm, rgamma)} and one wants the distributions to be \code{runif(elements, min = 1, max = 10)}, \code{rnorm(elements, mean = 8)} and \code{rgamma(elements, shape = 1, log = TRUE)}, the additional arguments should be passed as
\code{c(list(min = 1, max = 10), list(mean = 8), list(shape = 1, log = TRUE)}. If no arguments have to be passed to a certain function, it can be left as \code{NULL} (e.g. \code{c(list(min = 1, max = 10), list(NULL), list(shape = 1, log = TRUE)}).

The \code{cor.matrix} argument should be a correlation matrix between the dimensions.
If not \code{NULL}, the multidimensional space is multiplied by the the Choleski decomposition (\code{\link[base]{chol}}) of the correlation matrix.
The \code{scree} argument is simply a value multiplier for each dimension to adjust their variance to approximate the \code{scree} one.
}
\examples{
## A square space
plot(space.maker(5000, 2, runif), pch = 20)

## A circular space
plot(space.maker(5000, 2, rnorm), pch = 20)

## A 2-dimensional cylindrical space
plot(space.maker(5000, 2, c(rnorm, runif)), pch = 20)

## A 4-dimensional space with different distributions
space.maker(5, 4, c(runif, runif, rnorm, rgamma),
     arguments = list(list(min = 1, max = 10), list(min = 1, max = 2),
     list(mean = 8), list(shape = 1)))

## A 3-dimensional correlated space
cor_matrix <- matrix(cbind(1, 0.8 ,0.2, 0.8, 1, 0.7, 0.2, 0.7, 1), nrow = 3)
space <- space.maker(10000, 3, rnorm, cor.matrix = cor_matrix)
round(cor(space), 1) ; cor_matrix ## Both should be really similar matrices

## A 3-dimensional space with a priori approximated variance for each dimension
space <- space.maker(10000, 3, rnorm, scree = c(0.6, 0.3, 0.1))
## The resulting screeplot
barplot(apply(space, 2, var))

\dontrun{
require(scatterplot3d)
## A cube space
scatterplot3d(space.maker(5000, 3, runif), pch = 20)

## A plane space
scatterplot3d(space.maker(5000, 3, c(runif, runif, runif),
     arguments = list(list(min = 0, max = 0), NULL, NULL)), pch = 20)

## A sphere space
scatterplot3d(space.maker(5000, 3, rnorm), pch = 20)

## A 3D cylindrical space
scatterplot3d(space.maker(5000, 3, c(rnorm, rnorm, runif)), pch = 20)

## Generating a doughnut space
doughnut <- space.maker(5000, 3, c(rnorm, random.circle),
     arguments = list(list(mean = 0), list(runif, inner = 0.5, outer = 1)))
## Reodering the axis for projecting the dougnut in 2D
scatterplot3d(doughnut[,c(2,1,3)], pch = 20)
}

}
\seealso{
\code{\link{null.test}}, \code{\link{test.dispRity}}.
}
\author{
Thomas Guillerme
}
