# Author: Robert J. Hijmans
# contact: r.hijmans@gmail.com
# Date : Febrary 2010
# Version 0.1
# Licence GPL v3




.avgDist <- function(xy, lonlat, r=6378137) {
	xy <- unique(xy)
	xy <- as.matrix(xy[,1:2])
	d <- matrix(nrow=nrow(xy), ncol=nrow(xy))
	if (lonlat) {
		for (i in 1:nrow(xy)) { 
			d[,i] <- pointDistance(xy[i,], xy, longlat=TRUE, r=r)
		}
	} else {
		for (i in 1:nrow(xy)) { 
			d[,i] <- pointDistance(xy[i,], xy, longlat=FALSE, r=r)
		}
	}
	median(apply(d, 1, function(x)quantile(x, 0.1)))
}


.generateCircles <- function(xy, d, n=360, lonlat, r=6378137 ) {
	if (length(d)==1) {
		d <- rep(d, nrow(xy))
	} else if (length(d) != nrow(xy)) {
		# recycling
		dd <- vector(length=nrow(xy))
		dd[] <- d
		d <- dd
	}
	
	xy <- as.matrix(xy[,1:2])
	n <- max(4, round(n))
	toRad <- pi/180
	brng <- 1:n * 360/n
	brng <- brng * toRad
	if (lonlat) { 
		xy = xy * toRad 
	}
	pols <- list()
	for (i in 1:nrow(xy)) {
		p <- xy[i, ]
		if (lonlat) {
			lon1 <- p[1]
			lat1 <- p[2]
			lat2 <- asin(sin(lat1) * cos(d[i]/r) + cos(lat1) * sin(d[i]/r) * cos(brng))
			lon2 <- lon1 + atan2(sin(brng) * sin(d[i]/r) * cos(lat1), cos(d[i]/r) - sin(lat1) * sin(lat2))
			lon2 <- (lon2 + pi)%%(2 * pi) - pi
			lon2[is.nan(lon2)] <- NA
			lat2[is.nan(lat2)] <- NA
			res <- cbind(lon2, lat2)/toRad
			colnames(res) <- c("lon", "lat")
		} else {
			x2 <- p[1] + d[i] * cos(brng)
			y2 <- p[2] + d[i] * sin(brng)
			res <- cbind(x2, y2)
			colnames(res) <- c("x", "y")
		}
		res <- rbind(res, res[1,])
		pols <- c(pols, Polygons(list(Polygon( res )), i))		
	}

	pols <- SpatialPolygons(pols)
	if (lonlat) {
		projection(pols) <- CRS('+proj=longlat')
	}
	return( pols )
}


setClass('CirclesRange',
	contains = 'DistModel',
	representation (
		polygons='SpatialPolygons'
	),	
	prototype (	
	),
	validity = function(object)	{
		return(TRUE)
	}
)


if (!isGeneric("circles")) {
	setGeneric("circles", function(p, ...)
		standardGeneric("circles"))
}	


setMethod('circles', signature(p='data.frame'), 
	function(p, d, lonlat, ...) {
		ci <- new('CirclesRange')
		ci@presence <- p
		if (missing(lonlat)) {
			stop("you must provide a 'lonlat' argument")
		}
		if (missing(d)) {
			d <- .avgDist(p, lonlat=lonlat, r=r) / 2
		}
		ci@polygons <- .generateCircles(p, d=d, lonlat=lonlat)
		return(ci)
	}
)


setMethod('circles', signature(p='matrix'), 
	function(p, ...) {
		circles(data.frame(p), ...)
	}
)

setMethod('circles', signature(p='SpatialPoints'), 
	function(p, d, lonlat, ...) {
		if (missing(lonlat)) lonlat <- isLonLat(p)
		p <- coordinates(p)
		if (missing(d)) { d <- .avgDist(p, lonlat=lonlat, r=r) / 2	}
		circles(p, d=d, lonlat=lonlat, ...)
	}
)


setMethod("plot", signature(x='CirclesRange', y='missing'), 
	function(x, ...) {
		plot(x@polygons, ...)
	}
)


