% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/difNLR.R
\name{difNLR}
\alias{AIC.difNLR}
\alias{BIC.difNLR}
\alias{coef.difNLR}
\alias{difNLR}
\alias{fitted.difNLR}
\alias{logLik.difNLR}
\alias{plot.difNLR}
\alias{predict.difNLR}
\alias{print.difNLR}
\title{Performs DIF detection using Non-Linear Regression method.}
\usage{
difNLR(Data, group, focal.name, model, type = "both",
test = "LR", alpha = 0.05, p.adjust.method = "none", start)

\method{print}{difNLR}(x, ...)

\method{plot}{difNLR}(x, plot.type = "cc", item = "all",
  col = c("dodgerblue2", "goldenrod2"), shape = 21, size = 0.8,
  linetype = c(2, 1), title, ...)

\method{fitted}{difNLR}(object, item = "all", ...)

\method{predict}{difNLR}(object, item = "all", score, group, ...)

\method{coef}{difNLR}(object, ...)

\method{logLik}{difNLR}(object, ...)

\method{AIC}{difNLR}(object, ...)

\method{BIC}{difNLR}(object, ...)
}
\arguments{
\item{Data}{numeric: either the scored data matrix only, or the scored data
matrix plus the vector of group. See \strong{Details}.}

\item{group}{numeric or character: either the binary vector of group membership or
the column indicator of group membership. See \strong{Details}.}

\item{focal.name}{numeric or character: indicates the level of \code{group} which corresponds to
focal group}

\item{model}{character: generalized logistic regression model to be fitted. See \strong{Details}.}

\item{type}{character: type of DIF to be tested (either "both" (default), "udif", or "nudif").
See \strong{Details}.}

\item{test}{character: test to be performed for DIF detection (either "LR" (default), or "F").
See \strong{Details}.}

\item{alpha}{numeric: significance level (default is 0.05).}

\item{p.adjust.method}{character: method for multiple comparison correction. See \strong{Details}.}

\item{start}{numeric: matrix with n rows (where n is the number of items) and 8 columns
containing initial item parameters estimates. See \strong{Details}.}

\item{x}{an object of 'difNLR' class}

\item{...}{other generic parameters for \code{print}, \code{plot}, \code{fitted},
\code{predict} or \code{coef} functions.}

\item{plot.type}{character: type of plot to be plotted (either "cc" for characteristic curve
(default), or "stat" for test statistics). See \strong{Details}.}

\item{item}{either character ("all"), or numeric vector, or single number corresponding to column indicators. See \strong{Details}.}

\item{col}{character: single value, or vector of two values representing colors for plot.}

\item{shape}{integer: shape parameter for plot.}

\item{size}{numeric: single number, or vector of two numbers representing line width in plot.}

\item{linetype}{numeric: single number, or vector of two numbers representing line type in plot for reference and focal group.}

\item{title}{string: title of plot.}

\item{object}{an object of 'difNLR' class}

\item{score}{numeric: standardized total score of subject.}
}
\value{
A list of class 'difNLR' with the following arguments:
\describe{
  \item{\code{Sval}}{the values of likelihood ratio test statistics.}
  \item{\code{nlrPAR}}{the estimates of final model.}
  \item{\code{nlrSE}}{the standard errors of estimates of final model.}
  \item{\code{parM0}}{the estimates of null model.}
  \item{\code{seM0}}{the standard errors of estimates of null model.}
  \item{\code{covM0}}{the covariance matrices of estimates of null model.}
  \item{\code{parM1}}{the estimates of alternative model.}
  \item{\code{seM1}}{the standard errors of estimates of alternative model.}
  \item{\code{covM1}}{the covariance matrices of estimates of alternative model.}
  \item{\code{alpha}}{numeric: significance level.}
  \item{\code{DIFitems}}{either the column indicators of the items which were detected as DIF, or
  \code{"No DIF item detected"}.}
  \item{\code{model}}{fitted model.}
  \item{\code{type}}{character: type of DIF that was tested.}
  \item{\code{p.adjust.method}}{character: method for multiple comparison correction which was applied.}
  \item{\code{pval}}{the p-values by likelihood ratio test.}
  \item{\code{adj.pval}}{the adjusted p-values by likelihood ratio test using \code{p.adjust.method}.}
  \item{\code{df}}{the degress of freedom of likelihood ratio test.}
  \item{\code{group}}{the vector of group membership.}
  \item{\code{Data}}{the data matrix.}
  \item{\code{conv.fail}}{numeric: number of convergence issues.}
  \item{\code{conv.fail.which}}{the indicators of the items which did not converge.}
  \item{\code{llM0}}{log-likelihood of null model.}
  \item{\code{llM1}}{log-likelihood of alternative model.}
  \item{\code{AICM0}}{AIC of null model.}
  \item{\code{AICM1}}{AIC of alternative model.}
  \item{\code{BICM0}}{BIC of null model.}
  \item{\code{BICM1}}{BIC of alternative model.}
}
}
\description{
Performs DIF detection procedure based on Non-Linear Regression model (generalized logistic
regression) and either likelihood-ratio or F test of submodel.
}
\details{
DIF detection procedure based on Non-Linear Regression is the extension of Logistic Regression
procedure (Swaminathan and Rogers, 1990).

The \code{Data} is a matrix whose rows represents scored examinee answers ("1" correct,
"0" incorrect) and columns correspond to the items. In addition, \code{Data} can hold
the vector of group membership. If so, \code{group} is a column indicator of \code{Data}.
Otherwise, \code{group} must be either a vector of the same length as \code{nrow(Data)}.

The options of \code{model} are as follows: \code{Rasch} for one-parameter logistic model with
discrimination parameter fixed on value 1 for both groups, \code{1PL} for one-parameter logistic
model with discrimination parameter fixed for both groups, \code{2PL} for logistic regression model,
\code{3PLcg} for three-parameter logistic regression model with fixed guessing for both groups,
\code{3PLdg} for three-parameter logistic regression model with fixed inattention for both groups, or
\code{4PLcgdg} for four-parameter logistic regression model with fixed guessing and inattention
parameter for both groups.

The \code{type} corresponds to type of DIF to be tested. Possible values are \code{"both"} to
detect any DIF (uniform and/or non-uniform), \code{"udif"} to detect only uniform DIF or
\code{"nudif"} to detect only non-uniform DIF.

The \code{p.adjust.method} is a character for \code{p.adjust} function from the \code{stats}
package. Possible values are \code{"holm"}, \code{"hochberg"}, \code{"hommel"},
\code{"bonferroni"}, \code{"BH"}, \code{"BY"}, \code{"fdr"}, \code{"none"}.

The \code{start} is a matrix with a number of rows equal to number of items and with 8 columns.
First 4 columns represent parameters (a, b, c, d) of generalized logistic regression model
for reference group. Last 4 columns represent differences of parameters (aDif, bDif, cDif, dDif)
of generalized logistic regression model between reference and focal group. If not specified, starting
values are calculated with \code{startNLR} function.

The output of the difNLR is displayed by the \code{print.difNLR} function.

Two types of plots are available. The first one is obtained by setting \code{plot.type = "cc"}
(default). The characteristic curve for item specified in \code{item} option is plotted. For default
option \code{"all"} of item, characteristic curves of all converged items are plotted. The drawn
curves represent best model.
The second plot is obtained by setting \code{plot.type = "stat"}. The  test statistics
(either LR-test, or F-test, depends on argument \code{test}) are displayed on the Y axis,
for each coverged item. The detection threshold is displayed by a horizontal line and items
detected as DIF are printed with the red color. Only parameters \code{size} and \code{title}
are used.

Fitted values are extracted by the \code{fitted.difNLR} function for item(s) specified in
\code{item} argument.

Predicted values are produced by the \code{predict.difNLR} function for item(s) specified in
\code{item} argument. \code{score} represents standardized total score of new subject and
\code{group} argument represents group membership of new subject.

Missing values are allowed but discarded for item estimation. They must be coded as
\code{NA} for both, \code{data} and \code{group} parameters.
}
\examples{
\dontrun{
# loading data based on GMAT
data(GMAT)

Data  <- GMAT[, colnames(GMAT) != "group"]
group <- GMAT[, "group"]

# Testing both DIF effects using likelihood-ratio test and
# 3PL model with fixed guessing for groups
x <- difNLR(Data, group, focal.name = 1, model = "3PLcg")

# Testing both DIF effects using F test and
# 3PL model with fixed guessing for groups
difNLR(Data, group, focal.name = 1, model = "3PLcg", test = "F")

# Testing both DIF effects using LR test,
# 3PL model with fixed guessing for groups
# and Benjamini-Hochberg correction
difNLR(Data, group, focal.name = 1, model = "3PLcg", p.adjust.method = "BH")

# Testing both DIF effects using Rasch model
difNLR(Data, group, focal.name = 1, model = "Rasch")

# Testing both DIF effects using 2PL model
difNLR(Data, group, focal.name = 1, model = "2PL")

# Testing uniform DIF effects
difNLR(Data, group, focal.name = 1, model = "2PL", type = "udif")
# Testing non-uniform DIF effects
difNLR(Data, group, focal.name = 1, model = "2PL", type = "nudif")

# Graphical devices
plot(x)
plot(x, item = x$DIFitems)
plot(x, plot.type = "stat")

# Fitted values
fitted(x)
fitted(x, item = 1)
# Predicted values
predict(x)
predict(x, item = 1)

# Predicted values for new subjects
predict(x, item = 1, score = 0, group = 1)
predict(x, item = 1, score = 0, group = 0)

# AIC, BIC, logLik
AIC(x)
BIC(x)
logLik(x)
}
}
\author{
Adela Drabinova \cr
Institute of Computer Science, The Czech Academy of Sciences \cr
Faculty of Mathematics and Physics, Charles University \cr
adela.drabinova@gmail.com \cr

Patricia Martinkova \cr
Institute of Computer Science, The Czech Academy of Sciences \cr
martinkova@cs.cas.cz \cr

Karel Zvara \cr
Faculty of Mathematics and Physics, Charles University \cr
}
\references{
Drabinova, A. and Martinkova P. (2016). Detection of Differenctial Item Functioning Based on Non-Linear Regression, Technical Report, V-1229, \url{http://hdl.handle.net/11104/0259498}.

Swaminathan, H. and Rogers, H. J. (1990). Detecting Differential Item Functioning Using Logistic Regression Procedures. Journal of Educational Measurement, 27, 361-370.
}
\seealso{
\code{\link[stats]{p.adjust}}
}
\keyword{DIF}

