#' Request generation of an transferrable model file for use in an on-premise DataRobot standalone
#' prediction environment.
#' This function can only be used if model export is enabled, and will only be useful
#' if you have an on-premise environment in which to import it.
#' This function does not download the exported file. Use DownloadTransferrableModel for that.
#' @inheritParams DeleteProject
#' @param modelId numeric. Unique alphanumeric identifier for the model of interest.
#' @return jobId
#' @examples
#' \dontrun{
#'   jobId < -RequestTransferrrableModel(projectObject, modelId)
#'   WaitForJobToComplete(projectObject, jobId)
#'   DownloadTransferrableModel(projectObject, modelId, file)
#' }
#' @export
RequestTransferrableModel <- function(project, modelId) {
  projectId <- ValidateProject(project)
  routeString <- "modelExports"
  body <- list(projectId = projectId, modelId = modelId)
  rawResponse <- DataRobotPOST(routeString, addUrl = TRUE, body = body, returnRawResponse = TRUE)
  routeString <- UrlJoin("projects", projectId, "jobs", JobIdFromResponse(rawResponse))
  jobsResponse <- DataRobotGET(routeString, addUrl = TRUE, query = NULL, simplifyDataFrame = FALSE)
  return(jobsResponse$id)
}


#' Download an transferrable model file for use in an on-premise DataRobot standalone
#' prediction environment.
#'
#' This function can only be used if model export is enabled, and will only be useful
#' if you have an on-premise environment in which to import it.
#' @inheritParams DeleteProject
#' @param modelId numeric. Unique alphanumeric identifier for the model of interest.
#' @param modelFile character. File name to be use for tranferrable model
#' @return NULL
#' @examples
#' \dontrun{
#'   projectId <- "59a5af20c80891534e3c2bde"
#'   modelId <- "5996f820af07fc605e81ead4"
#'   DownloadTransferrableModel(projectId, modelId, "model.drmodel")
#' }
#' @export
DownloadTransferrableModel <- function(project, modelId, modelFile) {
  projectId <- ValidateProject(project)
  routeString <- UrlJoin("projects", projectId, "models", modelId, "export")
  response <- DataRobotGET(routeString, addUrl = TRUE, query = NULL, as = "text")
  writeBin(response, modelFile)
}


#' Import a previously exported model for predictions.
#' @param  modelFile character. Path to binary transeffable model file.
#' @param maxWait integer. Specifies how many seconds to wait for upload to finish.
#' @return A list describing uploaded transferrable model
#' with the following components:
#' \itemize{
#'   \item note. Character string Manually added node about this imported model.
#'   \item datasetName. Character string Filename of the dataset used to create the project the
#'     model belonged to.
#'   \item modelName. Character string Model type describing the model generated by DataRobot.
#'   \item displayName. Character string Manually specified human-readable name of the imported
#'     model.
#'   \item target. Character string The target of the project the model belonged to prior to export.
#'   \item projectName. Character string Name of the project the model belonged to prior to export.
#'   \item importedByUsername. Character string Username of the user who imported the model.
#'   \item importedAt. Character string The time the model was imported.
#'   \item version. Numeric Project version of the project the model belonged to.
#'   \item projectId. Character id of the project the model belonged to prior to export.
#'   \item featurelistName. Character string Name of the featurelist used to train the model.
#'   \item createdByUsername. Character string Username of the user who created the model prior to
#'     export.
#'   \item importedById. Character string id of the user who imported the model.
#'   \item id. Character string id of the import.
#'   \item createdById. Character string id of the user who created the model prior to export.
#'   \item modelId. Character string original id of the model prior to export.
#'   \item originUrl. Character string URL.
#' }
#' @examples
#' \dontrun{
#'   UploadTransferrableModel("model.drmodel")
#' }
#' @export
UploadTransferrableModel <- function(modelFile, maxWait = 600) {
  dataList <- list(file = httr::upload_file(modelFile), name = basename(modelFile))
  routeString <- "importedModels/"
  rawReturn <- DataRobotPOST(routeString, addUrl = TRUE, body = dataList,
                             returnRawResponse = TRUE, timeout = maxWait)
  modelInfo <- WaitForAsyncReturn(httr::headers(rawReturn)$location,
                            addUrl = FALSE,
                            maxWait = maxWait,
                            failureStatuses = "ERROR")
  return(as.dataRobotTransferrableModel(modelInfo))
}


#' Retrieve imported model info using import id
#' @param importId character. Id of the import.
#' @return A list describing uploaded transferrable model
#' with the following components:
#' \itemize{
#'   \item note. Character string Manually added node about this imported model.
#'   \item datasetName. Character string Filename of the dataset used to create the project the
#'     model belonged to.
#'   \item modelName. Character string Model type describing the model generated by DataRobot.
#'   \item displayName. Character string Manually specified human-readable name of the imported
#'     model.
#'   \item target. Character string The target of the project the model belonged to prior to export.
#'   \item projectName. Character string Name of the project the model belonged to prior to export.
#'   \item importedByUsername. Character string Username of the user who imported the model.
#'   \item importedAt. Character string The time the model was imported.
#'   \item version. Numeric Project version of the project the model belonged to.
#'   \item projectId. Character id of the project the model belonged to prior to export.
#'   \item featurelistName. Character string Name of the featurelist used to train the model.
#'   \item createdByUsername. Character string Username of the user who created the model prior to
#'     export.
#'   \item importedById. Character string id of the user who imported the model.
#'   \item id. Character string id of the import.
#'   \item createdById. Character string id of the user who created the model prior to export.
#'   \item modelId. Character string original id of the model prior to export.
#'   \item originUrl. Character string URL.
#' }
#' @examples
#' \dontrun{
#'   id <- UploadTransferrableModel("model.drmodel")
#'   GetTransferrableModel(id)
#' }
#' @export
GetTransferrableModel <- function(importId) {
  routeString <- UrlJoin("importedModels", importId)
  modelInfo <- DataRobotGET(routeString, addUrl = TRUE)
  return(as.dataRobotTransferrableModel(modelInfo))
}


#' Retrieve information about all imported models
#' This function returns a data.frame that describes all imported models
#' @param limit integer. The number of records to return. The server will use a (possibly finite)
#'   default if not specified.
#' @param offset integer. The number of records to skip.
#' @return A data.frame describing uploaded transferrable model
#' with the following components:
#' \itemize{
#'   \item note. Character string Manually added node about this imported model.
#'   \item datasetName. Character string Filename of the dataset used to create the project the
#'     model belonged to.
#'   \item modelName. Character string Model type describing the model generated by DataRobot.
#'   \item displayName. Character string Manually specified human-readable name of the imported
#'     model.
#'   \item target. Character string The target of the project the model belonged to prior to export.
#'   \item projectName. Character string Name of the project the model belonged to prior to export.
#'   \item importedByUsername. Character string Username of the user who imported the model.
#'   \item importedAt. Character string The time the model was imported.
#'   \item version. Numeric Project version of the project the model belonged to.
#'   \item projectId. Character id of the project the model belonged to prior to export.
#'   \item featurelistName. Character string Name of the featurelist used to train the model.
#'   \item createdByUsername. Character string Username of the user who created the model prior to
#'     export.
#'   \item importedById. Character string id of the user who imported the model.
#'   \item id. Character string id of the import.
#'   \item createdById. Character string id of the user who created the model prior to export.
#'   \item modelId. Character string original id of the model prior to export.
#'   \item originUrl. Character string URL.
#' }
#' @examples
#' \dontrun{
#'   ListTransferrableModels()
#' }
#' @export
ListTransferrableModels <- function(limit = NULL, offset = NULL) {
  routeString <- "importedModels/"
  body <- list(limit = limit, offset = offset)
  modelsInfo <- DataRobotGET(routeString, addUrl = TRUE, body = body)
  return(as.dataRobotTransferrableModel(modelsInfo$data))
}


#' Update the display name or note for an imported model.
#'
#' @inheritParams GetTransferrableModel
#' @param displayName character. The new display name.
#' @param note character. The new note.
#' @return A list describing uploaded transferrable model
#' with the following components:
#' \itemize{
#'   \item note. Character string Manually added node about this imported model.
#'   \item datasetName. Character string Filename of the dataset used to create the project the
#'     model belonged to.
#'   \item modelName. Character string Model type describing the model generated by DataRobot.
#'   \item displayName. Character string Manually specified human-readable name of the imported
#'     model.
#'   \item target. Character string The target of the project the model belonged to prior to export.
#'   \item projectName. Character string Name of the project the model belonged to prior to export.
#'   \item importedByUsername. Character string Username of the user who imported the model.
#'   \item importedAt. Character string The time the model was imported.
#'   \item version. Numeric Project version of the project the model belonged to.
#'   \item projectId. Character id of the project the model belonged to prior to export.
#'   \item featurelistName. Character string Name of the featurelist used to train the model.
#'   \item createdByUsername. Character string Username of the user who created the model prior to
#'     export.
#'   \item importedById. Character string id of the user who imported the model.
#'   \item id. Character string id of the import.
#'   \item createdById. Character string id of the user who created the model prior to export.
#'   \item modelId. Character string original id of the model prior to export.
#'   \item originUrl. Character string URL.
#' }
#' @examples
#' \dontrun{
#'   id <- UploadTransferrableModel("model.drmodel")
#'   UpdateTransferrableModel(id, displayName = "NewName", note = "This is my note.")
#' }
#' @export
UpdateTransferrableModel <- function(importId, displayName = NULL, note = NULL) {
  if (!is.null(displayName) || !is.null(note)) {
    routeString <- UrlJoin("importedModels",  importId)
    body <- list(displayName = displayName, note = note)
    DataRobotPATCH(routeString, addUrl = TRUE, body = body)
  }
  return(GetTransferrableModel(importId))
}


as.dataRobotTransferrableModel <- function(inList) {
  elements <- c("note",
                "datasetName",
                "modelName",
                "displayName",
                "target",
                "projectName",
                "importedByUsername",
                "importedAt",
                "version",
                "projectId",
                "featurelistName",
                "createdByUsername",
                "importedById",
                "id",
                "createdById",
                "modelId",
                "originUrl")
  return(ApplySchema(inList, elements))
}


#' Delete this imported model.
#'
#' @inheritParams GetTransferrableModel
#' @return NULL
#' @examples
#' \dontrun{
#'   id <- UploadTransferrableModel("model.drmodel")
#'   DeleteTransferrableModel(id)
#' }
#' @export
DeleteTransferrableModel <- function(importId) {
  model <- GetTransferrableModel(importId)
  print(model)
  routeString <- UrlJoin("importedModels",  importId)
  response <- DataRobotDELETE(routeString, addUrl = TRUE)
  message(paste("Model", model$modelName,
                "(import Id = ", model$id, ") deleted from prediction server"))
  return(invisible(NULL))
}
