#' Get Session (Slot) Data From A Databrary Volume By Session Name.
#' 
#' @param session_name A string. The name of the target session. Defaults to "Advisory
#' Board Meeting", the name of several of the sessions in the public Volume 1.
#' @param vol_id An integer indicating the volume identifier. Default is 1.
#' @param vb A logical value. Show verbose feedback. Default is FALSE.
#' @param rq An httr2 request, such as that generated by `make_default_request()`.
#' 
#' @returns One or more JSON blobs (as lists) whose session name(s) match 
#' `name` in the given volume.
#' 
#' @examples
#' \donttest{
#' \dontrun{
#' # Returns multiple JSON blobs with the name "Advisory Board Meeting" in Volume 1.
#' get_session_by_name() 
#' 
#' get_session_by_name(name = 'Top-level materials')
#' }
#' }
#' @export
get_session_by_name <-
  function(session_name = "Advisory Board Meeting",
           vol_id = 1,
           vb = FALSE,
           rq = NULL) {
    
    assertthat::is.string(session_name)
    assertthat::assert_that(length(session_name) == 1)
    
    assertthat::assert_that(is.numeric(vol_id))
    assertthat::assert_that(vol_id > 0)
    assertthat::assert_that(length(vol_id) == 1)
    
    assertthat::assert_that(is.logical(vb))
    assertthat::assert_that(length(vb) == 1)
    
    assertthat::assert_that(is.null(rq) |
                              ("httr2_request" %in% class(rq)))
    
    if (is.null(rq)) {
      if (vb) {
        message("NULL request object. Will generate default.")
        message("\nNot logged in. Only public information will be returned.")  
      }
      rq <- databraryr::make_default_request()
    }
    
    #--------------------------------------------------------------------------
    extract_session_metadata <- function(volume_json) {
      assertthat::assert_that(is.list(volume_json))
      
      extract_single_session <- function(i, sessions) {
        this_session <- sessions$value[[i]]
        tibble::tibble(id = this_session$id, top = this_session$top, name = this_session$name)
      }
      
      these_sessions <- tibble::enframe(volume_json$containers)
      n_sessions <- dim(these_sessions)[1]
      purrr::map(1:n_sessions, extract_single_session, these_sessions) %>%
        purrr::list_rbind()
    }
    #--------------------------------------------------------------------------
  
    volume_json <- NULL
    volume_json <- get_volume_by_id(vol_id, vb, rq)
    session_metadata <- extract_session_metadata(volume_json) 
    
    name <- NULL
    name_matches <- dplyr::filter(session_metadata, name == session_name)
    
    if (is.null(name_matches)) {
      message("No matches")
      return(NULL)
    }
    if (dim(name_matches)[1] == 0) {
      message("Empty array")
      return(NULL)
    }
    if (dim(name_matches)[1] > 1) {
      message("\nMultiple sessions with name '", session_name, "'.")
    }
    purrr::map(name_matches$id, get_session_by_id, vol_id, rq = rq)
  }