% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/readNWISdata.r
\name{readNWISdata}
\alias{readNWISdata}
\title{General Data Import from NWIS}
\usage{
readNWISdata(..., asDateTime = TRUE, convertType = TRUE, tz = "UTC")
}
\arguments{
\item{\dots}{see \url{https://waterservices.usgs.gov/rest/Site-Service.html} for a complete list of options.  A 
list of arguments can also be supplied. One important argument to include is 'service'. Possible values are "iv" 
(for instantaneous), "iv_recent" (for instantaneous values within the last 120 days),
"dv" (for daily values), "gwlevels" (for groundwater levels),
"site" (for site service), "qw" (water-quality),"measurement", and "stat" (for 
statistics service). Note: "qw" and "measurement" calls go to: 
\url{https://nwis.waterdata.usgs.gov/usa/nwis} for data requests, and use different call requests schemes.
The statistics service has a limited selection of arguments (see \url{https://waterservices.usgs.gov/rest/Statistics-Service-Test-Tool.html}).}

\item{asDateTime}{logical, if \code{TRUE} returns date and time as POSIXct, if \code{FALSE}, Date}

\item{convertType}{logical, defaults to \code{TRUE}. If \code{TRUE}, the function will convert the data to dates, datetimes,
numerics based on a standard algorithm. If false, everything is returned as a character}

\item{tz}{character to set timezone attribute of dateTime. Default is "UTC", and converts the 
date times to UTC, properly accounting for daylight savings times based on the data's provided tz_cd column.
Possible values to provide are "America/New_York","America/Chicago", "America/Denver","America/Los_Angeles",
"America/Anchorage", as well as the following which do not use daylight savings time: "America/Honolulu",
"America/Jamaica","America/Managua","America/Phoenix", and "America/Metlakatla". See also  \code{OlsonNames()} 
for more information on time zones.}
}
\value{
A data frame with the following columns:
\tabular{lll}{
Name \tab Type \tab Description \cr
agency \tab character \tab The NWIS code for the agency reporting the data\cr
site \tab character \tab The USGS site number \cr
dateTime \tab POSIXct \tab The date and time (if applicable) of the measurement, 
          converted to UTC for unit value data. R only allows one time zone attribute per column. For unit data 
          spanning a time zone change, converting the data to UTC solves this problem. For daily data,
          the time zone attribute is the time zone of the first returned measurement.
           \cr
tz_cd \tab character \tab The time zone code for dateTime column\cr
code \tab character \tab Any codes that qualify the corresponding value\cr
value \tab numeric \tab The numeric value for the parameter \cr
}
Note that code and value are repeated for the parameters requested. The names are of the form 
X_D_P_S, where X is literal, 
D is an option description of the parameter, 
P is the parameter code, 
and S is the statistic code (if applicable).

There are also several useful attributes attached to the data frame:
\tabular{lll}{
Name \tab Type \tab Description \cr
url \tab character \tab The url used to generate the data \cr
siteInfo \tab data.frame \tab A data frame containing information on the requested sites \cr
variableInfo \tab data.frame \tab A data frame containing information on the requested parameters \cr
statisticInfo \tab data.frame \tab A data frame containing information on the requested statistics on the data \cr
queryTime \tab POSIXct \tab The time the data was returned \cr
}
}
\description{
Returns data from the NWIS web service.
Arguments to the function should be based on \url{https://waterservices.usgs.gov} service calls.
See examples below for ideas of constructing queries.
}
\examples{
\donttest{
# Examples not run for time considerations
dataTemp <- readNWISdata(stateCd="OH",parameterCd="00010", service="dv")
instFlow <- readNWISdata(sites="05114000", service="iv", 
                   parameterCd="00060", 
                   startDate="2014-05-01T00:00Z",endDate="2014-05-01T12:00Z")
                                                   
instFlowCDT <- readNWISdata(sites="05114000", service="iv", 
                   parameterCd="00060", 
                   startDate="2014-05-01T00:00",endDate="2014-05-01T12:00",
                   tz="America/Chicago")

#Empty:
multiSite <- readNWISdata(sites=c("04025000","04072150"), service="iv", 
                           parameterCd="00010")
#Not empty:
multiSite <- readNWISdata(sites=c("04025500","040263491"), 
                           service="iv", parameterCd="00060")
bBoxEx <- readNWISdata(bBox=c(-83,36.5,-81,38.5), parameterCd="00010")

startDate <- as.Date("2013-10-01")
endDate <- as.Date("2014-09-30")
waterYear <- readNWISdata(bBox=c(-83,36.5,-81,38.5), parameterCd="00010", 
                  service="dv", startDate=startDate, endDate=endDate)
siteInfo <- readNWISdata(stateCd="WI", parameterCd="00010",
                  hasDataTypeCd="iv", service="site")
qwData <- readNWISdata(bBox=c(-82.5,41.52,-81,41),startDate=as.Date("2000-01-01"),
                  drain_area_va_min=50, qw_count_nu=50,qw_attributes="expanded",
                  qw_sample_wide="wide",list_of_search_criteria=c("lat_long_bounding_box",
                  "drain_area_va","obs_count_nu"),service="qw")
temp <- readNWISdata(bBox=c(-83,36.5,-81,38.5), parameterCd="00010", service="site", 
                   seriesCatalogOutput=TRUE)
wiGWL <- readNWISdata(stateCd="WI",service="gwlevels")
meas <- readNWISdata(state_cd="WI",service="measurements",format="rdb_expanded")

waterYearStat <- readNWISdata(site=c("03112500"),service="stat",statReportType="annual",
                 statYearType="water", missingData="on")
monthlyStat <- readNWISdata(site=c("03112500","03111520"),
                            service="stat",
                            statReportType="monthly")                                   
dailyStat <- readNWISdata(site=c("03112500","03111520"),
                          service="stat",
                          statReportType="daily",
                          statType=c("p25","p50","p75","min","max"),
                          parameterCd="00065")
allDailyStats <- readNWISdata(site=c("03111548"),
                              service="stat",
                              statReportType="daily",
                              statType=c("p25","p50","p75","min","max"),
                              parameterCd="00060")

dailyWV <- readNWISdata(stateCd = "West Virginia", parameterCd = "00060")

arg.list <- list(site="03111548",
                 statReportType="daily",
                 statType=c("p25","p50","p75","min","max"),
                 parameterCd="00060")
allDailyStats_2 <- readNWISdata(arg.list, service="stat")

#' # use county names to get data
dailyStaffordVA <- readNWISdata(stateCd = "Virginia",
                                countyCd="Stafford",
                                parameterCd = "00060",
                                startDate = "2015-01-01",
                                endDate = "2015-01-30")
va_counties <- c("51001","51003","51005","51007","51009","51011","51013","51015")
va_counties_data <- readNWISdata(startDate = "2015-01-01", endDate = "2015-12-31", 
parameterCd = "00060", countycode = va_counties)
site_id <- '01594440'
rating_curve <- readNWISdata(service = "rating", site_no = site_id, file_type="base")
all_sites_base <- readNWISdata(service = "rating", file_type="base")
all_sites_core <- readNWISdata(service = "rating", file_type="corr")
all_sites_exsa <- readNWISdata(service = "rating", file_type="exsa")
all_sites_24hrs <- readNWISdata(service = "rating", file_type="exsa", period = 24)

today <- readNWISdata(service="iv", startDate = Sys.Date(), 
                      parameterCd = "00060", siteNumber = "05114000")

}
}
\seealso{
\code{\link{renameNWISColumns}},  \code{\link{importWaterML1}}, \code{\link{importRDB1}}
}
