% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fromQuantiles.R
\name{MinimalInformative}
\alias{MinimalInformative}
\title{Construct a minimally informative prior}
\usage{
MinimalInformative(
  dosegrid,
  refDose,
  threshmin = 0.2,
  threshmax = 0.3,
  probmin = 0.05,
  probmax = 0.05,
  ...
)
}
\arguments{
\item{dosegrid}{the dose grid}

\item{refDose}{the reference dose}

\item{threshmin}{Any toxicity probability above this threshold would
be very unlikely (see \code{probmin}) at the minimum dose (default: 0.2)}

\item{threshmax}{Any toxicity probability below this threshold would
be very unlikely (see \code{probmax}) at the maximum dose (default: 0.3)}

\item{probmin}{the prior probability of exceeding \code{threshmin} at the
minimum dose (default: 0.05)}

\item{probmax}{the prior probability of being below \code{threshmax} at the
maximum dose (default: 0.05)}

\item{\dots}{additional arguments for computations, see
\code{\link{Quantiles2LogisticNormal}}, e.g. \code{refDose} and
\code{logNormal=TRUE} to obtain a minimal informative log normal prior.}
}
\value{
see \code{\link{Quantiles2LogisticNormal}}
}
\description{
This function constructs a minimally informative prior, which is captured in
a \code{\linkS4class{LogisticNormal}} (or
\code{\linkS4class{LogisticLogNormal}}) object.
}
\details{
Based on the proposal by Neuenschwander et al (2008, Statistics in
Medicine), a minimally informative prior distribution is constructed. The
required key input is the minimum (\eqn{d_{1}} in the notation of the
Appendix A.1 of that paper) and the maximum value (\eqn{d_{J}}) of the dose
grid supplied to this function. Then \code{threshmin} is the probability
threshold \eqn{q_{1}}, such that any probability of DLT larger than
\eqn{q_{1}} has only 5\% probability. Therefore \eqn{q_{1}} is the 95\%
quantile of the beta distribution and hence \eqn{p_{1} = 0.95}. Likewise,
\code{threshmax} is the probability threshold \eqn{q_{J}}, such that any
probability of DLT smaller than \eqn{q_{J}} has only 5\% probability
(\eqn{p_{J} = 0.05}). The probabilities \eqn{1 - p_{1}} and \eqn{p_{J}} can be 
controlled with the arguments \code{probmin} and \code{probmax}, respectively. 
Subsequently, for all doses supplied in the
\code{dosegrid} argument, beta distributions are set up from the assumption
that the prior medians are linear in log-dose on the logit scale, and
\code{\link{Quantiles2LogisticNormal}} is used to transform the resulting
quantiles into an approximating \code{\linkS4class{LogisticNormal}} (or
\code{\linkS4class{LogisticLogNormal}}) model. Note that the reference dose
is not required for these computations.
}
\examples{

# Setting up a minimal informative prior 
# max.time is quite small only for the purpose of showing the example. They 
# should be increased for a real case.
set.seed(132)
coarseGrid <- c(0.1, 10, 30, 60, 100)
minInfModel <- MinimalInformative(dosegrid = coarseGrid,
                                  refDose=50,
                                  threshmin=0.2,
                                  threshmax=0.3,
                                  control=## for real case: leave out control 
                                    list(max.time=0.1)) 

# Plotting the result
matplot(x=coarseGrid,
        y=minInfModel$required,
        type="b", pch=19, col="blue", lty=1,
        xlab="dose",
        ylab="prior probability of DLT")
matlines(x=coarseGrid,
         y=minInfModel$quantiles,
         type="b", pch=19, col="red", lty=1)
legend("right",
       legend=c("quantiles", "approximation"),
       col=c("blue", "red"),
       lty=1,
       bty="n")

}
\keyword{programming}
