#' Return the nodes that can be reached in a detour time set around the shortest path
#' 
#' @param Graph  An object generated by cppRouting::makegraph() function.
#' @param from A vector of one or more vertices from which shortest path are calculated (origin).
#' @param to A vector of one or more vertices (destination).
#' @param extra numeric. Additional cost 
#' @param allcores Logical. If TRUE, all cores are used.
#' @return List of nodes that can be reached 
#' @note 'from' and 'to' must be the same length.
#' @details Each returned nodes \emph{n} meet the following condition : \cr
#'  \strong{SP(o,n) + SP(n,d) < SP(o,d) + t} \cr
#'  with \emph{SP} shortest distance/time, \emph{o} the origin node, \emph{d} the destination node and \emph{t} the extra cost.
#' 
#' Modified bidirectional Dijkstra algorithm is ran for each path.
#' @examples 
#' if(requireNamespace("igraph",quietly = TRUE)){
#' 
#' #Generate fully connected graph
#' gf<- igraph::make_full_graph(400)
#' igraph::V(gf)$names<-1:400
#'
#' #Convert to data frame and add random weights
#' df<-igraph::as_long_data_frame(gf)
#' df$dist<-sample(1:100,nrow(df),replace = TRUE)
#' 
#' #Construct cppRouting graph
#' graph<-makegraph(df[,c(1,2,5)],directed = FALSE)
#'
#' #Pick up random origin and destination node
#' origin<-sample(1:400,1)
#' destination<-sample(1:400,1)
#'
#' #Compute distance from origin to all nodes
#' or_to_all<-get_distance_matrix(graph,from=origin,to=1:400,allcores=FALSE)
#' 
#' #Compute distance from all nodes to destination 
#' all_to_dest<-get_distance_matrix(graph,from=1:400,to=destination,allcores=FALSE)
#'
#' #Get all shortest paths from origin to destination, passing by each node of the graph
#' total_paths<-rowSums(cbind(t(or_to_all),all_to_dest))
#'
#' #Compute shortest path between origin and destination 
#' distance<-get_distance_pair(graph,from=origin,to=destination,allcores=FALSE)
#'
#' #Compute detour with an additional cost of 3
#' det<-get_detour(graph,from=origin,to=destination,extra=3,allcores=FALSE)
#' 
#' #Check result validity
#' length(unlist(det))
#' length(total_paths[total_paths < distance + 3])
#' 
#' } 

get_detour<-function(Graph,from,to,extra=NULL,allcores=FALSE){
  if (length(from)!=length(to)) stop("From and to have not the same length")
  if (is.null(extra)) stop("No additional cost provided")
  if (length(extra)!=1) stop("extra should be one number")
  extra<-as.numeric(extra)
  if (extra<=0) stop("extra must be positive")
  if (any(is.na(cbind(from,to)))) stop("NAs are not allowed in origin/destination nodes")
  from<-as.character(from)
  
  to<-as.character(to)
  allnodes<-c(from,to)
  if (sum(allnodes %in% Graph$dict$ref)<length(allnodes)) stop("Some nodes are not in the graph")
  
  from_id<-Graph$dict$id[match(from,Graph$dict$ref)]
  to_id<-Graph$dict$id[match(to,Graph$dict$ref)]
  
  if (allcores==FALSE) res<-Detour(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,t=extra,Graph$dict$ref)
  else {
    numWorkers <- parallel::detectCores()
    cl <- parallel::makeCluster(numWorkers, type = "PSOCK")
    parallel::clusterEvalQ(cl = cl,library("cppRouting"))
    chunks <- parallel::splitIndices(length(from), ncl = numWorkers)
    mylist<-lapply(chunks,function(x) from_id[x])
    mylist2<-lapply(chunks,function(x) to_id[x])
    
    
    res<-parallel::clusterMap(cl,Detour,dep=mylist,arr=mylist2,
                              MoreArgs = list(gfrom=Graph$data$from,gto=Graph$data$to,gw=Graph$data$dist,NbNodes=Graph$nbnode,t=extra,dict=Graph$dict$ref))
    parallel::stopCluster(cl)
    
    res<-do.call(c,res)
  }
  
  names(res)<-paste0(from,"_",to)
  return(res)
}

