% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crplot.R
\name{crplot}
\alias{crplot}
\title{Plotting Two-Dimensional Confidence Regions}
\usage{
crplot(dataset, alpha, distn,
                heuristic = 1,
                maxdeg    = 5,
                ellipse_n = 4,
                pts       = TRUE,
                mlelab    = TRUE,
                sf        = c(5, 5),
                mar       = c(4, 4.5, 2, 1.5),
                xlab      = "",
                ylab      = "",
                main      = "",
                xlas      = 1,
                ylas      = 2,
                origin    = FALSE,
                xlim      = NULL,
                ylim      = NULL,
                tol       = .Machine$double.eps ^ 0.5,
                info      = FALSE,
                repair    = TRUE,
                showplot  = TRUE )
}
\arguments{
\item{dataset}{a 1 x n vector of dataset values.}

\item{alpha}{significance level; resulting plot illustrates a 100(1 - alpha)\% confidence region.}

\item{distn}{distribution to fit the dataset to; accepted values: \code{'gamma'}, \code{'invgauss'},
\code{'llogis'}, \code{'lnorm'}, \code{'norm'}, \code{'unif'}, \code{'weibull'}.}

\item{heuristic}{numeric value selecting method for plotting: 0 for elliptic-oriented point distribution, and
1 for smoothing boundary search heuristic.}

\item{maxdeg}{maximum angle tolerance between consecutive plot segments in degrees.}

\item{ellipse_n}{number of roughly equidistant confidence region points to plot using the
elliptic-oriented point distribution (must be a multiple of four because its algorithm
exploits symmetry in the quadrants of an ellipse).}

\item{pts}{displays confidence region boundary points identified if \code{TRUE}.}

\item{mlelab}{logical argument to include the maximum
likelihood estimate coordinate point (default is \code{TRUE}).}

\item{sf}{specifies the number of significant figures on axes labels.}

\item{mar}{specifies margin values for \code{par(mar = c( ))} (see \code{mar} in \code{\link{par}}).}

\item{xlab}{string specifying the x axis label.}

\item{ylab}{string specifying the y axis label.}

\item{main}{string specifying the plot title.}

\item{xlas}{numeric in {0, 1, 2, 3} specifying the style of axis labels (see \code{las} in \code{\link{par}}).}

\item{ylas}{numeric in {0, 1, 2, 3} specifying the style of axis labels (see \code{las} in \code{\link{par}}).}

\item{origin}{logical argument to include the plot origin (default is \code{FALSE}).}

\item{xlim}{two element vector containing horizontal axis minimum and maximum values.}

\item{ylim}{two element vector containing vertical axis minimum and maximum values.}

\item{tol}{the \code{\link{uniroot}} parameter specifying its required accuracy.}

\item{info}{logical argument to return plot information: MLE prints to screen; (x, y) plot point coordinates
and corresponding phi angles (with respect to MLE) are returned as a list.}

\item{repair}{logical argument to repair regions inaccessible using a radial angle from its MLE due to multiple
roots at select \eqn{\phi} angles.}

\item{showplot}{logical argument specifying if a plot is output; altering from its default of \code{TRUE} is
only logical assuming \code{crplot} is run for its data only (see the \code{info} argument).}
}
\value{
if the optional argument \code{info = TRUE} is included then a list of plot coordinates and phi angles is returned
}
\description{
Plots the two-dimensional confidence region for probability distribution parameters (supported distribution
suffixes: gamma, invgauss, lnorm, llogis, norm, unif, weibull) corresponding to a user given dataset and level
of significance.
}
\details{
This function plots confidence regions for a variety of two-parameter distributions.  It requires
\itemize{
\item a vector of dataset values,
\item the level of significance (alpha), and
\item a distribution to fit the data to.
}
Plots display according to probability density function parameterization given later in this section.
Two heuristics (and their associated combination) are available to plot confidence regions.  Along
with their descriptions, they are:
\enumerate{
\item \emph{Smoothing Boundary Search Heuristic (default)}.  This heuristic plots more points in areas of
greater curvature to ensure a smooth appearance throughout the confidence region boundary.  Its
\code{maxdeg} parameter specifies the maximum tolerable angle between three successive points.
Lower values of \code{maxdeg} result in smoother plots, and its default value of 5 degrees
provides adequate smoothing in most circumstances.  Values of \code{maxdeg} \eqn{\le} 3 are not
permitted due to their complicating implications to trigonometric numerical approximations near 0
and 1.
\item \emph{Elliptic-Oriented Point Distribution}.  This heuristic allows the user to specify
a number of points to plot along the confidence region boundary at roughly uniform intervals.
Its name is derived from the technique it uses to choose these points---an extension of the Steiner
generation of a non-degenerate conic section, also known as the parallelogram method---which identifies
points along an ellipse that are approximately equidistant.  To exploit the computational benefits of
ellipse symmetry over its four quadrants, \code{ellipse_n} value must be divisible by four.}
By default, \code{crplot} implements the smoothing boundary search heuristic.  Alternatively,
the user can plot using the elliptic-oriented point distribution algorithm, or a combination
of them both.  Combining the two techniques initializes the plot using the elliptic-oriented point
distribution algorithm, and then subsequently populates additional points in areas of high curvature
(those outside of the maximum angle tolerance parameterization) in accordance with the smoothing
boundary search heuristic.  This combination results when the smoothing boundary search heuristic
is specified in conjunction with an \code{ellipse_n} value greater than four.

Both of the aforementioned heuristics use a radial profile log likelihood function to identify
points along the confidence region boundary.  It cuts the log likelihood function in a directional
azimuth from its MLE, and locates the associated confidence region boundary point using the
asymptotic results associated with the ratio test statistic \eqn{-2 [log L(\theta) - log L(\theta hat)]}
which converges in distribution to the chi-square distribution with two degrees of freedom (for
a two parameter distribution).

This package uses the following parameterization for its supported distributions, and illustrates
the corresponding confidence regions accordingly:

\itemize{
\item The gamma distribution
for shape parameter \eqn{\kappa > 0}, scale parameter \eqn{\theta > 0}, and \eqn{x > 0},
has the probability density function
\deqn{1 / (Gamma(\kappa) \theta ^ \kappa) x ^ {(\kappa - 1)} exp(-x / \theta).}

\item The inverse Gaussian distribution
for mean \eqn{\mu > 0}, shape parameter \eqn{\lambda > 0}, and \eqn{x > 0},
has the probability density function
\deqn{\sqrt (\lambda / (2 \pi x ^ 3)) exp( - \lambda (x - \mu) ^ 2 / (2 \mu ^ 2 x)).}

\item The log logistic distribution
for scale parameter \eqn{\lambda > 0}, shape parameter \eqn{\kappa > 0}, and \eqn{x \ge 0},
has a probability density function
\deqn{(\kappa \lambda) (x \lambda) ^ {(\kappa - 1)} / (1 + (\lambda x) ^ \kappa) ^ 2.}

\item The log normal distribution
for the real-numbered mean \eqn{\mu} of the logarithm, standard deviation \eqn{\sigma > 0}
of the logarithm, and \eqn{x > 0},
has the probability density function
\deqn{1 / (x \sigma \sqrt(2 \pi)) exp(-(\log x - \mu) ^ 2 / (2 \sigma ^ 2)).}

\item The normal distribution
for the real-numbered mean \eqn{\mu}, standard deviation \eqn{\sigma > 0}, and \eqn{x} is a real number,
has the probability density function
\deqn{1 / \sqrt (2 \pi \sigma ^ 2) exp(-(x - \mu) ^ 2 / (2 \sigma ^ 2)).}

\item The uniform distribution for real-valued parameters \eqn{a} and \eqn{b} where \eqn{a < b}
and \eqn{a \le x \le b},
has the probability density function
\deqn{1 / (b - a).}

\item The Weibull distribution
for scale parameter \eqn{\lambda > 0}, shape parameter \eqn{\kappa > 0}, and \eqn{x > 0},
has the probability density function
\deqn{\kappa (\lambda ^ \kappa) x ^ {(\kappa - 1)} exp(-(\lambda x) ^ \kappa).}
}

The confidence region horizontal and vertical axis convention in use by \code{crplot} for each
distribution is:
\itemize{
\item The gamma distribution confidence region plot shows \eqn{\theta} on its horizontal axis,
and \eqn{\kappa} on its vertical axis.

\item The inverse Gaussian distribution confidence region plot shows \eqn{\mu} on its horizontal
axis, and \eqn{\lambda} on its vertical axis.

\item The log normal distribution confidence region plot shows \eqn{\mu} on its horizontal axis,
and \eqn{\sigma} on its vertical axis.

\item The log logistic distribution confidence region plot shows \eqn{\lambda} on its
horizontal axis, and \eqn{\kappa} on its vertical axis.

\item The normal distribution confidence region plot shows \eqn{\mu} on its horizontal axis, and
\eqn{\sigma} on its vertical axis.

\item The uniform distribution confidence region plot shows \eqn{a} on its horizontal axis, and
\eqn{b} on its vertical axis.

\item The Weibull distribution confidence region plot shows \eqn{\kappa} on its horizontal axis,
and \eqn{\lambda} on its vertical axis.
}
}
\examples{
## plot the 95\% confidence region for Weibull shape and scale parameters
## corresponding to the given ballbearing dataset
ballbearing <- c(17.88, 28.92, 33.00, 41.52, 42.12, 45.60, 48.48, 51.84,
                 51.96, 54.12, 55.56, 67.80, 68.64, 68.64, 68.88, 84.12,
                 93.12, 98.64, 105.12, 105.84, 127.92, 128.04, 173.40)
crplot(dataset = ballbearing, distn = "weibull", alpha = 0.05)

## repeat this plot using the elliptic-oriented point distribution heuristic
crplot(dataset = ballbearing, distn = "weibull", alpha = 0.05,
       heuristic = 0, ellipse_n = 80)

## combine the two heuristics, compensating any elliptic-oriented point verticies whose apparent
## angles > 6 degrees with additional points, and expand the plot area to include the origin
crplot(dataset = ballbearing, distn = "weibull", alpha = 0.05,
       maxdeg = 6, ellipse_n = 80, origin = TRUE)

## next use the inverse Gaussian distribution and show no plot points
crplot(dataset = ballbearing, distn = "invgauss", alpha = 0.05,
       pts = FALSE)
}
\references{
Jaeger, A. (2016), "Computation of Two- and Three-Dimensional Confidence Regions with the Likelihood Ratio",
The American Statistician, 49, 48--53.
}
\seealso{
\code{\link{coversim}}, \code{\link{uniroot}}
}
\author{
Christopher Weld (\email{ceweld@email.wm.edu})

Lawrence Leemis (\email{leemis@math.wm.edu})
}
\concept{
confidence region plot
}
\keyword{confidence}
\keyword{data}
\keyword{estimation,}
\keyword{graphical}
\keyword{graphics,}
\keyword{intervals,}
\keyword{methods,}
\keyword{numerical}
\keyword{optimization}
\keyword{parameter}
\keyword{region,}
\keyword{statistical}
\keyword{visualization,}
