context("fbetween / B and fwithin / W")

x <- rnorm(100)
w <- abs(100*rnorm(100))
wdat <- abs(100*rnorm(32))
xNA <- x
wNA <- w
wdatNA <- wdat
xNA[sample.int(100,20)] <- NA
wNA[sample.int(100,20)] <- NA
wdatNA[sample.int(32, 5)] <- NA
f <- as.factor(rep(1:10, each = 10))
g <- as.factor(rep(c(1,2,2,3,3,3,4,4,4,4,5,5,5,5,5,7,7,7,7,7,7,7,10,10,10,10,10,10,10,10,10,10)))
mtcNA <- na_insert(mtcars)
mtcNA[1,1] <- NA # single group NA !!
m <- as.matrix(mtcars)
mNA <- as.matrix(mtcNA)
mNAc <- mNA
storage.mode(mNAc) <- "character"

# x = rnorm(1e7)
# xNA = x
# xNA[sample.int(1e7,1e6)] <- NA
# w = abs(100*rnorm(1e7))
# wNA = w
# wNA[sample.int(1e7,1e6)] <- NA

# microbenchmark(fwithin(xNA), fbetween(xNA), fbetween(xNA, w = w), fwithin(xNA, w = w), fbetween(xNA, w = wNA), fwithin(xNA, w = wNA))
# Unit: milliseconds
# expr      min       lq      mean   median       uq      max neval  cld
# fwithin(xNA) 59.89809 61.45215  81.20188 63.21997 65.99563 303.5464   100 a
# fbetween(xNA) 71.32829 73.00953  86.06850 74.51227 77.79108 275.6274   100 ab
# fbetween(xNA, w = w) 81.95167 84.85050 106.61714 86.65870 90.92104 314.8245   100   cd
# fwithin(xNA, w = w) 71.24841 73.72264  88.08572 75.32935 80.46232 279.5597   100 a c
# fbetween(xNA, w = wNA) 90.99712 93.71455 107.38818 95.91545 98.16989 328.8951   100    d
# fwithin(xNA, w = wNA) 80.13678 83.62511 103.55614 86.22361 93.18352 301.7070   100  bcd

baseB <- function(x, na.rm = FALSE) {
  if(!na.rm) return(ave(x))
  cc <- !is.na(x)
  x[cc] <- ave(x[cc])
  return(x)
}
baseW <- function(x, na.rm = FALSE, mean = 0) {
  if(!na.rm) return(x - ave(x) + mean)
  cc <- !is.na(x)
  m <- sum(x[cc]) / sum(cc)
  return(x - m + mean)
}

# NOTE: This is what fbetween and fwithin currently do: If missing values, compute weighted mean on available obs, and center x using it. but don't insert aditional missing values in x for missing weights ..
wbaseB <- function(x, w, na.rm = FALSE) {
  if(na.rm) {
    xcc <- !is.na(x)
    cc <- xcc & !is.na(w)
    w <- w[cc]
    wm <- sum(w * x[cc]) / sum(w)
    x[xcc] <- rep(wm, sum(xcc))
    return(x)
  } else {
    wm <- sum(w * x) / sum(w)
    return(rep(wm, length(x)))
  }
}
wbaseW <- function(x, w, na.rm = FALSE, mean = 0) {
  if(na.rm) {
    cc <- complete.cases(x, w)
    w <- w[cc]
    wm <- sum(w * x[cc]) / sum(w)
  } else wm <- sum(w * x) / sum(w)
    return(x - wm + mean)
}

# fbetween

test_that("fbetween performs like baseB", {
  expect_equal(fbetween(NA), as.double(baseB(NA)))
  expect_equal(fbetween(NA, na.rm = FALSE), as.double(baseB(NA)))
  expect_equal(fbetween(1), baseB(1, na.rm = TRUE))
  expect_equal(fbetween(1:3), baseB(1:3, na.rm = TRUE))
  expect_equal(fbetween(-1:1), baseB(-1:1, na.rm = TRUE))
  expect_equal(fbetween(1, na.rm = FALSE), baseB(1))
  expect_equal(fbetween(1:3, na.rm = FALSE), baseB(1:3))
  expect_equal(fbetween(-1:1, na.rm = FALSE), baseB(-1:1))
  expect_equal(fbetween(x), baseB(x, na.rm = TRUE))
  expect_equal(fbetween(x, na.rm = FALSE), baseB(x))
  expect_equal(fbetween(xNA, na.rm = FALSE), baseB(xNA))
  expect_equal(fbetween(xNA), baseB(xNA, na.rm = TRUE))
  expect_equal(qM(fbetween(mtcars)), fbetween(m))
  expect_equal(fbetween(m), dapply(m, baseB, na.rm = TRUE))
  expect_equal(fbetween(m, na.rm = FALSE), dapply(m, baseB))
  expect_equal(fbetween(mNA, na.rm = FALSE), dapply(mNA, baseB))
  expect_equal(fbetween(mNA), dapply(mNA, baseB, na.rm = TRUE))
  expect_equal(fbetween(x, f), BY(x, f, baseB, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fbetween(x, f, na.rm = FALSE), BY(x, f, baseB, use.g.names = FALSE))
  expect_equal(fbetween(xNA, f, na.rm = FALSE), BY(xNA, f, baseB, use.g.names = FALSE))
  expect_equal(fbetween(xNA, f), BY(xNA, f, baseB, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fbetween(m, g), BY(m, g, baseB, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fbetween(m, g, na.rm = FALSE), BY(m, g, baseB, use.g.names = FALSE))
  expect_equal(fbetween(mNA, g, na.rm = FALSE), BY(mNA, g, baseB, use.g.names = FALSE))
  expect_equal(fbetween(mNA, g), BY(mNA, g, baseB, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fbetween(mtcars, g), BY(mtcars, g, baseB, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fbetween(mtcars, g, na.rm = FALSE), BY(mtcars, g, baseB, use.g.names = FALSE))
  expect_equal(fbetween(mtcNA, g, na.rm = FALSE), BY(mtcNA, g, baseB, use.g.names = FALSE))
  expect_equal(fbetween(mtcNA, g), BY(mtcNA, g, baseB, na.rm = TRUE, use.g.names = FALSE))
})

test_that("fbetween performs like fbetween with weights all equal", {
  expect_equal(fbetween(NA), fbetween(NA, w = 0.99999999))
  expect_equal(fbetween(NA, na.rm = FALSE), fbetween(NA, w = 2.946, na.rm = FALSE))
  expect_equal(fbetween(1), fbetween(1, w = 3))
  expect_equal(fbetween(1:3), fbetween(1:3, w = rep(0.999,3)))
  expect_equal(fbetween(-1:1), fbetween(-1:1, w = rep(4.2,3)))
  expect_equal(fbetween(1, na.rm = FALSE), fbetween(1, w = 5, na.rm = FALSE))
  expect_equal(fbetween(1:3, na.rm = FALSE), fbetween(1:3, w = rep(1.44565, 3), na.rm = FALSE))
  expect_equal(fbetween(-1:1, na.rm = FALSE), fbetween(-1:1, w = rep(1.44565, 3), na.rm = FALSE))
  expect_equal(fbetween(x), fbetween(x, w = rep(1,100)))
  expect_equal(fbetween(x, na.rm = FALSE), fbetween(x, w = rep(1.44565, 100), na.rm = FALSE))
  expect_equal(fbetween(xNA, na.rm = FALSE), fbetween(xNA, w = rep(4.676587, 100), na.rm = FALSE))
  expect_equal(fbetween(xNA), fbetween(xNA, w = rep(4.676587, 100)))
  expect_equal(fbetween(m), fbetween(m, w = rep(6587.3454, 32)))
  expect_equal(fbetween(m, na.rm = FALSE), fbetween(m, w = rep(6587.3454, 32), na.rm = FALSE))
  expect_equal(fbetween(mNA, na.rm = FALSE), fbetween(mNA, w = rep(6587.3454, 32), na.rm = FALSE))
  expect_equal(fbetween(mNA), fbetween(mNA, w = rep(6587.3454, 32)))
  expect_equal(fbetween(mtcars), fbetween(mtcars, w = rep(6787.3454, 32)))
  expect_equal(fbetween(mtcars, na.rm = FALSE), fbetween(mtcars, w = rep(6787.3454, 32), na.rm = FALSE))
  expect_equal(fbetween(mtcNA, na.rm = FALSE), fbetween(mtcNA, w = rep(6787.3454, 32), na.rm = FALSE))
  expect_equal(fbetween(mtcNA), fbetween(mtcNA, w = rep(6787.3454, 32)))
  expect_equal(fbetween(x, f), fbetween(x, f, rep(546.78,100)))
  expect_equal(fbetween(x, f, na.rm = FALSE), fbetween(x, f, rep(5.88,100), na.rm = FALSE))
  expect_equal(fbetween(xNA, f, na.rm = FALSE), fbetween(xNA, f, rep(52.7,100), na.rm = FALSE))
  expect_equal(fbetween(xNA, f), fbetween(xNA, f, rep(5997456,100)))
  expect_equal(fbetween(m, g), fbetween(m, g, rep(546.78,32)))
  expect_equal(fbetween(m, g, na.rm = FALSE), fbetween(m, g, rep(0.0001,32), na.rm = FALSE))
  expect_equal(fbetween(mNA, g, na.rm = FALSE), fbetween(mNA, g, rep(5.7,32), na.rm = FALSE))
  expect_equal(fbetween(mNA, g), fbetween(mNA, g, rep(1.1,32)))
  expect_equal(fbetween(mtcars, g), fbetween(mtcars, g, rep(53,32)))
  expect_equal(fbetween(mtcars, g, na.rm = FALSE), fbetween(mtcars, g, rep(546.78,32), na.rm = FALSE))
  expect_equal(fbetween(mtcNA, g, na.rm = FALSE), fbetween(mtcNA, g, rep(0.999999,32), na.rm = FALSE))
  expect_equal(fbetween(mtcNA, g), fbetween(mtcNA, g, rep(999.9999,32)))
})

test_that("fbetween with weights performs like wbaseB (defined above)", {
  # complete weights
  expect_equal(fbetween(NA, w = 1), wbaseB(NA, 1))
  expect_equal(fbetween(NA, w = 1, na.rm = FALSE), wbaseB(NA, 1))
  expect_equal(fbetween(1, w = 1), wbaseB(1, w = 1))
  expect_equal(fbetween(1:3, w = 1:3), wbaseB(1:3, 1:3))
  expect_equal(fbetween(-1:1, w = 1:3), wbaseB(-1:1, 1:3))
  expect_equal(fbetween(1, w = 1, na.rm = FALSE), wbaseB(1, 1))
  expect_equal(fbetween(1:3, w = c(0.99,3454,1.111), na.rm = FALSE), wbaseB(1:3, c(0.99,3454,1.111)))
  expect_equal(fbetween(-1:1, w = 1:3, na.rm = FALSE), wbaseB(-1:1, 1:3))
  expect_equal(fbetween(x, w = w), wbaseB(x, w))
  expect_equal(fbetween(x, w = w, na.rm = FALSE), wbaseB(x, w))
  expect_equal(fbetween(xNA, w = w, na.rm = FALSE), wbaseB(xNA, w))
  expect_equal(fbetween(xNA, w = w), wbaseB(xNA, w, na.rm = TRUE))
  expect_equal(qM(fbetween(mtcars, w = wdat)), fbetween(m, w = wdat))
  expect_equal(fbetween(m, w = wdat), dapply(m, wbaseB, wdat, na.rm = TRUE))
  expect_equal(fbetween(m, w = wdat, na.rm = FALSE), dapply(m, wbaseB, wdat))
  expect_equal(fbetween(mNA, w = wdat, na.rm = FALSE), dapply(mNA, wbaseB, wdat))
  expect_equal(fbetween(mNA, w = wdat), dapply(mNA, wbaseB, wdat, na.rm = TRUE))
  expect_equal(fbetween(x, f, w), unlist(Map(wbaseB, split(x, f), split(w, f)), use.names = FALSE))
  expect_equal(fbetween(x, f, w, na.rm = FALSE), unlist(Map(wbaseB, split(x, f), split(w, f)), use.names = FALSE))
  expect_equal(fbetween(xNA, f, w, na.rm = FALSE), unlist(Map(wbaseB, split(xNA, f), split(w, f)), use.names = FALSE))
  expect_equal(fbetween(xNA, f, w), unlist(Map(wbaseB, split(xNA, f), split(w, f), na.rm = TRUE), use.names = FALSE))
  # missing weights
  expect_equal(fbetween(NA, w = NA), wbaseB(NA, NA))
  expect_equal(fbetween(NA, w = NA, na.rm = FALSE), wbaseB(NA, NA))
  expect_equal(fbetween(1, w = NA), wbaseB(1, w = NA))
  expect_equal(fbetween(1:3, w = c(NA,1:2)), wbaseB(1:3, c(NA,1:2), na.rm = TRUE))
  expect_equal(fbetween(-1:1, w = c(NA,1:2)), wbaseB(-1:1, c(NA,1:2), na.rm = TRUE))
  expect_equal(fbetween(1, w = NA, na.rm = FALSE), wbaseB(1, NA))
  expect_equal(fbetween(1:3, w = c(NA,1:2), na.rm = FALSE), wbaseB(1:3, c(NA,1:2)))
  expect_equal(fbetween(-1:1, w = c(NA,1:2), na.rm = FALSE), wbaseB(-1:1, c(NA,1:2)))
  expect_equal(fbetween(x, w = wNA), wbaseB(x, wNA, na.rm = TRUE))
  expect_equal(fbetween(x, w = wNA, na.rm = FALSE), wbaseB(x, wNA))
  expect_equal(fbetween(xNA, w = wNA, na.rm = FALSE), wbaseB(xNA, wNA))
  expect_equal(fbetween(xNA, w = wNA), wbaseB(xNA, wNA, na.rm = TRUE))
  expect_equal(qM(fbetween(mtcars, w = wdatNA)), fbetween(m, w = wdatNA))
  expect_equal(fbetween(m, w = wdatNA), dapply(m, wbaseB, wdatNA, na.rm = TRUE))
  expect_equal(fbetween(m, w = wdatNA, na.rm = FALSE), dapply(m, wbaseB, wdatNA))
  expect_equal(fbetween(mNA, w = wdatNA, na.rm = FALSE), dapply(mNA, wbaseB, wdatNA))
  expect_equal(fbetween(mNA, w = wdatNA), dapply(mNA, wbaseB, wdatNA, na.rm = TRUE))
  expect_equal(fbetween(x, f, wNA), unlist(Map(wbaseB, split(x, f), split(wNA, f), na.rm = TRUE), use.names = FALSE))
  expect_equal(fbetween(x, f, wNA, na.rm = FALSE), unlist(Map(wbaseB, split(x, f), split(wNA, f)), use.names = FALSE))
  expect_equal(fbetween(xNA, f, wNA, na.rm = FALSE), unlist(Map(wbaseB, split(xNA, f), split(wNA, f)), use.names = FALSE))
  expect_equal(fbetween(xNA, f, wNA), unlist(Map(wbaseB, split(xNA, f), split(wNA, f), na.rm = TRUE), use.names = FALSE))
})

test_that("fbetween performs numerically stable", {
  expect_true(all_obj_equal(replicate(50, fbetween(1), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(NA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(NA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, f), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, f, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, f, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, f), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, g), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, g, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, g, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, g), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, g), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, g, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, g, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, g), simplify = FALSE)))
})

test_that("fbetween with complete weights performs numerically stable", {
  expect_true(all_obj_equal(replicate(50, fbetween(1, w = 1), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(NA, w = 1), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(NA, w = 1, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, w = w), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, w = w, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, w = w, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, w = w), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, w = wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, w = wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, w = wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, w = wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, w = wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, w = wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, w = wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, w = wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, f, w), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, f, w, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, f, w, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, f, w), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, g, wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, g, wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, g, wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, g, wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, g, wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, g, wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, g, wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, g, wdat), simplify = FALSE)))
})

test_that("fbetween with missing weights performs numerically stable", {
  expect_true(all_obj_equal(replicate(50, fbetween(1, w = NA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(NA, w = NA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(NA, w = NA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, w = wNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, w = wNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, w = wNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, w = wNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, w = wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, w = wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, w = wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, w = wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, w = wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, w = wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, w = wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, w = wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, f, wNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(x, f, wNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, f, wNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(xNA, f, wNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, g, wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(m, g, wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, g, wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mNA, g, wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, g, wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcars, g, wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, g, wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fbetween(mtcNA, g, wdatNA), simplify = FALSE)))
})

test_that("fbetween handles special values in the right way", {
  expect_equal(fbetween(NA), NA_real_)
  expect_equal(fbetween(NaN), NaN)
  expect_equal(fbetween(Inf), Inf)
  expect_equal(fbetween(c(Inf,Inf)), c(Inf,Inf))
  expect_equal(fbetween(-Inf), -Inf)
  expect_equal(fbetween(c(-Inf,-Inf)), c(-Inf,-Inf))
  expect_equal(fbetween(TRUE), 1)
  expect_equal(fbetween(FALSE), 0)
  expect_equal(fbetween(NA, na.rm = FALSE), NA_real_)
  expect_equal(fbetween(NaN, na.rm = FALSE), NaN)
  expect_equal(fbetween(Inf, na.rm = FALSE), Inf)
  expect_equal(fbetween(c(Inf,Inf), na.rm = FALSE), c(Inf,Inf))
  expect_equal(fbetween(-Inf, na.rm = FALSE), -Inf)
  expect_equal(fbetween(c(-Inf,-Inf), na.rm = FALSE), c(-Inf,-Inf))
  expect_equal(fbetween(TRUE, na.rm = FALSE), 1)
  expect_equal(fbetween(FALSE, na.rm = FALSE), 0)
  expect_equal(fbetween(c(1,NA)), c(1,NA_real_))
  expect_equal(fbetween(c(1,NaN)), c(1,NaN))
  expect_equal(fbetween(c(1,Inf)), c(Inf,Inf))
  expect_equal(fbetween(c(1,-Inf)), c(-Inf,-Inf))
  expect_equal(fbetween(c(1,Inf), na.rm = FALSE), c(Inf,Inf))
  expect_equal(fbetween(c(1,-Inf), na.rm = FALSE), c(-Inf,-Inf))
  expect_equal(fbetween(c(NA,-Inf), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(c(FALSE,FALSE), na.rm = FALSE), c(0,0))
  expect_equal(fbetween(c(1,NA), na.rm = FALSE), c(NA_real_,NA_real_))
})

test_that("fbetween with weights handles special values in the right way", {
  expect_equal(fbetween(NA, w = 1), NA_real_)
  expect_equal(fbetween(NaN, w = 1), NaN)
  expect_equal(fbetween(Inf, w = 1), Inf)
  expect_equal(fbetween(c(Inf,Inf), w = 1:2), c(Inf,Inf))
  expect_equal(fbetween(-Inf, w = 1), -Inf)
  expect_equal(fbetween(c(-Inf,-Inf), w = 1:2), c(-Inf,-Inf))
  expect_equal(fbetween(TRUE, w = 1), 1)
  expect_equal(fbetween(FALSE, w = 1), 0)
  expect_equal(fbetween(NA, w = 1, na.rm = FALSE), NA_real_)
  expect_equal(fbetween(NaN, w = 1, na.rm = FALSE), NaN)
  expect_equal(fbetween(Inf, w = 1, na.rm = FALSE), Inf)
  expect_equal(fbetween(c(Inf,Inf), w = 1:2, na.rm = FALSE), c(Inf,Inf))
  expect_equal(fbetween(-Inf, w = 1, na.rm = FALSE), -Inf)
  expect_equal(fbetween(c(-Inf,-Inf), w = 1:2, na.rm = FALSE), c(-Inf,-Inf))
  expect_equal(fbetween(TRUE, w = 1, na.rm = FALSE), 1)
  expect_equal(fbetween(FALSE, w = 1, na.rm = FALSE), 0)
  expect_equal(fbetween(c(1,NA), w = 1:2), c(1,NA_real_))
  expect_equal(fbetween(c(1,NaN), w = 1:2), c(1,NaN))
  expect_equal(fbetween(c(1,Inf), w = 1:2), c(Inf,Inf))
  expect_equal(fbetween(c(1,-Inf), w = 1:2), c(-Inf,-Inf))
  expect_equal(fbetween(c(1,Inf), w = 1:2, na.rm = FALSE), c(Inf,Inf))
  expect_equal(fbetween(c(1,-Inf), w = 1:2, na.rm = FALSE), c(-Inf,-Inf))
  expect_equal(fbetween(c(NA,-Inf), w = 1:2, na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(c(FALSE,FALSE), w = 1:2, na.rm = FALSE), c(0,0))
  expect_equal(fbetween(c(1,NA), w = 1:2, na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(1:3, w = c(1,Inf,3)), c(NA_real_,NA_real_,NA_real_))
  expect_equal(fbetween(1:3, w = c(1,-Inf,3)), c(NA_real_,NA_real_,NA_real_))
  expect_equal(fbetween(1:3, w = c(1,Inf,3), na.rm = FALSE), c(NA_real_,NA_real_,NA_real_))
  expect_equal(fbetween(1:3, w = c(1,-Inf,3), na.rm = FALSE), c(NA_real_,NA_real_,NA_real_))

  expect_equal(fbetween(NA, w = NA), NA_real_)
  expect_equal(fbetween(NaN, w = NA), NaN)
  expect_equal(fbetween(Inf, w = NA), NA_real_)
  expect_equal(fbetween(c(Inf,Inf), w = c(NA,2)), c(Inf,Inf))
  expect_equal(fbetween(-Inf, w = NA), NA_real_)
  expect_equal(fbetween(c(-Inf,-Inf), w = c(NA,2)), c(-Inf,-Inf))
  expect_equal(fbetween(TRUE, w = NA), NA_real_)
  expect_equal(fbetween(FALSE, w = NA), NA_real_)
  expect_equal(fbetween(NA, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fbetween(NaN, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fbetween(Inf, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fbetween(c(Inf,Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(-Inf, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fbetween(c(-Inf,-Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(TRUE, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fbetween(FALSE, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fbetween(c(1,NA), w = c(NA,2)), c(NA_real_,NA_real_))
  expect_equal(fbetween(c(1,NaN), w = c(NA,2)), c(NA_real_,NA_real_))
  expect_equal(fbetween(c(1,Inf), w = c(NA,2)), c(Inf,Inf))
  expect_equal(fbetween(c(1,-Inf), w = c(NA,2)), c(-Inf,-Inf))
  expect_equal(fbetween(c(1,Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(c(1,-Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(c(NA,-Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(c(FALSE,FALSE), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(c(1,NA), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fbetween(1:3, w = c(NA,Inf,3)), c(NA_real_,NA_real_,NA_real_))
  expect_equal(fbetween(1:3, w = c(NA,-Inf,3)), c(NA_real_,NA_real_,NA_real_))
  expect_equal(fbetween(1:3, w = c(NA,Inf,3), na.rm = FALSE), c(NA_real_,NA_real_,NA_real_))
  expect_equal(fbetween(1:3, w = c(NA,-Inf,3), na.rm = FALSE), c(NA_real_,NA_real_,NA_real_))
})

test_that("fbetween produces errors for wrong input", {
  expect_error(fbetween("a"))
  expect_error(fbetween(NA_character_))
  expect_error(fbetween(mNAc))
  expect_error(fbetween(mNAc, f))
  expect_error(fbetween(1:2,1:3))
  expect_error(fbetween(m,1:31))
  expect_error(fbetween(mtcars,1:31))
  expect_error(fbetween("a", w = 1))
  expect_error(fbetween(1:2, w = 1:3))
  expect_error(fbetween(NA_character_, w = 1))
  expect_error(fbetween(mNAc, w = wdat))
  expect_error(fbetween(mNAc, f, wdat))
  expect_error(fbetween(mNA, w = 1:33))
  expect_error(fbetween(1:2,1:2, 1:3))
  expect_error(fbetween(m,1:32,1:20))
  expect_error(fbetween(mtcars,1:32,1:10))
  expect_error(fbetween(1:2, w = c("a","b")))
  expect_error(fbetween(wlddev))
  expect_error(fbetween(wlddev, w = wlddev$year))
  expect_error(fbetween(wlddev, wlddev$iso3c))
  expect_error(fbetween(wlddev, wlddev$iso3c, wlddev$year))
})

# B

test_that("B produces errors for wrong input", {
  expect_error(B("a"))
  expect_error(B(NA_character_))
  expect_error(B(mNAc))
  expect_error(B(mNAc, f))
  expect_error(B(1:2,1:3))
  expect_error(B(m,1:31))
  expect_error(B(mtcars,1:31))
  expect_error(B("a", w = 1))
  expect_error(B(1:2, w = c("a","b")))
  expect_error(B(1:2, w = 1:3))
  expect_error(B(NA_character_, w = 1))
  expect_error(B(mNAc, w = wdat))
  expect_error(B(mNAc, f, wdat))
  expect_error(B(mNA, w = 1:33))
  expect_error(B(mtcNA, w = 1:33))
  expect_error(B(1:2,1:2, 1:3))
  expect_error(B(m,1:32,1:20))
  expect_error(B(mtcars,1:32,1:10))
  expect_error(B(1:2, 1:3, 1:2))
  expect_error(B(m,1:31,1:32))
  expect_error(B(mtcars,1:33,1:32))
})

test_that("B.data.frame method is foolproof", {
  expect_visible(B(wlddev))
  expect_visible(B(wlddev, w = wlddev$year))
  expect_visible(B(wlddev, w = ~year))
  expect_visible(B(wlddev, wlddev$iso3c))
  expect_visible(B(wlddev, ~iso3c))
  expect_visible(B(wlddev, ~iso3c + region))
  expect_visible(B(wlddev, wlddev$iso3c, wlddev$year))
  expect_visible(B(wlddev, ~iso3c, ~year))
  expect_visible(B(wlddev, cols = 9:12))
  expect_visible(B(wlddev, w = wlddev$year, cols = 9:12))
  expect_visible(B(wlddev, w = ~year, cols = 9:12))
  expect_visible(B(wlddev, wlddev$iso3c, cols = 9:12))
  expect_visible(B(wlddev, ~iso3c, cols = 9:12))
  expect_visible(B(wlddev, wlddev$iso3c, wlddev$year, cols = 9:12))
  expect_visible(B(wlddev, ~iso3c, ~year, cols = 9:12))
  expect_visible(B(wlddev, cols = c("PCGDP","LIFEEX")))
  expect_visible(B(wlddev, w = wlddev$year, cols = c("PCGDP","LIFEEX")))
  expect_visible(B(wlddev, w = ~year, cols = c("PCGDP","LIFEEX")))
  expect_visible(B(wlddev, wlddev$iso3c, cols = c("PCGDP","LIFEEX")))
  expect_visible(B(wlddev, ~iso3c, cols = c("PCGDP","LIFEEX")))
  expect_visible(B(wlddev, wlddev$iso3c, wlddev$year, cols = c("PCGDP","LIFEEX")))
  expect_visible(B(wlddev, ~iso3c, ~year, cols = c("PCGDP","LIFEEX")))

  expect_error(B(wlddev, cols = NULL))
  expect_error(B(wlddev, w = wlddev$year, cols = NULL))
  expect_error(B(wlddev, w = ~year, cols = NULL))
  expect_error(B(wlddev, wlddev$iso3c, cols = NULL))
  expect_error(B(wlddev, ~iso3c, cols = NULL))
  expect_error(B(wlddev, wlddev$iso3c, wlddev$year, cols = NULL))
  expect_error(B(wlddev, ~iso3c, ~year, cols = NULL))
  expect_error(B(wlddev, cols = 9:13))
  expect_error(B(wlddev, w = wlddev$year, cols = 9:13))
  expect_error(B(wlddev, w = ~year, cols = 9:13))
  expect_error(B(wlddev, wlddev$iso3c, cols = 9:13))
  expect_error(B(wlddev, ~iso3c, cols = 9:13))
  expect_error(B(wlddev, wlddev$iso3c, wlddev$year, cols = 9:13))
  expect_error(B(wlddev, ~iso3c, ~year, cols = 9:13))
  expect_error(B(wlddev, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(B(wlddev, w = wlddev$year, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(B(wlddev, w = ~year, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(B(wlddev, wlddev$iso3c, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(B(wlddev, ~iso3c, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(B(wlddev, wlddev$iso3c, wlddev$year, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(B(wlddev, ~iso3c, ~year, cols = c("PCGDP","LIFEEX","bla")))

  expect_error(B(wlddev, w = mtcars))
  expect_error(B(wlddev, w = 4))
  expect_error(B(wlddev, w = "year"))
  expect_error(B(wlddev, w = ~year2))
  expect_error(B(wlddev, w = ~year + region))
  expect_error(B(wlddev, mtcars))
  expect_error(B(wlddev, 2))
  expect_error(B(wlddev, "iso3c"))
  expect_error(B(wlddev, ~iso3c2))
  expect_error(B(wlddev, ~iso3c + bla))
  expect_error(B(wlddev, mtcars$mpg, mtcars$cyl))
  expect_error(B(wlddev, 2, 4))
  expect_error(B(wlddev, ~iso3c2, ~year2))
  expect_error(B(wlddev, cols = ~bla))
  expect_error(B(wlddev, w = ~bla, cols = 9:12))
  expect_error(B(wlddev, w = 4, cols = 9:12))
  expect_error(B(wlddev, w = "year", cols = 9:12))
  expect_error(B(wlddev, w = ~yewar, cols = 9:12))
  expect_error(B(wlddev, mtcars$mpg, cols = 9:12))
  expect_error(B(wlddev, ~iso3c + ss, cols = 9:12))
  expect_error(B(wlddev, 2, cols = 9:12))
  expect_error(B(wlddev, "iso3c", cols = 9:12))
  expect_error(B(wlddev, wlddev$iso3c, ~year + bla, cols = 9:12))
  expect_error(B(wlddev, ~iso3c3, ~year, cols = 9:12))
  expect_error(B(wlddev, cols = c("PC3GDP","LIFEEX")))
})


# fwithin

test_that("fwithin performs like baseW", {
  expect_equal(fwithin(NA), as.double(baseW(NA)))
  expect_equal(fwithin(NA, na.rm = FALSE), as.double(baseW(NA)))
  expect_equal(fwithin(1), baseW(1, na.rm = TRUE))
  expect_equal(fwithin(1:3), baseW(1:3, na.rm = TRUE))
  expect_equal(fwithin(-1:1), baseW(-1:1, na.rm = TRUE))
  expect_equal(fwithin(1, na.rm = FALSE), baseW(1))
  expect_equal(fwithin(1:3, na.rm = FALSE), baseW(1:3))
  expect_equal(fwithin(-1:1, na.rm = FALSE), baseW(-1:1))
  expect_equal(fwithin(x), baseW(x, na.rm = TRUE))
  expect_equal(fwithin(x, na.rm = FALSE), baseW(x))
  expect_equal(fwithin(xNA, na.rm = FALSE), baseW(xNA))
  expect_equal(fwithin(xNA), baseW(xNA, na.rm = TRUE))
  expect_equal(qM(fwithin(mtcars)), fwithin(m))
  expect_equal(fwithin(m), dapply(m, baseW, na.rm = TRUE))
  expect_equal(fwithin(m, na.rm = FALSE), dapply(m, baseW))
  expect_equal(fwithin(mNA, na.rm = FALSE), dapply(mNA, baseW))
  expect_equal(fwithin(mNA), dapply(mNA, baseW, na.rm = TRUE))
  expect_equal(fwithin(x, f), BY(x, f, baseW, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fwithin(x, f, na.rm = FALSE), BY(x, f, baseW, use.g.names = FALSE))
  expect_equal(fwithin(xNA, f, na.rm = FALSE), BY(xNA, f, baseW, use.g.names = FALSE))
  expect_equal(fwithin(xNA, f), BY(xNA, f, baseW, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fwithin(m, g), BY(m, g, baseW, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fwithin(m, g, na.rm = FALSE), BY(m, g, baseW, use.g.names = FALSE))
  expect_equal(fwithin(mNA, g, na.rm = FALSE), BY(mNA, g, baseW, use.g.names = FALSE))
  expect_equal(fwithin(mNA, g), BY(mNA, g, baseW, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fwithin(mtcars, g), BY(mtcars, g, baseW, na.rm = TRUE, use.g.names = FALSE))
  expect_equal(fwithin(mtcars, g, na.rm = FALSE), BY(mtcars, g, baseW, use.g.names = FALSE))
  expect_equal(fwithin(mtcNA, g, na.rm = FALSE), BY(mtcNA, g, baseW, use.g.names = FALSE))
  expect_equal(fwithin(mtcNA, g), BY(mtcNA, g, baseW, na.rm = TRUE, use.g.names = FALSE))
})

test_that("fwithin with custom mean performs like baseW (defined above)", {
  expect_equal(fwithin(x, mean = 4.8456), baseW(x, na.rm = TRUE, mean = 4.8456))
  expect_equal(fwithin(x, na.rm = FALSE, mean = 4.8456), baseW(x, mean = 4.8456))
  expect_equal(fwithin(xNA, na.rm = FALSE, mean = 4.8456), baseW(xNA, mean = 4.8456))
  expect_equal(fwithin(xNA, mean = 4.8456), baseW(xNA, na.rm = TRUE, mean = 4.8456))
  expect_equal(qM(fwithin(mtcars, mean = 4.8456)), fwithin(m, mean = 4.8456))
  expect_equal(fwithin(m, mean = 4.8456), dapply(m, baseW, na.rm = TRUE, mean = 4.8456))
  expect_equal(fwithin(m, na.rm = FALSE, mean = 4.8456), dapply(m, baseW, mean = 4.8456))
  expect_equal(fwithin(mNA, na.rm = FALSE, mean = 4.8456), dapply(mNA, baseW, mean = 4.8456))
  expect_equal(fwithin(mNA, mean = 4.8456), dapply(mNA, baseW, na.rm = TRUE, mean = 4.8456))
  expect_equal(fwithin(x, f, mean = 4.8456), BY(x, f, baseW, na.rm = TRUE, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(x, f, na.rm = FALSE, mean = 4.8456), BY(x, f, baseW, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(xNA, f, na.rm = FALSE, mean = 4.8456), BY(xNA, f, baseW, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(xNA, f, mean = 4.8456), BY(xNA, f, baseW, na.rm = TRUE, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(m, g, mean = 4.8456), BY(m, g, baseW, na.rm = TRUE, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(m, g, na.rm = FALSE, mean = 4.8456), BY(m, g, baseW, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(mNA, g, na.rm = FALSE, mean = 4.8456), BY(mNA, g, baseW, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(mNA, g, mean = 4.8456), BY(mNA, g, baseW, na.rm = TRUE, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(mtcars, g, mean = 4.8456), BY(mtcars, g, baseW, na.rm = TRUE, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(mtcars, g, na.rm = FALSE, mean = 4.8456), BY(mtcars, g, baseW, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(mtcNA, g, na.rm = FALSE, mean = 4.8456), BY(mtcNA, g, baseW, use.g.names = FALSE, mean = 4.8456))
  expect_equal(fwithin(mtcNA, g, mean = 4.8456), BY(mtcNA, g, baseW, na.rm = TRUE, use.g.names = FALSE, mean = 4.8456))
})

test_that("Centering on overall mean performs as indended", {
  expect_equal(fwithin(x, f, mean = "overall.mean"), BY(x, f, baseW, na.rm = TRUE, use.g.names = FALSE) + ave(x))
  expect_equal(fwithin(x, f, na.rm = FALSE, mean = "overall.mean"), BY(x, f, baseW, use.g.names = FALSE) + ave(x))
  # expect_equal(fwithin(xNA, f, na.rm = FALSE, mean = "overall.mean"), BY(xNA, f, baseW, use.g.names = FALSE) + B(xNA)) # Not the same !!
  expect_equal(fwithin(xNA, f, mean = "overall.mean"), BY(xNA, f, baseW, na.rm = TRUE, use.g.names = FALSE) + B(xNA))
  expect_equal(fwithin(m, g, mean = "overall.mean"), BY(m, g, baseW, na.rm = TRUE, use.g.names = FALSE) + B(m))
  expect_equal(fwithin(m, g, na.rm = FALSE, mean = "overall.mean"), BY(m, g, baseW, use.g.names = FALSE) + B(m))
  # expect_equal(fwithin(mNA, g, na.rm = FALSE, mean = "overall.mean"), BY(mNA, g, baseW, use.g.names = FALSE) + B(mNA))
  expect_equal(fwithin(mNA, g, mean = "overall.mean"), BY(mNA, g, baseW, na.rm = TRUE, use.g.names = FALSE) + B(mNA))
  expect_equal(fwithin(mtcars, g, mean = "overall.mean"), BY(mtcars, g, baseW, na.rm = TRUE, use.g.names = FALSE) + B(mtcars))
  expect_equal(fwithin(mtcars, g, na.rm = FALSE, mean = "overall.mean"), BY(mtcars, g, baseW, use.g.names = FALSE) + B(mtcars))
  # expect_equal(fwithin(mtcNA, g, na.rm = FALSE, mean = "overall.mean"), BY(mtcNA, g, baseW, use.g.names = FALSE) + B(mtcNA))
  expect_equal(fwithin(mtcNA, g, mean = "overall.mean"), BY(mtcNA, g, baseW, na.rm = TRUE, use.g.names = FALSE)+ B(mtcNA))
})

test_that("fwithin performs like fwithin with weights all equal", {
  expect_equal(fwithin(NA), fwithin(NA, w = 0.99999999))
  expect_equal(fwithin(NA, na.rm = FALSE), fwithin(NA, w = 2.946, na.rm = FALSE))
  expect_equal(fwithin(1), fwithin(1, w = 3))
  expect_equal(fwithin(1:3), fwithin(1:3, w = rep(0.999,3)))
  expect_equal(fwithin(-1:1), fwithin(-1:1, w = rep(4.2,3)))
  expect_equal(fwithin(1, na.rm = FALSE), fwithin(1, w = 5, na.rm = FALSE))
  expect_equal(fwithin(1:3, na.rm = FALSE), fwithin(1:3, w = rep(1.44565, 3), na.rm = FALSE))
  expect_equal(fwithin(-1:1, na.rm = FALSE), fwithin(-1:1, w = rep(1.44565, 3), na.rm = FALSE))
  expect_equal(fwithin(x), fwithin(x, w = rep(1,100)))
  expect_equal(fwithin(x, na.rm = FALSE), fwithin(x, w = rep(1.44565, 100), na.rm = FALSE))
  expect_equal(fwithin(xNA, na.rm = FALSE), fwithin(xNA, w = rep(4.676587, 100), na.rm = FALSE))
  expect_equal(fwithin(xNA), fwithin(xNA, w = rep(4.676587, 100)))
  expect_equal(fwithin(m), fwithin(m, w = rep(6587.3454, 32)))
  expect_equal(fwithin(m, na.rm = FALSE), fwithin(m, w = rep(6587.3454, 32), na.rm = FALSE))
  expect_equal(fwithin(mNA, na.rm = FALSE), fwithin(mNA, w = rep(6587.3454, 32), na.rm = FALSE))
  expect_equal(fwithin(mNA), fwithin(mNA, w = rep(6587.3454, 32)))
  expect_equal(fwithin(mtcars), fwithin(mtcars, w = rep(6787.3454, 32)))
  expect_equal(fwithin(mtcars, na.rm = FALSE), fwithin(mtcars, w = rep(6787.3454, 32), na.rm = FALSE))
  expect_equal(fwithin(mtcNA, na.rm = FALSE), fwithin(mtcNA, w = rep(6787.3454, 32), na.rm = FALSE))
  expect_equal(fwithin(mtcNA), fwithin(mtcNA, w = rep(6787.3454, 32)))
  expect_equal(fwithin(x, f), fwithin(x, f, rep(546.78,100)))
  expect_equal(fwithin(x, f, na.rm = FALSE), fwithin(x, f, rep(5.88,100), na.rm = FALSE))
  expect_equal(fwithin(xNA, f, na.rm = FALSE), fwithin(xNA, f, rep(52.7,100), na.rm = FALSE))
  expect_equal(fwithin(xNA, f), fwithin(xNA, f, rep(5997456,100)))
  expect_equal(fwithin(m, g), fwithin(m, g, rep(546.78,32)))
  expect_equal(fwithin(m, g, na.rm = FALSE), fwithin(m, g, rep(0.0001,32), na.rm = FALSE))
  expect_equal(fwithin(mNA, g, na.rm = FALSE), fwithin(mNA, g, rep(5.7,32), na.rm = FALSE))
  expect_equal(fwithin(mNA, g), fwithin(mNA, g, rep(1.1,32)))
  expect_equal(fwithin(mtcars, g), fwithin(mtcars, g, rep(53,32)))
  expect_equal(fwithin(mtcars, g, na.rm = FALSE), fwithin(mtcars, g, rep(546.78,32), na.rm = FALSE))
  expect_equal(fwithin(mtcNA, g, na.rm = FALSE), fwithin(mtcNA, g, rep(0.999999,32), na.rm = FALSE))
  expect_equal(fwithin(mtcNA, g), fwithin(mtcNA, g, rep(999.9999,32)))
})

test_that("fwithin with weights performs like wbaseW (defined above)", {
  # complete weights
  expect_equal(fwithin(NA, w = 1), wbaseW(NA, 1))
  expect_equal(fwithin(NA, w = 1, na.rm = FALSE), wbaseW(NA, 1))
  expect_equal(fwithin(1, w = 1), wbaseW(1, w = 1))
  expect_equal(fwithin(1:3, w = 1:3), wbaseW(1:3, 1:3))
  expect_equal(fwithin(-1:1, w = 1:3), wbaseW(-1:1, 1:3))
  expect_equal(fwithin(1, w = 1, na.rm = FALSE), wbaseW(1, 1))
  expect_equal(fwithin(1:3, w = c(0.99,3454,1.111), na.rm = FALSE), wbaseW(1:3, c(0.99,3454,1.111)))
  expect_equal(fwithin(-1:1, w = 1:3, na.rm = FALSE), wbaseW(-1:1, 1:3))
  expect_equal(fwithin(x, w = w), wbaseW(x, w))
  expect_equal(fwithin(x, w = w, na.rm = FALSE), wbaseW(x, w))
  expect_equal(fwithin(xNA, w = w, na.rm = FALSE), wbaseW(xNA, w))
  expect_equal(fwithin(xNA, w = w), wbaseW(xNA, w, na.rm = TRUE))
  expect_equal(qM(fwithin(mtcars, w = wdat)), fwithin(m, w = wdat))
  expect_equal(fwithin(m, w = wdat), dapply(m, wbaseW, wdat, na.rm = TRUE))
  expect_equal(fwithin(m, w = wdat, na.rm = FALSE), dapply(m, wbaseW, wdat))
  expect_equal(fwithin(mNA, w = wdat, na.rm = FALSE), dapply(mNA, wbaseW, wdat))
  expect_equal(fwithin(mNA, w = wdat), dapply(mNA, wbaseW, wdat, na.rm = TRUE))
  expect_equal(fwithin(x, f, w), unlist(Map(wbaseW, split(x, f), split(w, f)), use.names = FALSE))
  expect_equal(fwithin(x, f, w, na.rm = FALSE), unlist(Map(wbaseW, split(x, f), split(w, f)), use.names = FALSE))
  expect_equal(fwithin(xNA, f, w, na.rm = FALSE), unlist(Map(wbaseW, split(xNA, f), split(w, f)), use.names = FALSE))
  expect_equal(fwithin(xNA, f, w), unlist(Map(wbaseW, split(xNA, f), split(w, f), na.rm = TRUE), use.names = FALSE))
  # missing weights
  expect_equal(fwithin(NA, w = NA), wbaseW(NA, NA))
  expect_equal(fwithin(NA, w = NA, na.rm = FALSE), wbaseW(NA, NA))
  expect_equal(fwithin(1, w = NA), wbaseW(1, w = NA))
  expect_equal(fwithin(1:3, w = c(NA,1:2)), wbaseW(1:3, c(NA,1:2), na.rm = TRUE))
  expect_equal(fwithin(-1:1, w = c(NA,1:2)), wbaseW(-1:1, c(NA,1:2), na.rm = TRUE))
  expect_equal(fwithin(1, w = NA, na.rm = FALSE), wbaseW(1, NA))
  expect_equal(fwithin(1:3, w = c(NA,1:2), na.rm = FALSE), wbaseW(1:3, c(NA,1:2)))
  expect_equal(fwithin(-1:1, w = c(NA,1:2), na.rm = FALSE), wbaseW(-1:1, c(NA,1:2)))
  expect_equal(fwithin(x, w = wNA), wbaseW(x, wNA, na.rm = TRUE))
  expect_equal(fwithin(x, w = wNA, na.rm = FALSE), wbaseW(x, wNA))
  expect_equal(fwithin(xNA, w = wNA, na.rm = FALSE), wbaseW(xNA, wNA))
  expect_equal(fwithin(xNA, w = wNA), wbaseW(xNA, wNA, na.rm = TRUE))
  expect_equal(qM(fwithin(mtcars, w = wdatNA)), fwithin(m, w = wdatNA))
  expect_equal(fwithin(m, w = wdatNA), dapply(m, wbaseW, wdatNA, na.rm = TRUE))
  expect_equal(fwithin(m, w = wdatNA, na.rm = FALSE), dapply(m, wbaseW, wdatNA))
  expect_equal(fwithin(mNA, w = wdatNA, na.rm = FALSE), dapply(mNA, wbaseW, wdatNA))
  expect_equal(fwithin(mNA, w = wdatNA), dapply(mNA, wbaseW, wdatNA, na.rm = TRUE))
  expect_equal(fwithin(x, f, wNA), unlist(Map(wbaseW, split(x, f), split(wNA, f), na.rm = TRUE), use.names = FALSE))
  expect_equal(fwithin(x, f, wNA, na.rm = FALSE), unlist(Map(wbaseW, split(x, f), split(wNA, f)), use.names = FALSE))
  expect_equal(fwithin(xNA, f, wNA, na.rm = FALSE), unlist(Map(wbaseW, split(xNA, f), split(wNA, f)), use.names = FALSE))
  expect_equal(fwithin(xNA, f, wNA), unlist(Map(wbaseW, split(xNA, f), split(wNA, f), na.rm = TRUE), use.names = FALSE))
})

test_that("fwithin with custom mean and weights performs like wbaseW (defined above)", {
  # complete weights
  expect_equal(fwithin(x, mean = 4.8456, w = w), wbaseW(x, na.rm = TRUE, mean = 4.8456, w = w))
  expect_equal(fwithin(x, na.rm = FALSE, mean = 4.8456, w = w), wbaseW(x, mean = 4.8456, w = w))
  expect_equal(fwithin(xNA, na.rm = FALSE, mean = 4.8456, w = w), wbaseW(xNA, mean = 4.8456, w = w))
  expect_equal(fwithin(xNA, mean = 4.8456, w = w), wbaseW(xNA, na.rm = TRUE, mean = 4.8456, w = w))
  expect_equal(qM(fwithin(mtcars, mean = 4.8456, w = wdat)), fwithin(m, mean = 4.8456, w = wdat))
  expect_equal(fwithin(m, mean = 4.8456, w = wdat), dapply(m, wbaseW, na.rm = TRUE, mean = 4.8456, w = wdat))
  expect_equal(fwithin(m, na.rm = FALSE, mean = 4.8456, w = wdat), dapply(m, wbaseW, mean = 4.8456, w = wdat))
  expect_equal(fwithin(mNA, na.rm = FALSE, mean = 4.8456, w = wdat), dapply(mNA, wbaseW, mean = 4.8456, w = wdat))
  expect_equal(fwithin(mNA, mean = 4.8456, w = wdat), dapply(mNA, wbaseW, na.rm = TRUE, mean = 4.8456, w = wdat))
  expect_equal(fwithin(x, f, w, mean = 4.8456), unlist(Map(wbaseW, split(x, f), split(w, f), na.rm = TRUE, mean = 4.8456), use.names = FALSE))
  expect_equal(fwithin(x, f, w, mean = 4.8456, na.rm = FALSE), unlist(Map(wbaseW, split(x, f), split(w, f), mean = 4.8456), use.names = FALSE))
  expect_equal(fwithin(xNA, f, w, mean = 4.8456, na.rm = FALSE), unlist(Map(wbaseW, split(xNA, f), split(w, f), mean = 4.8456), use.names = FALSE))
  expect_equal(fwithin(xNA, f, w, mean = 4.8456), unlist(Map(wbaseW, split(xNA, f), split(w, f), na.rm = TRUE, mean = 4.8456), use.names = FALSE))
  # missing weights
  expect_equal(fwithin(x, mean = 4.8456, w = wNA), wbaseW(x, na.rm = TRUE, mean = 4.8456, w = wNA))
  expect_equal(fwithin(x, na.rm = FALSE, mean = 4.8456, w = wNA), wbaseW(x, mean = 4.8456, w = wNA))
  expect_equal(fwithin(xNA, na.rm = FALSE, mean = 4.8456, w = wNA), wbaseW(xNA, mean = 4.8456, w = wNA))
  expect_equal(fwithin(xNA, mean = 4.8456, w = wNA), wbaseW(xNA, na.rm = TRUE, mean = 4.8456, w = wNA))
  expect_equal(qM(fwithin(mtcars, mean = 4.8456, w = wdatNA)), fwithin(m, mean = 4.8456, w = wdatNA))
  expect_equal(fwithin(m, mean = 4.8456, w = wdatNA), dapply(m, wbaseW, na.rm = TRUE, mean = 4.8456, w = wdatNA))
  expect_equal(fwithin(m, na.rm = FALSE, mean = 4.8456, w = wdatNA), dapply(m, wbaseW, mean = 4.8456, w = wdatNA))
  expect_equal(fwithin(mNA, na.rm = FALSE, mean = 4.8456, w = wdatNA), dapply(mNA, wbaseW, mean = 4.8456, w = wdatNA))
  expect_equal(fwithin(mNA, mean = 4.8456, w = wdatNA), dapply(mNA, wbaseW, na.rm = TRUE, mean = 4.8456, w = wdatNA))
  expect_equal(fwithin(x, f, wNA, mean = 4.8456), unlist(Map(wbaseW, split(x, f), split(wNA, f), na.rm = TRUE, mean = 4.8456), use.names = FALSE))
  expect_equal(fwithin(x, f, wNA, mean = 4.8456, na.rm = FALSE), unlist(Map(wbaseW, split(x, f), split(wNA, f), mean = 4.8456), use.names = FALSE))
  expect_equal(fwithin(xNA, f, wNA, mean = 4.8456, na.rm = FALSE), unlist(Map(wbaseW, split(xNA, f), split(wNA, f), mean = 4.8456), use.names = FALSE))
  expect_equal(fwithin(xNA, f, wNA, mean = 4.8456), unlist(Map(wbaseW, split(xNA, f), split(wNA, f), na.rm = TRUE, mean = 4.8456), use.names = FALSE))
})

test_that("Weighted centering on overall weighted mean performs as indended", {
  # complete weights
  expect_equal(fwithin(x, f, w, mean = "overall.mean"), as.numeric(mapply(wbaseW, split(x, f), split(w, f), na.rm = TRUE)) + B(x, w = w))
  expect_equal(fwithin(x, f, w, na.rm = FALSE, mean = "overall.mean"), as.numeric(mapply(wbaseW, split(x, f), split(w, f))) + B(x, w = w))
  # expect_equal(fwithin(xNA, f, w, na.rm = FALSE, mean = "overall.mean"), as.numeric(mapply(wbaseW, split(xNA, f), split(w, f))) + B(xNA, w = w)) # Not the same !!
  expect_equal(fwithin(xNA, f, w, mean = "overall.mean"), as.numeric(mapply(wbaseW, split(xNA, f), split(w, f), na.rm = TRUE)) + B(xNA, w = w))
})
# Do more than this to test the rest ...

test_that("fwithin performs numerically stable", {
  expect_true(all_obj_equal(replicate(50, fwithin(1), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(NA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(NA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, f), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, f, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, f, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, f), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, g), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, g, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, g, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, g), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, g), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, g, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, g, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, g), simplify = FALSE)))
})

test_that("fwithin with complete weights performs numerically stable", {
  expect_true(all_obj_equal(replicate(50, fwithin(1, w = 1), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(NA, w = 1), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(NA, w = 1, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, w = w), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, w = w, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, w = w, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, w = w), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, w = wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, w = wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, w = wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, w = wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, w = wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, w = wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, w = wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, w = wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, f, w), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, f, w, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, f, w, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, f, w), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, g, wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, g, wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, g, wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, g, wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, g, wdat), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, g, wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, g, wdat, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, g, wdat), simplify = FALSE)))
})

test_that("fwithin with missing weights performs numerically stable", {
  expect_true(all_obj_equal(replicate(50, fwithin(1, w = NA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(NA, w = NA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(NA, w = NA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, w = wNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, w = wNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, w = wNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, w = wNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, w = wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, w = wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, w = wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, w = wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, w = wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, w = wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, w = wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, w = wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, f, wNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(x, f, wNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, f, wNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(xNA, f, wNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, g, wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(m, g, wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, g, wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mNA, g, wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, g, wdatNA), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcars, g, wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, g, wdatNA, na.rm = FALSE), simplify = FALSE)))
  expect_true(all_obj_equal(replicate(50, fwithin(mtcNA, g, wdatNA), simplify = FALSE)))
})

test_that("fwithin handles special values in the right way", {
  expect_equal(fwithin(NA), NA_real_)
  expect_equal(fwithin(NaN), NaN)
  expect_equal(fwithin(Inf), NaN)
  expect_equal(fwithin(c(Inf,Inf)), c(NaN,NaN))
  expect_equal(fwithin(-Inf), NaN)
  expect_equal(fwithin(c(-Inf,-Inf)), c(NaN,NaN))
  expect_equal(fwithin(TRUE), 0)
  expect_equal(fwithin(FALSE), 0)
  expect_equal(fwithin(NA, na.rm = FALSE), NA_real_)
  expect_equal(fwithin(NaN, na.rm = FALSE), NaN)
  expect_equal(fwithin(Inf, na.rm = FALSE), NaN)
  expect_equal(fwithin(c(Inf,Inf), na.rm = FALSE), c(NaN,NaN))
  expect_equal(fwithin(-Inf, na.rm = FALSE), NaN)
  expect_equal(fwithin(c(-Inf,-Inf), na.rm = FALSE), c(NaN,NaN))
  expect_equal(fwithin(TRUE, na.rm = FALSE), 0)
  expect_equal(fwithin(FALSE, na.rm = FALSE), 0)
  expect_equal(fwithin(c(1,NA)), c(0,NA_real_))
  expect_equal(fwithin(c(1,NaN)), c(0,NaN))
  expect_equal(fwithin(c(1,Inf)), c(-Inf,NaN))
  expect_equal(fwithin(c(1,-Inf)), c(Inf,NaN))
  expect_equal(fwithin(c(1,Inf), na.rm = FALSE), c(-Inf,NaN))
  expect_equal(fwithin(c(1,-Inf), na.rm = FALSE), c(Inf,NaN))
  expect_equal(fwithin(c(NA,-Inf), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(c(TRUE,TRUE), na.rm = FALSE), c(0,0))
  expect_equal(fwithin(c(1,NA), na.rm = FALSE), c(NA_real_,NA_real_))
})

test_that("fwithin with weights handles special values in the right way", {
  expect_equal(fwithin(NA, w = 1), NA_real_)
  expect_equal(fwithin(NaN, w = 1), NaN)
  expect_equal(fwithin(Inf, w = 1), NaN)
  expect_equal(fwithin(c(Inf,Inf), w = 1:2), c(NaN,NaN))
  expect_equal(fwithin(-Inf, w = 1), NaN)
  expect_equal(fwithin(c(-Inf,-Inf), w = 1:2), c(NaN,NaN))
  expect_equal(fwithin(TRUE, w = 1), 0)
  expect_equal(fwithin(FALSE, w = 1), 0)
  expect_equal(fwithin(NA, w = 1, na.rm = FALSE), NA_real_)
  expect_equal(fwithin(NaN, w = 1, na.rm = FALSE), NaN)
  expect_equal(fwithin(Inf, w = 1, na.rm = FALSE), NaN)
  expect_equal(fwithin(c(Inf,Inf), w = 1:2, na.rm = FALSE), c(NaN,NaN))
  expect_equal(fwithin(-Inf, w = 1, na.rm = FALSE), NaN)
  expect_equal(fwithin(c(-Inf,-Inf), w = 1:2, na.rm = FALSE), c(NaN,NaN))
  expect_equal(fwithin(TRUE, w = 1, na.rm = FALSE), 0)
  expect_equal(fwithin(FALSE, w = 1, na.rm = FALSE), 0)
  expect_equal(fwithin(c(1,NA), w = 1:2), c(0,NA))
  expect_equal(fwithin(c(1,NaN), w = 1:2), c(0,NaN))
  expect_equal(fwithin(c(1,Inf), w = 1:2), c(-Inf,NaN))
  expect_equal(fwithin(c(1,-Inf), w = 1:2), c(Inf,NaN))
  expect_equal(fwithin(c(1,Inf), w = 1:2, na.rm = FALSE), c(-Inf,NaN))
  expect_equal(fwithin(c(1,-Inf), w = 1:2, na.rm = FALSE), c(Inf,NaN))
  expect_equal(fwithin(c(NA,-Inf), w = 1:2, na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(c(FALSE,FALSE), w = 1:2, na.rm = FALSE), c(0,0))
  expect_equal(fwithin(c(1,NA), w = 1:2, na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(1:3, w = c(1,Inf,3)), c(NaN,NaN,NaN))
  expect_equal(fwithin(1:3, w = c(1,-Inf,3)), c(NaN,NaN,NaN))
  expect_equal(fwithin(1:3, w = c(1,Inf,3), na.rm = FALSE), c(NaN,NaN,NaN))
  expect_equal(fwithin(1:3, w = c(1,-Inf,3), na.rm = FALSE), c(NaN,NaN,NaN))

  expect_equal(fwithin(NA, w = NA), NA_real_)
  expect_equal(fwithin(NaN, w = NA), NaN)
  expect_equal(fwithin(Inf, w = NA), NaN)
  expect_equal(fwithin(c(Inf,Inf), w = c(NA,2)), c(NaN,NaN))
  expect_equal(fwithin(-Inf, w = NA), NA_real_)
  expect_equal(fwithin(c(-Inf,-Inf), w = c(NA,2)), c(NaN,NaN))
  expect_equal(fwithin(TRUE, w = NA), NA_real_)
  expect_equal(fwithin(FALSE, w = NA), NA_real_)
  expect_equal(fwithin(NA, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fwithin(NaN, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fwithin(Inf, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fwithin(c(Inf,Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(-Inf, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fwithin(c(-Inf,-Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(TRUE, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fwithin(FALSE, w = NA, na.rm = FALSE), NA_real_)
  expect_equal(fwithin(c(1,NA), w = c(NA,2)), c(NA_real_,NA_real_))
  expect_equal(fwithin(c(1,NaN), w = c(NA,2)), c(NA_real_,NA_real_))
  expect_equal(fwithin(c(1,Inf), w = c(NA,2)), c(-Inf,NaN))
  expect_equal(fwithin(c(1,-Inf), w = c(NA,2)), c(Inf,NaN))
  expect_equal(fwithin(c(1,Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(c(1,-Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(c(NA,-Inf), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(c(FALSE,FALSE), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(c(1,NA), w = c(NA,2), na.rm = FALSE), c(NA_real_,NA_real_))
  expect_equal(fwithin(1:3, w = c(NA,Inf,3)), c(NaN,NaN,NaN))
  expect_equal(fwithin(1:3, w = c(NA,-Inf,3)), c(NaN,NaN,NaN))
  expect_equal(fwithin(1:3, w = c(NA,Inf,3), na.rm = FALSE), c(NA_real_,NA_real_,NA_real_))
  expect_equal(fwithin(1:3, w = c(NA,-Inf,3), na.rm = FALSE), c(NA_real_,NA_real_,NA_real_))
})

test_that("fwithin produces errors for wrong input", {
  expect_error(fwithin("a"))
  expect_error(fwithin(NA_character_))
  expect_error(fwithin(mNAc))
  expect_error(fwithin(mNAc, f))
  expect_error(fwithin(1:2,1:3))
  expect_error(fwithin(m,1:31))
  expect_error(fwithin(mtcars,1:31))
  expect_error(fwithin("a", w = 1))
  expect_error(fwithin(1:2, w = 1:3))
  expect_error(fwithin(NA_character_, w = 1))
  expect_error(fwithin(mNAc, w = wdat))
  expect_error(fwithin(mNAc, f, wdat))
  expect_error(fwithin(mNA, w = 1:33))
  expect_error(fwithin(1:2,1:2, 1:3))
  expect_error(fwithin(m,1:32,1:20))
  expect_error(fwithin(mtcars,1:32,1:10))
  expect_error(fwithin(1:2, w = c("a","b")))
  expect_error(fwithin(wlddev))
  expect_error(fwithin(wlddev, w = wlddev$year))
  expect_error(fwithin(wlddev, wlddev$iso3c))
  expect_error(fwithin(wlddev, wlddev$iso3c, wlddev$year))
})

test_that("fwithin shoots errors for wrong input to mean", {
  expect_error(fwithin(x, mean = FALSE))
  expect_error(fwithin(m, mean = FALSE))
  expect_error(fwithin(mtcars, mean = FALSE))
  expect_error(fwithin(x, mean = "overall.mean"))
  expect_error(fwithin(m, mean = "overall.mean"))
  expect_error(fwithin(mtcars, mean = "overall.mean"))
  expect_error(fwithin(m, mean = fmean(m)))
  expect_error(fwithin(mtcars, mean = fmean(mtcars)))
})


# W

test_that("W produces errors for wrong input", {
  expect_error(W("a"))
  expect_error(W(NA_character_))
  expect_error(W(mNAc))
  expect_error(W(mNAc, f))
  expect_error(W(1:2,1:3))
  expect_error(W(m,1:31))
  expect_error(W(mtcars,1:31))
  expect_error(W("a", w = 1))
  expect_error(W(1:2, w = c("a","b")))
  expect_error(W(1:2, w = 1:3))
  expect_error(W(NA_character_, w = 1))
  expect_error(W(mNAc, w = wdat))
  expect_error(W(mNAc, f, wdat))
  expect_error(W(mNA, w = 1:33))
  expect_error(W(mtcNA, w = 1:33))
  expect_error(W(1:2,1:2, 1:3))
  expect_error(W(m,1:32,1:20))
  expect_error(W(mtcars,1:32,1:10))
  expect_error(W(1:2, 1:3, 1:2))
  expect_error(W(m,1:31,1:32))
  expect_error(W(mtcars,1:33,1:32))
})

test_that("W.data.frame method is foolproof", {
  expect_visible(W(wlddev))
  expect_visible(W(wlddev, w = wlddev$year))
  expect_visible(W(wlddev, w = ~year))
  expect_visible(W(wlddev, wlddev$iso3c))
  expect_visible(W(wlddev, ~iso3c))
  expect_visible(W(wlddev, ~iso3c + region))
  expect_visible(W(wlddev, wlddev$iso3c, wlddev$year))
  expect_visible(W(wlddev, ~iso3c, ~year))
  expect_visible(W(wlddev, cols = 9:12))
  expect_visible(W(wlddev, w = wlddev$year, cols = 9:12))
  expect_visible(W(wlddev, w = ~year, cols = 9:12))
  expect_visible(W(wlddev, wlddev$iso3c, cols = 9:12))
  expect_visible(W(wlddev, ~iso3c, cols = 9:12))
  expect_visible(W(wlddev, wlddev$iso3c, wlddev$year, cols = 9:12))
  expect_visible(W(wlddev, ~iso3c, ~year, cols = 9:12))
  expect_visible(W(wlddev, cols = c("PCGDP","LIFEEX")))
  expect_visible(W(wlddev, w = wlddev$year, cols = c("PCGDP","LIFEEX")))
  expect_visible(W(wlddev, w = ~year, cols = c("PCGDP","LIFEEX")))
  expect_visible(W(wlddev, wlddev$iso3c, cols = c("PCGDP","LIFEEX")))
  expect_visible(W(wlddev, ~iso3c, cols = c("PCGDP","LIFEEX")))
  expect_visible(W(wlddev, wlddev$iso3c, wlddev$year, cols = c("PCGDP","LIFEEX")))
  expect_visible(W(wlddev, ~iso3c, ~year, cols = c("PCGDP","LIFEEX")))

  expect_error(W(wlddev, cols = NULL))
  expect_error(W(wlddev, w = wlddev$year, cols = NULL))
  expect_error(W(wlddev, w = ~year, cols = NULL))
  expect_error(W(wlddev, wlddev$iso3c, cols = NULL))
  expect_error(W(wlddev, ~iso3c, cols = NULL))
  expect_error(W(wlddev, wlddev$iso3c, wlddev$year, cols = NULL))
  expect_error(W(wlddev, ~iso3c, ~year, cols = NULL))
  expect_error(W(wlddev, cols = 9:13))
  expect_error(W(wlddev, w = wlddev$year, cols = 9:13))
  expect_error(W(wlddev, w = ~year, cols = 9:13))
  expect_error(W(wlddev, wlddev$iso3c, cols = 9:13))
  expect_error(W(wlddev, ~iso3c, cols = 9:13))
  expect_error(W(wlddev, wlddev$iso3c, wlddev$year, cols = 9:13))
  expect_error(W(wlddev, ~iso3c, ~year, cols = 9:13))
  expect_error(W(wlddev, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(W(wlddev, w = wlddev$year, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(W(wlddev, w = ~year, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(W(wlddev, wlddev$iso3c, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(W(wlddev, ~iso3c, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(W(wlddev, wlddev$iso3c, wlddev$year, cols = c("PCGDP","LIFEEX","bla")))
  expect_error(W(wlddev, ~iso3c, ~year, cols = c("PCGDP","LIFEEX","bla")))

  expect_error(W(wlddev, w = mtcars))
  expect_error(W(wlddev, w = 4))
  expect_error(W(wlddev, w = "year"))
  expect_error(W(wlddev, w = ~year2))
  expect_error(W(wlddev, w = ~year + region))
  expect_error(W(wlddev, mtcars))
  expect_error(W(wlddev, 2))
  expect_error(W(wlddev, "iso3c"))
  expect_error(W(wlddev, ~iso3c2))
  expect_error(W(wlddev, ~iso3c + bla))
  expect_error(W(wlddev, mtcars$mpg, mtcars$cyl))
  expect_error(W(wlddev, 2, 4))
  expect_error(W(wlddev, ~iso3c2, ~year2))
  expect_error(W(wlddev, cols = ~bla))
  expect_error(W(wlddev, w = ~bla, cols = 9:12))
  expect_error(W(wlddev, w = 4, cols = 9:12))
  expect_error(W(wlddev, w = "year", cols = 9:12))
  expect_error(W(wlddev, w = ~yewar, cols = 9:12))
  expect_error(W(wlddev, mtcars$mpg, cols = 9:12))
  expect_error(W(wlddev, ~iso3c + ss, cols = 9:12))
  expect_error(W(wlddev, 2, cols = 9:12))
  expect_error(W(wlddev, "iso3c", cols = 9:12))
  expect_error(W(wlddev, wlddev$iso3c, ~year + bla, cols = 9:12))
  expect_error(W(wlddev, ~iso3c3, ~year, cols = 9:12))
  expect_error(W(wlddev, cols = c("PC3GDP","LIFEEX")))
})
