% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classiKnn.R
\name{classiKnn}
\alias{classiKnn}
\title{Create a knn estimator for functional data classification.}
\usage{
classiKnn(classes, fdata, grid = 1:ncol(fdata), knn = 1L,
  metric = "Euclidean", nderiv = 0L, derived = FALSE,
  deriv.method = "base.diff", custom.metric = function(x, y, ...) {    
  return(sqrt(sum((x - y)^2))) }, ...)
}
\arguments{
\item{classes}{[\code{factor(nrow(fdata))}]\cr
factor of length \code{nrow(fdata)} containing the classes of the observations.}

\item{fdata}{[\code{matrix}]\cr
matrix containing the functional observations as rows.}

\item{grid}{[\code{numeric(ncol(fdata))}]\cr
numeric vector of length \code{ncol(fdata)} containing the grid on which the functional observations were
evaluated.}

\item{knn}{[\code{integer(1)}]\cr
number of nearest neighbors to use in the k nearest neighbor algorithm.}

\item{metric}{[\code{character(1)}]\cr
character string specifying the (semi-)metric to be used.
For a an overview of what is available see the
\code{method} argument in \code{\link{computeDistMat}}. For a full list
execute \code{\link{metricChoices}()}.}

\item{nderiv}{[\code{integer(1)}]\cr
order of derivation on which the metric shall be computed.
The default is 0L.}

\item{derived}{[\code{logical(1)}]\cr
Is the data given in \code{fdata} already derived? Default is set to \code{FALSE},
which will lead to numerical derivation if \code{nderiv >= 1L} by applying
\code{\link[fda]{deriv.fd}} on a \code{\link[fda]{Data2fd}} representation of
\code{fdata}.}

\item{deriv.method}{[\code{character(1)}]\cr
character indicate which method should be used for derivation. Currently
implemented are \code{"base.diff"}, the default, and \code{"fda.deriv.fd"}.
\code{"base.diff"} uses the method \code{base::\link[base]{diff}} for equidistant measures
without missing values, which is faster than transforming the data into the
class \code{\link[fda]{fd}} and deriving this using \code{fda::\link[fda]{deriv.fd}}.
The second variant implies smoothing, which can be preferable for calculating
high order derivatives.}

\item{custom.metric}{[\code{function(x, y, ...)}]\cr
only used if \code{deriv.method = "custom.method"}.
A function of functional observations
\code{x} and \code{y} returning their distance.
The default is the Euclidean distance.
See how to implement your distance function in \code{\link[proxy]{dist}}.}

\item{...}{further arguments to and from other methods. Hand over additional arguments to
\code{\link{computeDistMat}}, usually additional arguments for the specified
(semi-)metric. Also, if \code{deriv.method == "fda.deriv.fd"} or
\code{fdata} is not observed on a regular grid, additional arguments to
\code{\link{fdataTransform}} can be specified which will be passed on to
\code{\link[fda]{Data2fd}}.}
}
\value{
\code{classiKnn} returns an object of class \code{"classiKnn"}.
This is a  list containing  at least the
following components:
 \describe{
  \item{\code{call}}{the original function call.}
  \item{\code{classes}}{a factor of length nrow(fdata) coding the response of
  the training data set.}
  \item{\code{fdata}}{the raw functional data as a matrix with the individual
  observations as rows.}
  \item{\code{grid}}{numeric vector containing the grid on which \code{fdata}
  is observed)}
  \item{\code{proc.fdata}}{the preprocessed data (missing values interpolated,
  derived and evenly spaced). This data is \code{this.fdataTransform(fdata)}.
  See \code{this.fdataTransform} for more details.}
  \item{\code{knn}}{integer coding the number of nearest neighbors used in the
  k nearest neighbor classification algorithm.}
  \item{\code{metric}}{character string coding the distance metric to be used
  in \code{\link{computeDistMat}}.}
  \item{\code{nderiv}}{integer giving the order of derivation that is applied
  to fdata before computing the distances between the observations.}
  \item{\code{this.fdataTransform}}{preprocessing function taking new data as
  a matrix. It is used to transform \code{fdata} into \code{proc.fdata} and
  is required to preprocess new data in order to predict it. This function
  ensures, that preprocessing (derivation, respacing and interpolation of
  missing values) is done in the exact same way for the original
  training data set and future (test) data sets.}
 }
}
\description{
Creates an efficient k nearest neighbor estimator for functional data
classification. Currently supported distance measures are all \code{metrics}
implemented in \code{\link[proxy]{dist}}
and all semimetrics suggested in Fuchs et al. (2015).
Additionally, all (semi-)metrics can be used on an arbitrary order of derivation.
}
\examples{
# Classification of the Phoneme data
data(Phoneme)
classes = Phoneme[,"target"]

set.seed(123)
# Use 80\% of data as training set and 20\% as test set
train_inds = sample(1:nrow(Phoneme), size = 0.8 * nrow(Phoneme), replace = FALSE)
test_inds = (1:nrow(Phoneme))[!(1:nrow(Phoneme)) \%in\% train_inds]

# create functional data as matrix with observations as rows
fdata = Phoneme[,!colnames(Phoneme) == "target"]

# create k = 3 nearest neighbors classifier with Euclidean distance (default) of the
# first order derivative of the data
mod = classiKnn(classes = classes[train_inds], fdata = fdata[train_inds,],
                 nderiv = 1L, knn = 3L)

# predict the model for the test set
pred = predict(mod, newdata =  fdata[test_inds,], predict.type = "prob")

}
\references{
Fuchs, K., J. Gertheiss, and G. Tutz (2015):
Nearest neighbor ensembles for functional data with interpretable feature selection.
Chemometrics and Intelligent Laboratory Systems 146, 186 - 197.
}
\seealso{
\link{predict.classiKnn}
}
