% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bkcusum.R
\name{bkcusum}
\alias{bkcusum}
\title{Continuous time BK-CUSUM}
\usage{
bkcusum(data, theta, coxphmod, cbaseh, ctimes, h, stoptime, C, pb = FALSE)
}
\arguments{
\item{data}{\code{data.frame} containing the following named columns:
\itemize{
\item \code{entrytime} numeric - time of entry into study,
\item \code{survtime} numeric - time from entry until event,
\item \code{censorid} integer - (optional) censoring indicator (0 = right censored, 1 = observed),
} and optionally additional covariates used for risk-adjustment.}

\item{theta}{expected ln(hazard ratio) \eqn{\theta}{\theta}}

\item{coxphmod}{(optional) a cox proportional hazards regression model as produced by
the function \code{\link[survival:coxph]{coxph()}}. Standard practice: \cr
\code{coxph(Surv(survtime, censorid) ~ covariates, data = data)}. \cr
Alternatively, a list with:
\itemize{
\item $formula (~ covariates)
\item $coefficients (named vector specifying risk adjustment coefficients
for covariates - names must be the same as in $formula and colnames of \code{data}).
}}

\item{cbaseh}{a function which returns the non risk-adjusted cumulative
baseline hazard \eqn{H_0(t)}{H_0(t)}. If \code{cbaseh} is missing but
\code{coxphmod} has been
specified as a survival object, this baseline hazard rate will be determined
using the provided \code{coxphmod}.}

\item{ctimes}{(optional) vector of construction times at which the value of the chart should be
determined. When not specified, the chart is constructed at all failure times.}

\item{h}{(optional) value of the control limit. The chart will only be
constructed until the value of the control limit has been reached or
surpassed.}

\item{stoptime}{(optional) time after which the value of the chart should no
longer be determined. Default = max(failure time). Useful when ctimes
has not been specified.}

\item{C}{(optional) a numeric value indicating how long after entering the study
patients should no longer influence the value of the chart. This is
equivalent to right-censoring every observation at time \code{entrytime} + C.}

\item{pb}{(optional) boolean indicating whether a progress bar should be shown.
Default = FALSE}
}
\value{
An object of class "bkcusum" containing:
\itemize{
\item \code{BK}: list containing
\itemize{
\item $time (times at which chart is constructed),
\item $value (value of the chart at corresponding times),
}
\item \code{stopind}: indicator for whether the chart was stopped by the control limit
\item \code{call}: the call used to obtain output
} There are \code{\link[cgrcusum:plot.bkcusum]{plot}} and
\code{\link[cgrcusum:runlength.bkcusum]{runlength}} methods for "bkcusum" objects.
}
\description{
This function performs the BK-CUSUM procedure based on the
Biswas & Kalbfleisch (2008) CUSUM. For detection purposes, it is sufficient
to only determine the value of the chart at the times of failure. This can be
achieved by leaving \code{ctimes} empty.
}
\details{
The BK-CUSUM can be used to test the hypothesis of an instant change
of fixed size \eqn{e^\theta}{exp(\theta)}
in the subject specific hazard rate from \eqn{h_i(t)}{h_i(t)} to
\eqn{h_i(t) e^\theta}{h_i(t) exp(\theta)}. The parameter \code{C} can be used
to ignore information provided by subjects C time units after their entry
into the study.
The BK-CUSUM is constructed as:
\deqn{G(t) = \max_{0 \leq k \leq t} \left( \theta N(k,t) - \left( e^\theta -1  \right) \Lambda(k,t)  \right)}{G(t) = max_{0 <= k <= t} (\theta N(k,t) - (e^\theta -1) \Lambda(k,t))}
with \eqn{\theta}{\theta} the ln(expected hazard ratio) and \deqn{N(k,t) = N(t) - N(k)}{N(k,t) = N(t)-N(k)}
with \eqn{N(t)}{N(t)} the counting process of all failures at time t and \deqn{\Lambda(k,t) = \Lambda(t) - \Lambda(k)}{\Lambda(k,t) = \Lambda(t) - \Lambda(k)} the
with \eqn{\Lambda(t)}{\Lambda(t)} the summed cumulative intensity of all subjects at time t.
}
\examples{
require(survival)
tdat <- subset(surgerydat, Hosp_num == 14)
tcbaseh <- function(t) chaz_exp(t, lambda = 0.01)
varsanalysis <- c("age", "sex", "BMI")
exprfit <- as.formula(paste("Surv(survtime, censorid) ~" ,paste(varsanalysis, collapse='+')))
tcoxmod <- coxph(exprfit, data= surgerydat)
#Alternatively, cbaseh can be left empty when specifying coxphmod through coxph()
bk <- bkcusum(data = tdat, theta = log(2), coxphmod = tcoxmod, cbaseh = tcbaseh, pb = TRUE)
plot(bk)
}
\references{
Biswas P. and Kalbfleisch J.D. (2008), A risk-adjusted CUSUM in
continuous time based on the Cox Model, \doi{doi:10.1002/sim.3216}
}
\seealso{
\code{\link[cgrcusum]{plot.bkcusum}}, \code{\link[cgrcusum]{runlength.bkcusum}}

Other qcchart: 
\code{\link{bercusum}()},
\code{\link{cgrcusum}()},
\code{\link{funnelplot}()}
}
\author{
Daniel Gomon
}
\concept{qcchart}
