% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/absoluteRisk-cr.R, R/absoluteRisk.R,
%   R/methods.R, R/plot_methods.R
\name{absoluteRisk.CompRisk}
\alias{absoluteRisk.CompRisk}
\alias{absoluteRisk}
\alias{print.absRiskCB}
\alias{plot.absRiskCB}
\title{Compute absolute risks using the fitted hazard function.}
\usage{
absoluteRisk.CompRisk(
  object,
  time,
  newdata,
  method = c("numerical", "montecarlo"),
  nsamp = 100,
  onlyMain = TRUE,
  type = c("CI", "survival"),
  addZero = TRUE
)

absoluteRisk(
  object,
  time,
  newdata,
  method = c("numerical", "montecarlo"),
  nsamp = 100,
  s = c("lambda.1se", "lambda.min"),
  onlyMain = TRUE,
  type = c("CI", "survival"),
  addZero = TRUE,
  ntimes = 100,
  ...
)

\method{print}{absRiskCB}(x, ...)

\method{plot}{absRiskCB}(
  x,
  ...,
  xlab = "time",
  ylab = ifelse(attr(x, "type") == "CI", "cumulative incidence",
    "survival probability"),
  type = "l",
  gg = TRUE,
  id.names,
  legend.title
)
}
\arguments{
\item{object}{Output of function \code{\link{fitSmoothHazard}}.}

\item{time}{A vector of time points at which we should compute the absolute
risks.}

\item{newdata}{Optionally, a data frame in which to look for variables with
which to predict. If omitted, the mean absolute risk is returned.
Alternatively, if \code{newdata = "typical"}, the absolute risk will be
computed at a "typical" covariate profile (see Details).}

\item{method}{Method used for integration. Defaults to \code{"numerical"},
which uses the trapezoidal rule to integrate over all time points together.
The only other option is \code{"montecarlo"}, which implements Monte-Carlo
integration.}

\item{nsamp}{Maximal number of subdivisions (if \code{method = "numerical"})
or number of sampled points (if \code{method = "montecarlo"}).}

\item{onlyMain}{Logical. For competing risks, should we return absolute risks
only for the main event of interest? Defaults to \code{TRUE}.}

\item{type}{Line type. Only used if \code{gg = FALSE}. This argument gets passed
to \code{\link[graphics:matplot]{graphics::matplot()}}. Default: 'l'}

\item{addZero}{Logical. Should we add time = 0 at the beginning of the
output? Defaults to \code{TRUE}.}

\item{s}{Value of the penalty parameter lambda at which predictions are
required (for class \code{cv.glmnet}).}

\item{ntimes}{Number of time points (only used if \code{time} is missing).}

\item{...}{further arguments passed to \code{matplot}. Only used if
\code{gg=FALSE}.}

\item{x}{Fitted object of class \code{absRiskCB}. This is the result from the
\code{\link[=absoluteRisk]{absoluteRisk()}} function.}

\item{xlab}{xaxis label, Default: 'time'}

\item{ylab}{yaxis label. By default, this will use the \code{"type"} attribute of
the \code{absRiskCB} object}

\item{gg}{Logical for whether the \code{ggplot2} package should be used for
plotting. Default: TRUE}

\item{id.names}{Optional character vector used as legend key when \code{gg=TRUE}.
If missing, defaults to V1, V2, ...}

\item{legend.title}{Optional character vector of the legend title. Only used
if \code{gg = FALSE}. Default is \code{'ID'}}
}
\value{
If \code{time} was provided, returns the estimated absolute risk for
the user-supplied covariate profiles. This will be stored in a matrix or a
higher dimensional array, depending on the input (see details). If both
\code{time} and \code{newdata} are missing, returns the original data
with a new column containing the risk estimate at failure times.

A plot of the cumulative incidence or survival curve
}
\description{
Using the output of the function \code{fitSmoothHazard}, we can compute
absolute risks by integrating the fitted hazard function over a time period
and then converting this to an estimated survival for each individual.

Plot method for objects returned by the \code{absoluteRisk}
function. Current plot types are cumulative incidence and survival
functions.
}
\details{
If \code{newdata = "typical"}, we create a typical covariate profile for the
absolute risk computation. This means that we take the median for numerical
and date variables, and we take the most common level for factor variables.

In general, the output will include a column corresponding to the provided
time points. Some modifications of the \code{time} vector are done:
\code{time=0} is added, the time points are ordered, and duplicates are
removed. All these modifications simplify the computations and give an output
that can easily be used to plot risk curves.

If there is no competing risk, the output is a matrix where each column
corresponds to the several covariate profiles, and where each row corresponds
to a time point. If there are competing risks, the output will be a
3-dimensional array, with the third dimension corresponding to the different
events.

The numerical method should be good enough in most situation, but Monte Carlo
integration can give more accurate results when the estimated hazard function
is not smooth (e.g. when modeling with time-varying covariates).
}
\examples{
# Simulate censored survival data for two outcome types
library(data.table)
set.seed(12345)
nobs <- 1000
tlim <- 20

# simulation parameters
b1 <- 200
b2 <- 50

# event type 0-censored, 1-event of interest, 2-competing event
# t observed time/endpoint
# z is a binary covariate
DT <- data.table(z = rbinom(nobs, 1, 0.5))
DT[,`:=` ("t_event" = rweibull(nobs, 1, b1),
          "t_comp" = rweibull(nobs, 1, b2))]
DT[,`:=`("event" = 1 * (t_event < t_comp) + 2 * (t_event >= t_comp),
         "time" = pmin(t_event, t_comp))]
DT[time >= tlim, `:=`("event" = 0, "time" = tlim)]

out_linear <- fitSmoothHazard(event ~ time + z, DT, ratio = 10)

linear_risk <- absoluteRisk(out_linear, time = 10,
                            newdata = data.table("z" = c(0,1)))
# Plot CI curves----
library(ggplot2)
data("brcancer")
mod_cb_tvc <- fitSmoothHazard(cens ~ estrec*log(time) +
                                horTh +
                                age +
                                menostat +
                                tsize +
                                tgrade +
                                pnodes +
                                progrec,
                              data = brcancer,
                              time = "time", ratio = 1)
smooth_risk_brcancer <- absoluteRisk(object = mod_cb_tvc,
                                     newdata = brcancer[c(1,50),])

class(smooth_risk_brcancer)
plot(smooth_risk_brcancer)
}
\seealso{
\code{\link[graphics]{matplot}},
\code{\link[casebase]{absoluteRisk}},
\code{\link[data.table]{as.data.table}}, \code{\link[data.table]{setattr}},
\code{\link[data.table]{melt.data.table}}
}
