% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sampling.R
\name{sampleCaseBase}
\alias{sampleCaseBase}
\title{Create case-base dataset for use in fitting parametric hazard functions}
\usage{
sampleCaseBase(data, time, event, ratio = 10, comprisk = FALSE,
  censored.indicator)
}
\arguments{
\item{data}{a data.frame or data.table containing the source dataset.}

\item{time}{a character string giving the name of the time variable. See Details.}

\item{event}{a character string giving the name of the event variable. See Details.}

\item{ratio}{Integer, giving the ratio of the size of the base series to that of the case series.
Defaults to 10.}

\item{comprisk}{Logical. Indicates whether we have multiple event types and that we want to
consider some of them as competing risks.}

\item{censored.indicator}{a character string of length 1 indicating which value in \code{event}
is the censored. This function will use \code{\link[stats]{relevel}} to set
\code{censored.indicator} as the reference level. This argument is ignored if the \code{event}
variable is a numeric}
}
\value{
The function returns a dataset, with the same format as the source dataset, and where
  each row corresponds to a person-moment sampled from the case or the base series. otherwise)
}
\description{
This function implements the case-base sampling approach described in Hanley and Miettinen, Int J
Biostatistics 2009. It can be used to fit smooth-in-time parametric functions easily, via
logistic regression.
}
\details{
The base series is sampled using a multinomial scheme: individuals are sampled proportionally to
their follow-up time.

It is assumed that \code{data} contains the two columns corresponding to the supplied time and
event variables. If either the \code{time} or \code{event} argument is missing, the function
looks for columns with appropriate-looking names (see \code{\link{checkArgsTimeEvent}}).
}
\examples{
# Simulate censored survival data for two outcome types from exponential distributions
library(data.table)
set.seed(12345)
nobs <- 5000
tlim <- 10

# simulation parameters
b1 <- 200
b2 <- 50

# event type 0-censored, 1-event of interest, 2-competing event
# t observed time/endpoint
# z is a binary covariate
DT <- data.table(z=rbinom(nobs, 1, 0.5))
DT[,`:=` ("t_event" = rweibull(nobs, 1, b1),
          "t_comp" = rweibull(nobs, 1, b2))]
DT[,`:=`("event" = 1 * (t_event < t_comp) + 2 * (t_event >= t_comp),
         "time" = pmin(t_event, t_comp))]
DT[time >= tlim, `:=`("event" = 0, "time" = tlim)]

out <- sampleCaseBase(DT, time = "time", event = "event", comprisk = TRUE)
}
