% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brm_data_chronologize.R
\name{brm_data_chronologize}
\alias{brm_data_chronologize}
\title{Chronologize a dataset}
\usage{
brm_data_chronologize(
  data,
  order = NULL,
  levels = NULL,
  time = attr(data, "brm_time")
)
}
\arguments{
\item{data}{Data frame or tibble with longitudinal data.}

\item{order}{Optional character string with the name of a variable in
the data for ordering the time variable.
Either \code{order} or \code{levels} must be supplied, but not both together.
If \code{order} is supplied,
the levels of \code{data[[order]]} must have a 1:1 correspondence with
those of \code{data[[time]]}, and \code{sort(unique(data[[order]]))} must
reflect the desired order of the levels of \code{data[[time]]}. For example,
suppose you have a CDISC dataset with categorical time variable \code{AVISIT}
and integer variable \code{AVISITN}. Then,
\code{brm_data_chronologize(time = "AVISIT", order = "AVISITN")} will turn
\code{AVISIT} into an ordered factor with levels that respect the ordering
in \code{AVISITN}.}

\item{levels}{Optional character vector of levels of \code{data[[time]]}
in chronological order. Used to turn \code{data[[time]]} into an
ordered factor.
Either \code{order} or \code{levels} must be supplied, but not both together.}

\item{time}{Character string with the name of the discrete time
variable in the data. This is the variable that \code{\link[=brm_data_chronologize]{brm_data_chronologize()}}
turns into an ordered factor. It needs to be specified explicitly
if and only if the \code{data} argument was not produced by a call to
\code{\link[=brm_data]{brm_data()}}.}
}
\value{
A data frame with the time column as an ordered factor.
}
\description{
Convert the discrete time variable into an ordered factor.
}
\details{
Most MMRMs should use an ordered factor for the \code{time} column
in the data. This way, individual time points are treated as
distinct factor levels for the purposes of fixed effect parameterizations
(see the "Contrasts" section), and the explicit ordering ensures
that informative prior archetypes and ARMA-like correlation structures
are expressed correctly. Without the ordering, problems can arise when
character vectors are sorted: e.g. if \code{AVISIT} has levels
\verb{"VISIT1", "VISIT2", ..., "VISIT10"}, then \code{brms} will mistake the
the order of scheduled study visits to be
\verb{"VISIT1", "VISIT10", "VISIT2", ...}, which is not chronological.

You can easily turn
the time variable into an ordered factor using
\code{\link[=brm_data_chronologize]{brm_data_chronologize()}}. Either supply an explicit character vector
of chronologically-ordered factor levels in the \code{levels} argument,
or supply the name of a time-ordered variable in the \code{order} argument.

\code{\link[=brm_data_chronologize]{brm_data_chronologize()}} can be called either before or just after
\code{\link[=brm_data]{brm_data()}}, but in the former case, the discrete time variable
needs to be specified explicitly in \code{time} argument. And in the latter,
\code{\link[=brm_data_chronologize]{brm_data_chronologize()}} must be called before any of the informative
prior archetype functions such as \code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}}.
}
\section{Contrasts}{

Ordinarily, ordered factors automatically use polynomial contrasts from
\code{\link[=contr.poly]{contr.poly()}}. This is undesirable for MMRMs, so if the time variable
is an ordered factor, then \code{\link[=brm_data]{brm_data()}}
manually sets \code{contrasts(data[[time]])} to a set of treatment contrasts
using \code{\link[=contr.treatment]{contr.treatment()}}. If you prefer different contrasts, please
manually set \code{contrasts(data[[time]])} to something else after
calling \code{\link[=brm_data]{brm_data()}}.
}

\examples{
data <- brm_simulate_outline(n_time = 12, n_patient = 4)
data$AVISIT <- gsub("_0", "_", data$time)
data$AVISITN <- as.integer(gsub("time_", "", data$time))
data[, c("AVISIT", "AVISITN")]
sort(unique(data$AVISIT)) # wrong order
data1 <- brm_data_chronologize(data, time = "AVISIT", order = "AVISITN")
sort(unique(data1$AVISIT)) # correct order
levels <- paste0("time_", seq_len(12))
data2 <- brm_data_chronologize(data, time = "AVISIT", levels = levels)
sort(unique(data2$AVISIT)) # correct order
}
\seealso{
Other data: 
\code{\link{brm_data}()},
\code{\link{brm_data_change}()}
}
\concept{data}
