% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_LoadModels.R
\name{BIOMOD_LoadModels}
\alias{BIOMOD_LoadModels}
\title{Load species distribution models built with \pkg{biomod2}}
\usage{
BIOMOD_LoadModels(bm.out, ...)
}
\arguments{
\item{bm.out}{a \code{\link{BIOMOD.models.out}} or \code{\link{BIOMOD.ensemble.models.out}} 
object that can be obtained with the \code{\link{BIOMOD_Modeling}} or 
\code{\link{BIOMOD_EnsembleModeling}} functions}

\item{\ldots}{(\emph{optional, see Details)})}
}
\value{
A \code{vector} containing the names of the loaded models.
}
\description{
This function loads individual models built with \code{\link{BIOMOD_Modeling}} 
or \code{\link{BIOMOD_EnsembleModeling}} functions.
}
\details{
This function might be of particular use to load models and make response plot analyses. \cr \cr

Running the function providing only \code{bm.out} argument will load all models built by the 
\code{\link{BIOMOD_Modeling}} or \code{\link{BIOMOD_EnsembleModeling}} function. But a 
subselection of models can be done using the following additional arguments :
\itemize{
  \item{\code{models} : }{a \code{vector} containing model names to be loaded, must be among 
  \code{GLM}, \code{GBM}, \code{GAM}, \code{CTA}, \code{ANN}, \code{SRE}, \code{FDA}, 
  \code{MARS}, \code{RF}, \code{MAXENT.Phillips}, \code{MAXENT.Phillips.2}}
  \item{\code{run.eval} : }{a \code{vector} containing repetition set to be loaded, must be 
  among \code{RUN1}, \code{RUN2}, \code{...}, \code{Full}}
  \item{\code{data.set} : }{a \code{vector} containing pseudo-absence set to be loaded, must 
  be among \code{PA1}, \code{PA2}, \code{...} \cr \cr}
  \item{\code{path} : }{a \code{character} corresponding to the location of the species folder 
  (if different from the current working directory) \cr \cr}
  \item{\code{full.name} : }{a \code{vector} containing model names to be kept, must be either 
  \code{all} or a sub-selection of model names \cr \cr}
  \item{\code{as} : }{a \code{character} to contain the loaded models}
}
}
\examples{

# Load species occurrences (6 species available)
myFile <- system.file('external/species/mammals_table.csv', package = 'biomod2')
DataSpecies <- read.csv(myFile, row.names = 1)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
myFiles <- paste0('external/bioclim/current/bio', c(3, 4, 7, 11, 12), '.grd')
myExpl <- raster::stack(system.file(myFiles, package = 'biomod2'))

\dontshow{
myExtent <- raster::extent(0,30,45,70)
myExpl <- raster::stack(raster::crop(myExpl, myExtent))
}

# ---------------------------------------------------------------
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                       expl.var = myExpl,
                                       resp.xy = myRespXY,
                                       resp.name = myRespName)

  # Create default modeling options
  myBiomodOptions <- BIOMOD_ModelingOptions()

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      bm.options = myBiomodOptions,
                                      nb.rep = 2,
                                      data.split.perc = 80,
                                      metric.eval = c('TSS','ROC'),
                                      var.import = 3,
                                      do.full.models = FALSE,
                                      seed.val = 42)
}


# ---------------------------------------------------------------
# Loading some models built
myLoadedModels <- BIOMOD_LoadModels(bm.out = myBiomodModelOut, models = 'RF')
myLoadedModels


}
\seealso{
\code{\link{BIOMOD_Modeling}}, \code{\link{BIOMOD_EnsembleModeling}}

Other Main functions: 
\code{\link{BIOMOD_CrossValidation}()},
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_ModelingOptions}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_PresenceOnly}()},
\code{\link{BIOMOD_Projection}()},
\code{\link{BIOMOD_RangeSize}()},
\code{\link{BIOMOD_Tuning}()}
}
\author{
Damien Georges
}
\concept{Main functions}
