% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/test_features.R
\name{test_features}
\alias{test_features}
\title{Permutation test for feature selection}
\usage{
test_features(
  target,
  features,
  criterion = "ig",
  adjust = "BH",
  threshold = 1,
  quick = TRUE,
  times = 1e+05
)
}
\arguments{
\item{target}{\code{integer} vector with target information (e.g. class labels).}

\item{features}{\code{integer} matrix of features with number of rows equal 
to the length of the target vector.}

\item{criterion}{criterion used in permutation test. See Details 
for the list of possible criterions.}

\item{adjust}{name of p-value adjustment method. See \code{\link[stats]{p.adjust}}
for the list of possible values. If \code{NULL}, p-values are not adjusted.}

\item{threshold}{\code{integer}. Features that occur less than \code{threshold}
and more often than \code{nrow(features)-threshold} are discarded from the permutation 
test.}

\item{quick}{\code{logical}, if \code{TRUE} Quick Permutation Test (QuiPT) is used. If 
\code{FALSE}, normal permutation test is performed.}

\item{times}{number of times procedure should be repeated. Ignored if \code{quick} is 
\code{TRUE}.}
}
\value{
an object of class \code{\link{feature_test}}.
}
\description{
Performs a feature selection on positioned n-gram data using a Fisher's 
permutation test.
}
\details{
Since the procedure involves multiple testing, it is advisable to use one
of the avaible p-value adjustment methods. Such methods can be used directly by 
specifying the \code{adjust} parameter.


Available criterions:
\describe{
  \item{ig}{Information Gain: \code{\link{calc_ig}}.}
  \item{kl}{Kullback-Leibler divergence: \code{\link{calc_kl}}.}
  \item{cs}{Chi-squared-based measure: \code{\link{calc_cs}}.}
}
}
\note{
Both \code{target} and \code{features} must be binary, i.e. contain only 0 
and 1 values.

Features occuring too often and too rarely are considered not informative and may be 
removed using the threshold parameter.
}
\examples{
# significant feature
tar_feat1 <- create_feature_target(10, 390, 0, 600) 
# significant feature
tar_feat2 <- create_feature_target(9, 391, 1, 599)
# insignificant feature
tar_feat3 <- create_feature_target(198, 202, 300, 300)
test_res <- test_features(tar_feat1[, 1], cbind(tar_feat1[, 2], tar_feat2[, 2], 
                          tar_feat3[, 2]))
summary(test_res)
cut(test_res)

# real data example
# we will analyze only a subsample of a dataset to make analysis quicker
ids <- c(1L:100, 701L:800)
deg_seqs <- degenerate(human_cleave[ids, 1L:9], 
                       list(`a` = c(1, 6, 8, 10, 11, 18), 
                            `b` = c(2, 5, 13, 14, 16, 17, 19, 20), 
                            `c` = c(3, 4, 7, 9, 12, 15)))

# positioned n-grams example
bigrams_pos <- count_ngrams(deg_seqs, 2, letters[1L:3], pos = TRUE)
test_features(human_cleave[ids, 10], bigrams_pos)

# unpositioned n-grams example, binarization required
bigrams_notpos <- count_ngrams(deg_seqs, 2, letters[1L:3], pos = TRUE)
test_features(human_cleave[ids, 10], binarize(bigrams_notpos))
}
\references{
Radivojac P, Obradovic Z, Dunker AK, Vucetic S, 
\emph{Feature selection filters based on the permutation test} in 
Machine Learning: ECML 2004, 15th European 
Conference on Machine Learning, Springer, 2004.
}
\seealso{
\code{\link{binarize}} - binarizes input data.

\code{\link{calc_criterion}} - computes selected criterion.

\code{\link{distr_crit}} - distribution of criterion used in QuiPT.

\code{\link{summary.feature_test}} - summary of results.

\code{\link{cut.feature_test}} - aggregates test results in groups based on feature's
p-value.
}
\keyword{nonparametric}
