#' Inspect an integrated profile (`vpi`)
#'
#' R base functions for inspecting an integrated profile of biological targets
#' (`vpi`) object.
#'
#' @param object A `vpi` object.
#' @param ... Additional arguments affecting the summary produced.
#'
#' @method summary vpi
#'
#' @export
#'
#' @details
#' A integrated profile of biological targets is a specially classed
#' `data.frame` generated by the function [integrate_profile()] with the
#' following quantities:
#' * `radar`: Radar identifier.
#' * `datetime`: POSIXct date of each profile in UTC.
#' * `vid`: Vertically Integrated Density in individuals/km^2. `vid` is a
#' surface density, whereas `dens` in `vp` objects is a volume density.
#' * `vir`: Vertically Integrated Reflectivity in cm^2/km^2.
#' * `mtr`: Migration Traffic Rate in individuals/km/h.
#' * `rtr`: Reflectivity Traffic Rate in cm^2/km/h.
#' * `mt`: Migration Traffic in individuals/km, cumulated from the start of the
#' time series up to `datetime`.
#' * `rt`: Reflectivity Traffic in cm^2/km, cumulated from the start of the
#' time series up to `datetime`.
#' * `ff`: Horizontal ground speed in m/s.
#' * `dd`: Horizontal ground speed direction in degrees.
#' * `u`: Ground speed component west to east in m/s.
#' * `v`: Ground speed component south to north in m/s.
#' * `height`: Mean flight height (height weighted by eta) in m above sea level.
#'
#' @seealso
#' * [integrate_profile()]
#' * [plot.vpi()]
#'
#' @examples
#' # Load the example vertical profile time series and integrate to a vpi
#' vpi <- integrate_profile(example_vpts)
#'
#' # Check if it is an object of class vpi
#' is.vpi(vpi)
#'
#' # Get summary info
#' summary(vpi)
summary.vpi <- function(object, ...) {
  stopifnot(inherits(object, "vpi"))
  cat("               Vertically integrated profile(s) (class vpi)\n\n")
  cat("           radar: ", attributes(object)$radar, "\n")
  cat("      # profiles: ", length(object$datetime), "\n")
  cat("time range (UTC): ", format(min(object$datetime), "%Y-%m-%d %H:%M:%S"),
    "-", format(max(object$datetime), "%Y-%m-%d %H:%M:%S"), "\n"
  )
}

#' Check if an object is of class `vpi`
#'
#' @param x A `vpi` object.
#'
#' @return For [is.vpi()]: `TRUE` for an object of class `vpi`, otherwise
#'   `FALSE`.
#'
#' @rdname summary.vpi
#'
#' @export
is.vpi <- function(x) {
  inherits(x, "vpi")
}
