#' recovery_d
#'
#' @description
#' This function applies `optimize_para` to each fake data in the list generated 
#'  by `simulate_list`. The results can be used for parameter recovery and 
#'  model recovery, helping evaluate the consistency and validity of the 
#'  reinforcement learning model.
#' 
#'  For more information, please refer to the GitHub repository:
#'  https://github.com/yuki-961004/binaryRL
#'
#' @param list [list] a list generated by function `simulate_list`
#' 
#' @param id [integer] default = 1
#' 
#' @param fit_model [function] fit model
#' 
#' @param funcs [vector] A character vector containing the names of all 
#'  user-defined functions required for the computation.
#' 
#' @param model_name [character] the name of your modal
#' 
#' @param n_params [integer] The number of free parameters in your model. 
#' 
#' @param n_trials [integer] The total number of trials in your experiment.
#'
#' @param initial_params [vector] Initial values for the free parameters. 
#'  These need to be set only when using L-BFGS-B. Other algorithms 
#'  automatically generate initial values.
#'  for `L-BFGS-B`, `GenSA`, set `initial = c(0, 0, ...)`
#'  
#' @param initial_size [integer] Initial values for the free parameters. 
#'  These need to be set only when using L-BFGS-B. Other algorithms 
#'  automatically generate initial values.
#'  for `Bayesian`, `GA`, set `initial = 50`
#' 
#' @param lower [vector] lower bounds of free parameters
#' 
#' @param upper [vector] upper bounds of free parameters
#' 
#' @param seed [integer] random seed. This ensures that the results are 
#'  reproducible and remain the same each time the function is run. 
#'  default: `seed = 123` 
#'  
#' @param iteration [integer] the number of iteration
#'  
#' @param nc [integer] Number of CPU cores to use for parallel computation.
#' 
#' @param algorithm [character] Choose a algorithm package from 
#'  `L-BFGS-B`, `GA`, `GenSA`, `DEoptim`
#'  
#' @return a data frame for parameter recovery and model recovery
#' @export
#'
recovery_data <- function(
    list,
    id = 1,
    fit_model,
    funcs = NULL,
    model_name,
    n_params, 
    n_trials,
    lower,
    upper,
    initial_params = NA,
    initial_size = 50,
    iteration = 10,
    seed = 123,
    nc = 1,
    algorithm
){
  # 创建一个空数据集, 用于存放结果
  recovery <- data.frame(
    fit_model = rep(model_name, length(list)),
    ACC = NA,
    LL = NA,
    AIC = NA,
    BIC = NA
  )
  # 检测是都用同一个被试的题目, 还是每次都更换题目
  if (length(id) == 1) {
    id <- rep(id, length(list))
  }
  
  # 增加放置输入参数的列
  n_input_params <- length(list[[1]]$input)
  
  for (i in 1:n_input_params) {
    recovery[, i + 5] <- NA
    names(recovery)[i + 5] <- paste0("input_param_", i)
  }
  # 增加放置输出参数的列
  n_output_params <- length(lower)
  
  for (i in 1:n_output_params) {
    recovery[, i + 5 + n_input_params] <- NA
    names(recovery)[i + 5 + n_input_params] <- paste0("output_param_", i)
  }
  

  # Check for internally parallel algorithms
  if (nc == 1) {
    
    progressr::handlers(progressr::handler_txtprogressbar)
    
    progressr::with_progress({
      
      p <- progressr::progressor(steps = length(list))
      
      for (i in 1:length(list)) {
        
        p()
        
        data <- list[[i]][[1]]
        
        binaryRL.res <- binaryRL::optimize_para(
          data = data,
          id = id[i],
          obj_func = fit_model,
          n_params = n_params,
          n_trials = n_trials,
          lower = lower,
          upper = upper,
          initial_params = initial_params,
          initial_size = initial_size,
          iteration = iteration,
          seed = seed,
          algorithm = algorithm
        )
        
        recovery[i, 2] <- binaryRL.res$acc
        recovery[i, 3] <- binaryRL.res$ll
        recovery[i, 4] <- binaryRL.res$aic
        recovery[i, 5] <- binaryRL.res$bic
        
        for (j in 1:n_input_params) {
          recovery[i, j + 5] <- list[[i]]$input[j]
        }
        
        for (j in 1:n_output_params) {
          recovery[i, j + 5 + n_input_params] <- binaryRL.res$output[j]
        }
      }
    })
  }
  else {
    
    future::plan(future::multisession, workers = nc)
    doFuture::registerDoFuture()

    # 以迭代次数作为进度条
    n_iterations <- length(list) 
    
    progressr::handlers(progressr::handler_txtprogressbar)
    
    progressr::with_progress({
      
      p <- progressr::progressor(steps = n_iterations)
      
      doRNG::registerDoRNG(seed = seed)
      
      temp_recovery <- foreach::foreach(
        i = 1:n_iterations, .combine = rbind,
        .packages = "binaryRL", 
        .export = funcs
      ) %dorng% {
        
        data_i <- list[[i]][[1]]
        
        binaryRL.res <- binaryRL::optimize_para(
          data = data_i,
          id = id[i],
          obj_func = fit_model,
          n_params = n_params,
          n_trials = n_trials,
          lower = lower,
          upper = upper,
          initial_params = initial_params,
          initial_size = initial_size,
          iteration = iteration,
          seed = seed,
          algorithm = algorithm
        )
        
        row_i <- data.frame(matrix(NA, nrow = 1, ncol = 5 + n_input_params + n_output_params))
        row_i[1, 1] <- model_name
        row_i[1, 2] <- binaryRL.res$acc
        row_i[1, 3] <- binaryRL.res$ll
        row_i[1, 4] <- binaryRL.res$aic
        row_i[1, 5] <- binaryRL.res$bic
        
        for (j in 1:n_input_params) {
          row_i[1, 5 + j] <- list[[i]]$input[j]
        }
        for (j in 1:n_output_params) {
          row_i[1, 5 + n_input_params + j] <- binaryRL.res$output[j]
        }
        
        # 更新進度條
        p() 
        
        return(row_i)
      }
    })
    
    # 继承recovery的列名
    colnames(temp_recovery) <- colnames(recovery)
    recovery <- temp_recovery
  }
  
  return(recovery)
}
