% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apeff_bife.R
\name{apeff_bife}
\alias{apeff_bife}
\title{Average Partial Effects for Binary Choice Models with Fixed Effects}
\usage{
apeff_bife(mod, discrete = NULL, bias_corr = "ana", iter_demeaning = 100,
  tol_demeaning = 1e-05, iter_offset = 1000, tol_offset = 1e-05)
}
\arguments{
\item{mod}{an object of class \code{bife}.}

\item{discrete}{a description of the variables that are discrete regressors.
For \code{apeff_bife} this has to be a character string naming the discrete regressors.
Default is \code{NULL} (no discrete regressor(s)).}

\item{bias_corr}{an optional string that specifies the type of the bias correction:
semi or analytical. The value should be any of the values \code{"semi"} or 
\code{"ana"}. Default is \code{"ana"} (analytical bias-correction). Details
are given under \code{Details}.}

\item{iter_demeaning}{an optional integer value that specifies the maximum number
of iterations of the demeaning algorithm. Default is \code{100}. Details
are given under \code{Details}.}

\item{tol_demeaning}{an optional number that specifies the tolerance level of the
demeaning algorithm. Default is \code{1e-5}. Details are given under \code{Details}.}

\item{iter_offset}{an optional integer value that specifies the maximum number of
iterations of the offset algorithm for the computation of bias-adjusted fixed
effects. Default is \code{1000}. Details are given under \code{Details}.}

\item{tol_offset}{an optional number that specifies the tolerance level of the 
offset algorithm for the computation of bias-adjusted fixed effects. Default
is \code{1e-5}. Details are given under \code{Details}.}
}
\value{
An object of \code{apeff_bife} returns a named matrix with at least a first
column "apeff" containing the uncorrected average partial effects of the
structural variables. An optional second column "apeff_corrected" is returned
containing the corrected average partial effects of the structural variables.
}
\description{
\code{apeff_bife} is a function used to compute average partial effects
for fixed effects binary choice models. It is able to compute
bias-corrected average partial effects derived by Newey and Hahn (2004) to
account for the incidental parameters bias.
}
\details{
The semi bias-corrected average partial effects are computed as usual partial
effects with the bias-adjusted fixed effects and the bias-corrected structural
parameters.

The analytical bias-corrected average partial effects follow
Newey and Hahn (2004). For further details consult the description of \code{bife}.

\strong{Note:} Bias-corrected partial effects can be only returned if the
object \code{mod} returns bias-corrected coefficients, i.e. if a bias-correction 
has been used in the previous \code{bife} command.
}
\examples{
library("bife")

# Load 'psid' dataset
dataset <- psid
head(dataset)

# Fixed effects logit model w/o bias-correction
mod_no <- bife(LFP ~ AGE + I(INCH / 1000) + KID1 + KID2 + KID3 | ID,
 data = dataset, bias_corr = "no")
               
# Compute uncorrected average partial effects for mod_no
# Note: bias_corr does not affect the result
apeff_bife(mod_no, discrete = c("KID1", "KID2", "KID3")) 

# Fixed effects logit model with analytical bias-correction
mod_ana <- bife(LFP ~ AGE + I(INCH / 1000) + KID1 + KID2 + KID3 | ID,
 data = dataset)
               
# Compute semi-corrected average partial effects for mod_ana
apeff_bife(mod_ana, discrete = c("KID1", "KID2", "KID3"),
 bias_corr = "semi")

# Compute analytical bias-corrected average partial effects
# for mod_ana
apeff_bife(mod_ana, discrete = c("KID1", "KID2", "KID3"))
 
}
\references{
Hahn, J., and W. Newey (2004). "Jackknife and analytical bias reduction 
for nonlinear panel models." Econometrica 72(4), 1295-1319.

Stammann, A., F. Heiss, and D. McFadden (2016). "Estimating Fixed Effects 
Logit Models with Large Panel Data". Working paper.
}
\seealso{
\code{\link{bife}}
}
\author{
Amrei Stammann, Daniel Czarnowske, Florian Heiss, Daniel McFadden
}
