% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/osdsm.R
\name{osdsm}
\alias{osdsm}
\title{Extract backbone using the Ordinal Stochastic Degree Sequence Model}
\usage{
osdsm(
  B,
  alpha = 0.05,
  trials = NULL,
  missing.as.zero = FALSE,
  signed = FALSE,
  mtc = "none",
  class = "original",
  narrative = FALSE,
  progress = TRUE,
  ...
)
}
\arguments{
\item{B}{An ordinally weighted bipartite graph, as: (1) an incidence matrix in the form of a matrix or sparse \code{\link{Matrix}}; (2) an edgelist in the form of a three-column dataframe; (3) an \code{\link{igraph}} object.}

\item{alpha}{real: significance level of hypothesis test(s)}

\item{trials}{integer: the number of bipartite graphs generated to approximate the edge weight distribution. If NULL, the number of trials is selected based on \code{alpha} (see details)}

\item{missing.as.zero}{boolean: should missing edges be treated as edges with zero weight and tested for significance}

\item{signed}{boolean: TRUE for a signed backbone, FALSE for a binary backbone (see details)}

\item{mtc}{string: type of Multiple Test Correction to be applied; can be any method allowed by \code{\link{p.adjust}}.}

\item{class}{string: the class of the returned backbone graph, one of c("original", "matrix", "Matrix", "igraph", "edgelist").
If "original", the backbone graph returned is of the same class as \code{B}.}

\item{narrative}{boolean: TRUE if suggested text & citations should be displayed.}

\item{progress}{boolean: TRUE if the progress of Monte Carlo trials should be displayed.}

\item{...}{optional arguments}
}
\value{
If \code{alpha} != NULL: Binary or signed backbone graph of class \code{class}.

If \code{alpha} == NULL: An S3 backbone object containing (1) the weighted graph as a matrix, (2) upper-tail p-values as a
matrix, (3, if \code{signed = TRUE}) lower-tail p-values as a matrix, (4, if present) node attributes as a dataframe, and
(5) several properties of the original graph and backbone model, from which a backbone can subsequently be extracted
using \code{\link[=backbone.extract]{backbone.extract()}}.
}
\description{
\code{osdsm} extracts the backbone of a bipartite projection using the Ordinal Stochastic Degree Sequence Model.
}
\details{
The \code{osdsm} function compares an edge's observed weight in the projection \code{B*t(B)} to the distribution of weights
expected in a projection obtained from a random bipartite network where both the rows and the columns contain
approximately the same number of each value. The edges in \code{B} must be integers, and are assumed to represent an
ordinal-level measure such as a Likert scale that starts at 0.

When \code{signed = FALSE}, a one-tailed test (is the weight stronger?) is performed for each edge. The resulting backbone
contains edges whose weights are significantly \emph{stronger} than expected in the null model. When \code{signed = TRUE}, a
two-tailed test (is the weight stronger or weaker?) is performed for each edge. The resulting backbone contains
positive edges for those whose weights are significantly \emph{stronger}, and negative edges for those whose weights are
significantly \emph{weaker}, than expected in the null model.

The p-values used to evaluate the statistical significance of each edge are computed using Monte Carlo methods. The number of
\code{trials} performed affects the precision of these p-values. This precision impacts the confidence that a given edge's p-value
is less than the desired \code{alpha} level, and therefore represents a statistically significant edge that should be retained in
the backbone. When \code{trials = NULL}, \code{\link[=trials.needed]{trials.needed()}} is used to estimate the required number of trials to evaluate the
statistical significance of an edges' p-values.
}
\examples{
#A weighted binary bipartite network of 20 agents & 50 artifacts; agents form two communities
B <- rbind(cbind(matrix(sample(0:3, 250, replace = TRUE, prob = ((1:4)^2)),10),
                 matrix(sample(0:3, 250, replace = TRUE, prob = ((4:1)^2)),10)),
           cbind(matrix(sample(0:3, 250, replace = TRUE, prob = ((4:1)^2)),10),
                 matrix(sample(0:3, 250, replace = TRUE, prob = ((1:4)^2)),10)))

P <- B\%*\%t(B) #An ordinary weighted projection...
plot(igraph::graph_from_adjacency_matrix(P, mode = "undirected",
                                         weighted = TRUE, diag = FALSE)) #...is a dense hairball

bb <- osdsm(B, alpha = 0.05, narrative = TRUE,  #An oSDSM backbone...
            class = "igraph", trials = 100)
plot(bb) #...is sparse with clear communities
}
\references{
package: {Neal, Z. P. (2022). backbone: An R Package to Extract Network Backbones. \emph{PLOS ONE, 17}, e0269137. \doi{10.1371/journal.pone.0269137}}

osdsm: {Neal, Z. P. (2017). Well connected compared to what? Rethinking frames of reference in world city network research. \emph{Environment and Planning A, 49}, 2859-2877. \doi{10.1177/0308518X16631339}}
}
