% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/osdsm.R
\name{osdsm}
\alias{osdsm}
\title{Extract backbone using the Ordinal Stochastic Degree Sequence Model}
\usage{
osdsm(
  B,
  trials = 1000,
  alpha = NULL,
  signed = FALSE,
  fwer = "none",
  class = "original",
  narrative = FALSE
)
}
\arguments{
\item{B}{An ordinally weighted bipartite graph, as: (1) an incidence matrix in the form of a matrix, sparse \code{\link{Matrix}}, or dataframe; (2) an edgelist in the form of a three-column matrix, sparse \code{\link{Matrix}}, or dataframe; (3) an \code{\link{igraph}} object; (4) a \code{\link{network}} object.
Any rows and columns of the associated bipartite matrix that contain only zeros or only ones are automatically removed before computations.}

\item{trials}{integer: Number of random bipartite graphs generated}

\item{alpha}{real: significance level of hypothesis test(s)}

\item{signed}{boolean: TRUE for a signed backbone, FALSE for a binary backbone (see details)}

\item{fwer}{string: type of familywise error rate correction to be applied; can be any method allowed by \code{\link{p.adjust}}.}

\item{class}{string: the class of the returned backbone graph, one of c("original", "matrix", "sparseMatrix", "igraph", "network", "edgelist").
If "original", the backbone graph returned is of the same class as \code{B}.}

\item{narrative}{boolean: TRUE if suggested text & citations should be displayed.}
}
\value{
If \code{alpha} != NULL: Binary or signed backbone graph of class \code{class}.

If \code{alpha} == NULL: An S3 backbone object containing three matrices (the weighted graph, edges' upper-tail p-values,
edges' lower-tail p-values), and a string indicating the null model used to compute p-values, from which a backbone
can subsequently be extracted using \code{\link[=backbone.extract]{backbone.extract()}}. The \code{signed}, \code{fwer}, \code{class}, and \code{narrative} parameters
are ignored.
}
\description{
\code{osdsm} extracts the backbone of a bipartite projection using the Ordinal Stochastic Degree Sequence Model.
}
\details{
The \code{osdsm} function compares an edge's observed weight in the projection \code{B*t(B)} to the distribution of weights
expected in a projection obtained from a random bipartite network where both the rows and the columns contain
approximately the same number of each value. The edges in \code{B} must be integers, and are assumed to represent an
ordinal-level measure such as a Likert scale.

When \code{signed = FALSE}, a one-tailed test (is the weight stronger) is performed for each edge with a non-zero weight. It
yields a backbone that perserves edges whose weights are significantly \emph{stronger} than expected in the chosen null
model. When \code{signed = TRUE}, a two-tailed test (is the weight stronger or weaker) is performed for each every pair of nodes.
It yields a backbone that contains positive edges for edges whose weights are significantly \emph{stronger}, and
negative edges for edges whose weights are significantly \emph{weaker}, than expected in the chosen null model.
\emph{NOTE: Before v2.0.0, all significance tests were two-tailed and zero-weight edges were evaluated.}
}
\examples{
#A weighted binary bipartite network of 20 agents & 50 artifacts; agents form two communities
B <- rbind(cbind(matrix(sample(0:3, 250, replace = TRUE, prob = ((1:4)^2)),10),
                 matrix(sample(0:3, 250, replace = TRUE, prob = ((4:1)^2)),10)),
           cbind(matrix(sample(0:3, 250, replace = TRUE, prob = ((4:1)^2)),10),
                 matrix(sample(0:3, 250, replace = TRUE, prob = ((1:4)^2)),10)))

P <- B\%*\%t(B) #An ordinary weighted projection...
plot(igraph::graph_from_adjacency_matrix(P, mode = "undirected",
                                         weighted = TRUE, diag = FALSE)) #...is a dense hairball

bb <- osdsm(B, alpha = 0.05, narrative = TRUE, class = "igraph") #An oSDSM backbone...
plot(bb) #...is sparse with clear communities
}
\references{
{Neal, Z. P. (2017). Well connected compared to what? Rethinking frames of reference in world city network research. \emph{Environment and Planning A, 49}, 2859-2877. \doi{10.1177/0308518X16631339}}
}
