% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gen.R
\name{gen}
\alias{gen}
\alias{yield}
\alias{yieldFrom}
\title{Create an iterator using sequential code.}
\usage{
gen(
  expr,
  ...,
  split_pipes = FALSE,
  trace = trace_,
  compileLevel = getOption("async.compileLevel")
)

yield(expr)

yieldFrom(it, err)
}
\arguments{
\item{expr}{An expression, to be turned into an iterator.}

\item{...}{Undocumented.}

\item{split_pipes}{Silently rewrite expressions where "yield"
appears in chained calls. See \link{async}.}

\item{trace}{Optional tracing function for debugging. See \link{async}.}

\item{compileLevel}{Current levels are 0 (no compilation) or
-1 (name munging only).}

\item{it}{A list, \link{iteror} or compatible object.}

\item{err}{An error handler}
}
\value{
`gen({...}) returns an \link{iteror}.

\code{yield(x)} returns the same value x.

yieldFrom returns NULL, invisibly.
}
\description{
\code{gen({...})} with an expression written in its argument, creates a
generator, an object which computes an indefinite sequence.

When written inside a generator expression, \code{yield(expr)} causes the
generator to return the given value, then pause until the next value is
requested.

When running in a generator expression, \verb{yieldFrom(it))}, given
a list or \link{iteror} in its argument, will yield successive values from that
iteror until it is exhausted, then continue.
}
\details{
On the "inside", that is the point of view of code you write in
\code{{...}}, is ordinary sequential code using conditionals, branches,
loops and such, outputting one value after another with \code{yield()}.
For example, this code creates a generator that computes a random
walk:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rwalk <- gen(\{
  x <- 0;
  repeat \{
    x <- x + rnorm(1)
    yield(x)
  \}
\})
}\if{html}{\out{</div>}}

On the "outside," that is, the object returned by \code{gen()}, a
generator behaves like an iterator over an indefinite
collection. So we can collect the first 100 values from the above
generator and compute their mean:

\if{html}{\out{<div class="sourceCode">}}\preformatted{rwalk |> itertools2::take(100) |> as.numeric() |> mean()
}\if{html}{\out{</div>}}

When \code{nextOr(rwalk, ...)} is called, the generator executes its
"inside" expression, in a local environment, until it reaches a
call to \verb{yield().} THe generator 'pauses', preserving its execution
state, and \code{nextElem} then returns what was passed to \code{yield}. The
next time \code{nextElem(rwalk)} is called, the generator resumes
executing its inside expression starting after the \code{yield()}.

If you call \code{gen} with a function expression, as in:

\if{html}{\out{<div class="sourceCode">}}\preformatted{gseq <- gen(function(x) for (i in 1:x) yield(i))
}\if{html}{\out{</div>}}

then instead of returning a single generator it will return a
\emph{generator function} (i.e. a function that constructs and returns a
generator.) The above is morally equivalent to:

\if{html}{\out{<div class="sourceCode">}}\preformatted{gseq <- function(x) \{force(x); gen(for (i in 1:x) yield(i))\}
}\if{html}{\out{</div>}}

so the generator function syntax just saves you writing the \link{force}
call.

A generator expression can use any R functions, but a call to
\code{yield} may only appear in the arguments of a "pausable" function.
The \code{async} package has several built-in pausable functions corresponding
to base R's control flow functions, such as \code{if}, \code{while}, \code{tryCatch},
\verb{<-}, \code{{}}, \code{||} and so on (see \link{pausables} for more details.)  A call
to \code{yield} may only appear in an argument of one of these pausable
functions. So this random walk generator:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rwalk <- gen(\{x <- 0; repeat \{x <- yield(x + rnorm(1))\}\})
}\if{html}{\out{</div>}}

is legal, because \code{yield} appears within arguments to \code{{}},
\code{repeat}, and \verb{<-}, for which this package has pausable
definitions. However, this:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rwalk <- gen(\{x <- rnorm(1); repeat \{x <- rnorm(1) + yield(x)\}\})
}\if{html}{\out{</div>}}

is not legal, because \code{yield} appears in an argument to \code{+}, which
does not have a pausable definition.
}
\examples{
chain <- function(...) {
  iterators <- list(...)
  gen(for (it in iterators) yieldFrom(it))
}
}
